/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!              Markus Gaug  3/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MBadPixelsCam                                                                                                                              //
//
// Storage container to store bad pixel of the camera...
//
/////////////////////////////////////////////////////////////////////////////
#include "MBadPixelsCam.h"

#include <fstream>

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MBadPixelsPix.h"

ClassImp(MBadPixelsCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor.
//
MBadPixelsCam::MBadPixelsCam(const char *name, const char *title)
{
    fName  = name  ? name  : "MBadPixelsCam";
    fTitle = title ? title : "Storage container to store bad pixel information";

    fArray = new TClonesArray("MBadPixelsPix", 1);
}

MBadPixelsCam::MBadPixelsCam(const MBadPixelsCam &cam)
{
    fName  = "MBadPixelsCam";
    fTitle = "Storage container to store bad pixel information";

    fArray = new TClonesArray("MBadPixelsPix", 1);
    cam.Copy(*this);
}

// --------------------------------------------------------------------------
//
// Delete the array conatining the bad pixel information
//
MBadPixelsCam::~MBadPixelsCam()
{
    delete fArray;
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MBadPixelsCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// --------------------------------------------------------------------------
//
// Get the size of the MBadPixelsCam
//
Int_t MBadPixelsCam::GetSize() const
{
    return fArray->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Copy 'constructor'
//
void MBadPixelsCam::Copy(TObject &object) const
{
    MBadPixelsCam &cam = (MBadPixelsCam&)object;

    const Int_t n = GetSize();

    if (n==0)
        return;

    cam.InitSize(n);
    for (int i=0; i<n; i++)
        (*this)[i].Copy(cam[i]);
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MBadPixelsPix &MBadPixelsCam::operator[](Int_t i)
{
    return *static_cast<MBadPixelsPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MBadPixelsPix &MBadPixelsCam::operator[](Int_t i) const
{
    return *static_cast<MBadPixelsPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Merge two MBadPixelsCam together, see also MBadPixelsPix::Merge
//
void MBadPixelsCam::Merge(const MBadPixelsCam &cam)
{
    const Int_t n = cam.GetSize();
    if (n==0)
    {
        *fLog << warn << "MBadPixelsCam::Merge: Container empty." << endl;
        return;
    }

    if (GetSize()==0)
        InitSize(n);

    if (n!=GetSize())
    {
        *fLog << warn << "MBadPixelsCam::Merge: Size mismatch... ignored." << endl;
        return;
    }

    for (int i=0; i<n; i++)
        (*this)[i].Merge(cam[i]);
}

// --------------------------------------------------------------------------
//
// Clear the contents of all bad pixels (set=0 means Ok)
//
void MBadPixelsCam::Clear(Option_t *o)
{
    fArray->ForEach(TObject, Clear)();
}

// --------------------------------------------------------------------------
//
// Print the contents of all bad pixels
//
void MBadPixelsCam::Print(Option_t *o) const
{
  *fLog << all << GetDescriptor() << ":" << endl;
  
  *fLog << "Pixels without problems:" << endl;
  *fLog << endl;
    
  Int_t count = 0;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if (!(*this)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  *fLog << endl;
  *fLog << count << " normal pixels :-))" << endl;
  *fLog << endl;
  count = 0;
  
  
  *fLog << "Pixels unsuited for the whole run:" << endl;
  *fLog << endl;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if ((*this)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  *fLog << endl;
  *fLog << count << " unsuited pixels :-(" << endl;
  *fLog << endl;
  
  count = 0;
  
  *fLog << all << "Pixels unreliable for the whole run:" << endl;
  *fLog << all << endl;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if ((*this)[i].IsUnsuitable(MBadPixelsPix::kUnreliableRun))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  
  *fLog << endl;
  *fLog << count << " unreliable pixels :-(" << endl;
  *fLog << endl;

  count = 0;
  
  *fLog << all << "Charge is Pedestal:" << endl;
  *fLog << all << endl;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if ((*this)[i].IsUncalibrated(MBadPixelsPix::kChargeIsPedestal))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  
  *fLog << endl;
  *fLog << count << " ChargeIsPedestal :-(" << endl;
  *fLog << endl;

  count = 0;
  
  *fLog << all << "Charge Sigma not valid:" << endl;
  *fLog << all << endl;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if ((*this)[i].IsUncalibrated(MBadPixelsPix::kChargeSigmaNotValid))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  
  *fLog << endl;
  *fLog << count << " ChargeSigmaNotValid :-(" << endl;
  *fLog << endl;

  count = 0;
  
  *fLog << all << "Rel. Error Charge not valid:" << endl;
  *fLog << all << endl;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if ((*this)[i].IsUncalibrated(MBadPixelsPix::kChargeRelErrNotValid))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  
  *fLog << endl;
  *fLog << count << " ChargeRelErrNotValid :-(" << endl;
  *fLog << endl;


  count = 0;
  
  *fLog << all << " Deviating number photo-electrons:" << endl;
  *fLog << all << endl;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if ((*this)[i].IsUncalibrated(MBadPixelsPix::kDeviatingNumPhes))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  
  *fLog << endl;
  *fLog << count << " DeviatingNumPhes :-(" << endl;
  *fLog << endl;

  count = 0;
  
  *fLog << all << " Deviating F-Factor:" << endl;
  *fLog << all << endl;
  
  for (Int_t i=0; i<GetSize(); i++)
    {
      if ((*this)[i].IsUncalibrated(MBadPixelsPix::kDeviatingFFactor))
        {
          *fLog << i << " ";
          count ++;
        }
      
      if (count == 0)
        continue;
      
      if (!(count % 25))
        *fLog << endl;
    }
  
  *fLog << endl;
  *fLog << count << " DeviatingFFactor :-(" << endl;
  *fLog << endl;

}

// --------------------------------------------------------------------------
//
// Read from an ascii file of the format:
//    pixel1 pixel2 pixel3 pixel4
// while pixel1,2,3,4 are the pixel indices used in the software.
//
// To read the pixels corresponding to a given run you can give run!=0
// and a file of the format:
//   1234: 17 193 292
//   1235: 17 193 292 293
//
void MBadPixelsCam::AsciiRead(ifstream &fin, UInt_t run=0)
{
    Int_t len;
    TString str;

    while (1)
    {
        str.ReadLine(fin);
        if (!fin)
            return;

        Int_t r;

        const Int_t n = sscanf(str.Data(), " %d : %n", &r, &len);
        if (n!=1)
            return;

        if (run==0 || run && (UInt_t)r==run)
            break;
    }

    str.Remove(0, len);

    while (1)
    {
        Int_t idx;
        const Int_t n = sscanf(str.Data(), " %d %n", &idx, &len);

        if (n!=1)
            break;

        str.Remove(0, len);

        if (idx>=GetSize())
            InitSize(idx+1);

        (*this)[idx].SetUnsuitable(MBadPixelsPix::kUnsuitableRun);
    }
}

// --------------------------------------------------------------------------
//
// Write the information into an ascii file. If a run-number is given the
// run-number will lead the line.
//
Bool_t MBadPixelsCam::AsciiWrite(ostream &fout, UInt_t run=0) const
{
    if (run)
        fout << run << ":";

    for (int i=0; i<GetSize(); i++)
        if ((*this)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
            fout << " " << i;

    if (run && GetSize())
        fout << endl;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// The types are the following:
// 0: MBadPixelsPix::GetInfo()[0]
// 1: MBadPixelsPix::IsUnsuitable(MBadPixelsPix::kUnsuitableRun)
// 2: MBadPixelsPix::IsUnsuitable(MBadPixelsPix::kUnsuitableEvt)
// 3: MBadPixelsPix::IsUnsuitable(MBadPixelsPix::kUnreliableRun)
// 4: MBadPixelsPix::IsHiGainBad()
// 5: MBadPixelsPix::IsLoGainBad()
// 6: MBadPixelsPix::GetUnsuitableCalibration()
// 7: MBadPixelsPix::GetUnreliableCalibration()
// 8: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kHiGainNotFitted)
// 9: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kLoGainNotFitted)
// 10: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kHiGainOscillating)
// 11: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kLoGainOscillating)
// 12: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kLoGainSaturation )
// 13: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kChargeIsPedestal )
// 14: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kChargeErrNotValid)
// 15: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kChargeRelErrNotValid)
// 16: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kChargeSigmaNotValid )
// 17: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kMeanTimeInFirstBin  )
// 18: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kMeanTimeInLast2Bins )
// 19: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kDeviatingNumPhes    )
// 20: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kRelTimeNotFitted )
// 21: MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kRelTimeOscillating  )
//
Bool_t MBadPixelsCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (idx >= GetSize())
    return kFALSE;

  switch (type)
    {
    case 0:
      return (*this)[idx].GetInfo()[0];
    case 1:
      if  ((*this)[idx].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        val = 1;
      else
        return kFALSE;
      break;
    case 2:
      if  ((*this)[idx].IsUnsuitable(MBadPixelsPix::kUnsuitableEvt))
        val = 1;
      else
        return kFALSE;
      break;
    case 3:
      if  ((*this)[idx].IsUnsuitable(MBadPixelsPix::kUnreliableRun))
        val = 1;
      else
        return kFALSE;
      break;
    case 4:
      if  ((*this)[idx].IsHiGainBad())
        val = 1;
      else
        return kFALSE;
      break;
    case 5:
      if  ((*this)[idx].IsLoGainBad())
        val = 1;
      else
        return kFALSE;
      break;
    case 6:
      if  ((*this)[idx].GetUnsuitableCalibration())
        val = (*this)[idx].GetUnsuitableCalibration();
      else
        return kFALSE;
      break;
    case 7:
      if  ((*this)[idx].GetUnreliableCalibration())
        val = (*this)[idx].GetUnreliableCalibration();
      else
        return kFALSE;
      break;
    case 8:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kHiGainNotFitted))
        val = 1;
      else
        return kFALSE;
      break;
    case 9:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kLoGainNotFitted))
        val = 1;
      else
        return kFALSE;
      break;
    case 10:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kHiGainOscillating))
        val = 1;
      else
        return kFALSE;
      break;
    case 11:
      if ((*this)[idx].IsUncalibrated(MBadPixelsPix::kLoGainOscillating))
        val = 1;
      else
        return kFALSE;
      break;
    case 12:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kLoGainSaturation ))
        val = 1;
      else
        return kFALSE;
      break;
    case 13:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kChargeIsPedestal ))
        val = 1;
      else
        return kFALSE;
      break;
    case 14:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kChargeErrNotValid))
        val = 1;
      else
        return kFALSE;
      break;
    case 15:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kChargeRelErrNotValid))
        val = 1;
      else
        return kFALSE;
      break;
    case 16:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kChargeSigmaNotValid ))
        val = 1;
      else
        return kFALSE;
      break;
    case 17:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kMeanTimeInFirstBin  ))
        val = 1;
      else
        return kFALSE;
      break;
    case 18:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kMeanTimeInLast2Bins ))
        val = 1;
      else
        return kFALSE;
      break;
    case 19:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kDeviatingNumPhes ))
        val = 1;
      else
        return kFALSE;
      break;
    case 20:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kRelTimeNotFitted ))
        val = 1;
      else
        return kFALSE;
      break;
    case 21:
      if  ((*this)[idx].IsUncalibrated(MBadPixelsPix::kRelTimeOscillating))
        val = 1;
      else
        return kFALSE;
      break;
    default:
      return kFALSE;
    }
  
  return kTRUE;
}

void MBadPixelsCam::DrawPixelContent(Int_t num) const
{
    *fLog << warn << "MBadPixelsCam::DrawPixelContent - not available." << endl;
}
