/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   07/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                               
// MCalibrationChargeBlindCam                                               
//                                                               
// Base class for Blind Pixels Calibration results. Derived classes intialize
// the actual values of the MCalibrationBlindPix's. 
//
// Contains TClonesArrays for the following objects:
// - fBlindPixels:    Array of classes derived from MCalibrationChargeBlindPix, one entry 
//                    per blind pixel. 
//
// All TClonesArrays have to enlarged by the corresponding calls to (e.g. in MGeomApply): 
// - InitSize()
//
// See also: MCalibrationChargeBlindCamOneOldStyle
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationChargeBlindCam.h"
#include "MCalibrationChargeBlindPix.h"

#include "MCalibrationCam.h"
#include "MParContainer.h"

ClassImp(MCalibrationChargeBlindCam);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Initializes:
// - fPulserColor to kNONE 
//
// Creates a TClonesArray of MCalibrationChargeBlindPix containers for the TClonesArray's: 
// - fBlindPixels
// all initialized to 1 entry
//
// Later, a call to InitSize() 
// has to be performed in order to get the dimension correctly.
//
MCalibrationChargeBlindCam::MCalibrationChargeBlindCam(UInt_t nblind,const char *name, const char *title)
    : fNumBlindPixels(nblind), 
      fPulserColor(MCalibrationCam::kNONE), 
      fBlindPixels(nblind)
{

  fName  = name  ? name  : "MCalibrationChargeBlindCam";
  fTitle = title ? title : "Calibration Information of blinded pixels in camera";

   //
  // make sure that the destructor delete all contained objects
  //
  fBlindPixels.SetOwner();
  
  for (UInt_t i=0; i<nblind; i++)
    fBlindPixels[i] = new MCalibrationChargeBlindPix;
  
}


// --------------------------------------
//
// Calls the ForEach macro for the TClonesArray fBlindPixels with the argument Clear()
// 
void MCalibrationChargeBlindCam::Clear(Option_t *o)
{
  fBlindPixels.ForEach(TObject, Clear)();
}

// -----------------------------------------------------
//
// copy 'constructor'
//
void MCalibrationChargeBlindCam::Copy(TObject& object) const
{

  MParContainer::Copy(object);

  MCalibrationChargeBlindCam &calib = (MCalibrationChargeBlindCam&)object;
  calib.fPulserColor    = fPulserColor;
  calib.fNumBlindPixels = fNumBlindPixels;

  for (UInt_t i=0; i<fNumBlindPixels; i++)
    {
      calib.fBlindPixels[i] = new MCalibrationChargeBlindPix;
      (*this)[i].Copy(calib[i]);
    }
}

// --------------------------------------------------------------------------
//
// Get i-th blind pixel (pixel number)
//
MCalibrationChargeBlindPix &MCalibrationChargeBlindCam::operator[](UInt_t i)
{
  return *static_cast<MCalibrationChargeBlindPix*>(fBlindPixels.UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MCalibrationChargeBlindPix &MCalibrationChargeBlindCam::operator[](UInt_t i) const
{
  return *static_cast<MCalibrationChargeBlindPix*>(fBlindPixels.UncheckedAt(i));
}


// --------------------------------------------------------------------------
//
// Print the results of the blind pixels 
//
void MCalibrationChargeBlindCam::Print(Option_t *o) const
{

  fBlindPixels.Print();
}

// --------------------------------------------------------------------------
//
// Set color to this class and to the MCalibrationBlindPix's
//
void  MCalibrationChargeBlindCam::SetColor ( const MCalibrationCam::PulserColor_t col )
{

  fPulserColor = col;
  fBlindPixels.ForEach(MCalibrationChargeBlindPix, SetColor)(col);

}
