/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibrateSignal
//
// This class is reading the output written by callisto. It calibrates
// signal and time.
//
// The signal and time extractors are read from the callisto-output. In
// pricipal you could overwrite these default using the resource file,
// but this is NOT recommended!
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibrateSignal.h"

#include <TEnv.h>
#include <TFile.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MSequence.h"
#include "MDirIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MJPedestal.h"
#include "MStatusDisplay.h"

#include "MHCamEvent.h"
#include "MPedestalCam.h"
#include "MBadPixelsCam.h"
#include "MCalibrationQECam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"

#include "MReadMarsFile.h"
#include "MGeomApply.h"
#include "MPedCalcFromLoGain.h"
#include "MExtractor.h"
#include "MTaskEnv.h"
#include "MCalibrateData.h"
#include "MCalibrateRelTimes.h"
#include "MBadPixelsMerge.h"
#include "MBadPixelsCalc.h"
#include "MBadPixelsTreat.h"
#include "MFillH.h"
#include "MWriteRootFile.h"

ClassImp(MJCalibrateSignal);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets fRuns to 0, fExtractor to NULL, fDataCheck to kFALSE
//
MJCalibrateSignal::MJCalibrateSignal(const char *name, const char *title) 
    : fOverwrite(kFALSE), fMaxEvents(0)
{
    fName  = name  ? name  : "MJCalibrateSignal";
    fTitle = title ? title : "Tool to calibrate data";
}

MJCalibrateSignal::~MJCalibrateSignal()
{
}

Bool_t MJCalibrateSignal::WriteResult(const MSequence &seq)
{
    if (fOutputPath.IsNull())
        return kTRUE;

    const TString oname = Form("%s/calped%06d.root", (const char*)fOutputPath, seq.GetSequence());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "RECREATE");

    *fLog << inf << " - MStatusDisplay..." << flush;
    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }
    *fLog << inf << "ok." << endl;

    return kTRUE;
}

void MJCalibrateSignal::SetOutputPath(const char *path)
{
    fOutputPath = path;
    if (fOutputPath.EndsWith("/"))
        fOutputPath = fOutputPath(0, fOutputPath.Length()-1);
}
void MJCalibrateSignal::SetInputPath(const char *path)
{
    fInputPath = path;
    if (fInputPath.EndsWith("/"))
        fInputPath = fInputPath(0, fInputPath.Length()-1);
}

void MJCalibrateSignal::CheckEnv(const char *env)
{
    TEnv e(env);

    TString e1 = e.GetValue("MJCalibrateSignal.OutputPath", "");
    if (!e1.IsNull())
    {
        e1.ReplaceAll("\015", "");
        SetOutputPath(e1);
    }

    fMaxEvents = e.GetValue("MJCalibrateSignal.MaxEvents", fMaxEvents);
}

Bool_t MJCalibrateSignal::ReadCalibration(const MSequence &seq, const char *path,
                                          MCalibrationCam &calcam,
                                          MCalibrationCam &qecam,
                                          MCalibrationCam &tmcam,
                                          MBadPixelsCam &badpix, MTask* &ext1, MTask* &ext2) const
{
    TString inpath(path);
    if (inpath.EndsWith("/"))
        inpath = inpath(0, inpath.Length()-1);

    const TString fname = Form("%s/calib%06d.root", inpath.Data(), seq.GetSequence());

    TFile file(fname, "READ");
    if (!file.IsOpen())
    {
        *fLog << err << dbginf << "ERROR - Could not open file " << fname << endl;
        return kFALSE;
    }

    if (calcam.Read("MCalibrationChargeCam")<=0)
    {
        *fLog << err << dbginf << "ERROR - Unable to read MCalibrationChargeCam from file " << fname << endl;
        return kFALSE;
    }
    if (qecam.Read("MCalibrationQECam")<=0)
    {
        *fLog << err << dbginf << "ERROR - Unable to read MCalibrationQECam from file " << fname << endl;
        return kFALSE;
    }
    if (tmcam.Read("MCalibrationRelTimeCam")<=0)
    {
        *fLog << err << dbginf << "ERROR - Unable to read MCalibrationRelTimeCam from file " << fname << endl;
        return kFALSE;
    }
    if (badpix.Read("MBadPixelsCam")<=0)
    {
        *fLog << err << dbginf << "ERROR - Unable to read MBadPixelsCam from file " << fname << endl;
        return kFALSE;
    }

    TObject *o = file.Get("ExtractSignal");
    if (!o)
    {
        *fLog << err << dbginf << "ERROR - Unable to read ExtractSignal from file " << fname << endl;
        return kFALSE;
    }
    if (!o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractSignal read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext1 = (MTask*)o->Clone();

    o = file.Get("ExtractTime");
    if (!o)
    {
        *fLog << err << dbginf << "ERROR - Unable to read ExtractTime from file " << fname << endl;
        return kFALSE;
    }
    if (!o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractTime read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext2 = (MTask*)o->Clone();

    return kTRUE;
}

Bool_t MJCalibrateSignal::ProcessFile(const MSequence &seq, const char *env, const char *path)
{
    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate calibrated data from runs ";
    *fLog << seq.GetName() << endl;
    *fLog << endl;

    CheckEnv(env);

    // -------------------------------------------------------------------------------

    MJPedestal job1;
    job1.SetSequence(&seq);
    job1.SetDisplay(fDisplay);;
    job1.SetEnv(env);

    if (!job1.ProcessFile())
        return kFALSE;

    // --------------------------------------------------------------------------------

    MDirIter iter;
    if (seq.SetupDatRuns(iter, fInputPath)==0)
    {
            *fLog << err << "ERROR - No input files of sequence found!" << endl;
            return kFALSE;
    }

    // Read File
    MCalibrationChargeCam  calcam;
    MCalibrationQECam      qecam;
    MCalibrationRelTimeCam tmcam;
    MBadPixelsCam          badpix;

    MTask *extractor1=0;
    MTask *extractor2=0;

    if (!ReadCalibration(seq, path, calcam, qecam, tmcam, badpix, extractor1, extractor2))
        return kFALSE;

    *fLog << all;
    *fLog << "Extractors read from file" << endl;
    *fLog << "=========================" << endl;
    extractor1->Print();
    *fLog << endl;
    extractor2->Print();
    *fLog << endl;

    // This is necessary for the case in which it is not in the files
    MBadPixelsCam badcam;

    // Setup Parlist
    MParList plist;
    plist.AddToList(this); // take care of fDisplay!
    plist.AddToList(&calcam);
    plist.AddToList(&qecam);     
    plist.AddToList(&tmcam);
    plist.AddToList(&badcam);
    plist.AddToList(&job1.GetPedestalCam());

    // Setup Tasklist
    MTaskList tlist;
    plist.AddToList(&tlist);

    MReadMarsFile read("Events");
    read.DisableAutoScheme();
    read.AddFiles(iter);
    //read.AddFiles(fnamein);

    MGeomApply             apply; // Only necessary to craete geometry
    MBadPixelsMerge        merge(&badpix);
    MPedCalcFromLoGain     pedlo;
    //MExtractSlidingWindow  extsignal;
    MTaskEnv taskenv1("ExtractSignal");
    taskenv1.SetDefault(extractor1);
    //MExtractTimeFastSpline exttime;
    MTaskEnv taskenv2("ExtractTime");
    taskenv2.SetDefault(extractor2);
    MCalibrateData         calib;
    MCalibrateRelTimes     caltm;
    MBadPixelsCalc         bpcal;
    MBadPixelsTreat        treat;

    MHCamEvent evt0("PedestalFLG");
    MHCamEvent evt1("Extra'd");
    MHCamEvent evt2("PedPhot");
    MHCamEvent evt3("Interp'd");
    MHCamEvent evt4("Unsuitable");
    MHCamEvent evt5("Times");
    evt0.SetType(0);
    evt1.SetType(0);
    evt2.SetType(0);
    evt3.SetType(0);
    evt4.SetType(2);
    evt5.SetType(0);
    MFillH fill0(&evt0, "MPedestalCam",        "FillPedestalFLG");
    MFillH fill1(&evt1, "MExtractedSignalCam", "FillExtracted");
    MFillH fill2(&evt2, "MPedPhotCam",         "FillPedPhot");
    MFillH fill3(&evt3, "MCerPhotEvt",         "FillInterpolated");
    MFillH fill4(&evt4, "MBadPixelsCam",       "FillUnsuitable");
    MFillH fill5(&evt5, "MArrivalTime",        "FillTimes");

    MWriteRootFile write(2, Form("%s{s/_D_/_Y_}", fOutputPath.Data()), fOverwrite);
    write.AddContainer("MCerPhotEvt",    "Events");
    write.AddContainer("MPedPhotCam",    "Events");
    write.AddContainer("MTime",          "Events");
    write.AddContainer("MRawEvtHeader",  "Events");
    write.AddContainer("MArrivalTime",   "Events");
    write.AddContainer("MRawRunHeader",  "RunHeaders");
    write.AddContainer("MBadPixelsCam",  "RunHeaders");
    write.AddContainer("MGeomCam",       "RunHeaders");

    tlist.AddToList(&read);
    tlist.AddToList(&apply);
    tlist.AddToList(&merge);
    tlist.AddToList(&pedlo);
    tlist.AddToList(&fill0);
    tlist.AddToList(&taskenv1);
    tlist.AddToList(&taskenv2);
    tlist.AddToList(&fill1);
    tlist.AddToList(&calib);
    tlist.AddToList(&caltm);
    tlist.AddToList(&fill2);
    tlist.AddToList(&bpcal);
    tlist.AddToList(&treat);
    tlist.AddToList(&fill3);
    tlist.AddToList(&fill4);
    tlist.AddToList(&fill5);
    tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    evtloop.ReadEnv(env);

    // Execute first analysis
    if (!evtloop.Eventloop(fMaxEvents))
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    if (!WriteResult(seq))
        return kFALSE;

    *fLog << all << GetDescriptor() << ": Done." << endl;

    return kTRUE;
}
