/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 08/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Daniela Dorner, 08/2004 <mailto:dorner@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// fillcalib.C
// ===========
//
// This macro is used to read the calibartion-/callisto-output files.
// These files are automatically called calib00000.root.
//
// From this file the MBadPixelsCam and the MGeomCam is extracted. If
// the geometry isn't found MGeomCamMagic is used as a default.
//
// Usage:
//   .x fillcalib.C("/data/MAGIC/Period014/calib00000.root", kTRUE)
//
// The second argument is the 'dummy-mode'. If it is kTRUE dummy-mode is
// switched on and nothing will be written into the database. This is usefull
// for tests.
//
// The corresponding sequence number is extracted from the filename...
// FIXME: MSeqeuence should be stored in the calib-file?
//
// The macro can also be run without ACLiC but this is a lot slower...
//
// Remark: Running it from the commandline looks like this:
//   root -q -l -b fillcalib.C+\(\"filename\"\,kFALSE\) 2>&1 | tee fillcalib.log
//
// Make sure, that database and password are corretly set in the macro.
//
// Returns 0 in case of failure and 1 in case of success.
//
/////////////////////////////////////////////////////////////////////////////
#include <iostream>

#include <TEnv.h>
#include <TRegexp.h>

#include <TFile.h>
#include <TSQLResult.h>

#include "MSQLServer.h"

#include "MGeomCamMagic.h"
#include "MBadPixelsCam.h"

using namespace std;

Bool_t Process(MSQLServer &serv, TString fname, Bool_t dummy)
{
    MBadPixelsCam badpix;

    TFile file(fname, "READ");
    if (badpix.Read("MBadPixelsCam")<=0)
    {
        cout << "ERROR - Reading of MBadPixelsCam failed." << endl;
        return kFALSE;
    }

    MGeomCamMagic def;

    MGeomCam *geom = (MGeomCam*)file.Get("MGeomCam");
    if (!geom)
    {
        cout << "WARNING - Reading of MGeomCam failed... using default <MGeomCamMagic>" << endl;
        geom = &def;
    }

    cout << "Camera Geometry: " << geom->ClassName() << endl;

    const Short_t unsin  = badpix.GetNumUnsuitable(MBadPixelsPix::kUnsuitableRun, geom, 0);
    const Short_t unsout = badpix.GetNumUnsuitable(MBadPixelsPix::kUnsuitableRun, geom, 1);

    const Short_t unrin  = badpix.GetNumUnsuitable(MBadPixelsPix::kUnreliableRun, geom, 0);
    const Short_t unrout = badpix.GetNumUnsuitable(MBadPixelsPix::kUnreliableRun, geom, 1);

    const Short_t isoin  = badpix.GetNumIsolated(*geom, 0);
    const Short_t isoout = badpix.GetNumIsolated(*geom, 1);

    const Short_t clumax = badpix.GetNumMaxCluster(*geom);

    if (unsin<0 || unsout<0 || unrin<0 || unrout<0 || isoin<0 || isoout<0 || clumax<0)
        return kFALSE;

    //     MHCamera hist(geom);
    //     hist.SetCamContent(badpix, 1);
    //     hist.DrawCopy();
    //     hist.SetCamContent(badpix, 3);
    //     hist.DrawCopy();

    TString sequence = fname(TRegexp("calib[0-9]+[.]root$"));
    if (sequence.IsNull())
        return kTRUE;

    Int_t seq = atoi(sequence.Data()+5);

    cout << "Sequence #" << seq << endl;
    cout << "  Unsuitable:   (i/o)  " << Form("%3d %3d", (int)unsin, (int)unsout) << endl; // Unbrauchbar
    cout << "  Unreliable:   (i/o)  " << Form("%3d %3d", (int)unrin, (int)unrout) << endl; // Unzuverlaessig
    cout << "  Isolated:     (i/o)  " << Form("%3d %3d", (int)isoin, (int)isoout) << endl; // Isolated (unbrauchbar)
    cout << "  Max.Cluseter: (i/o)  " << Form("%3d", (int)clumax) << endl;                 // Max Cluster

    if (dummy)
        return kTRUE;

    TString query = Form("INSERT MyMagic.Calibration SET"
                         " fSequenceFirst=%d,"
                         " fUnsuitableInner=%d, "
                         " fUnsuitableOuter=%d, "
                         " fUnreliableInner=%d, "
                         " fUnreliableOuter=%d, "
                         " fIsolatedInner=%d, "
                         " fIsolatedOuter=%d, "
                         " fIsolatedMaxCluster=%d",
                         seq, (int)unsin, (int)unsout, (int)unrin,
                         (int)unrout, (int)isoin, (int)isoout, (int)clumax);


    TSQLResult *res = serv.Query(query);
    if (!res)
    {
        cout << "ERROR - Query failed: " << query << endl;
        return kFALSE;
    }

    return kTRUE;
}

void fillcallisto(TString fname, Bool_t dummy=kTRUE)
{
    TEnv env("sql.rc");

    MSQLServer serv(env);
    if (!serv.IsConnected())
    {
        cout << "ERROR - Connection to database failed." << endl;
        return;
    }

    cout << "fillcalib" << endl;
    cout << "---------" << endl;
    cout << endl;
    cout << "Connected to " << serv.GetName() << endl;
    cout << "File: " << fname << endl;
    cout << endl;

    cout << (Process(serv, fname, dummy) ? "Done." : "failed!") << endl << endl;
}
