/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!              
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//                                                                        
// MHCalibrationRelTimeCam                                                
//                                                                        
// Fills the extracted relative arrival times of MArrivalTimeCam into 
// the MHCalibrationPix-classes MHCalibrationPix for every:
//
// - Pixel, stored in the TObjArray's MHCalibrationCam::fHiGainArray  
//   or MHCalibrationCam::fHiGainArray, respectively, depending if 
//   MArrivalTimePix::IsLoGainUsed() is set.
//
// - Average pixel per AREA index (e.g. inner and outer for the MAGIC camera), 
//   stored in the TObjArray's MHCalibrationCam::fAverageHiGainAreas and 
//   MHCalibrationCam::fAverageHiGainAreas
//
// - Average pixel per camera SECTOR (e.g. sectors 1-6 for the MAGIC camera), 
//   stored in the TObjArray's MHCalibrationCam::fAverageHiGainSectors 
//   and MHCalibrationCam::fAverageHiGainSectors 
//
// Every relative time is calculated as the difference between the individual 
// pixel arrival time and the one of pixel 1 (hardware number: 2). 
// The relative times are filled into a histogram and an array, in order to perform 
// a Fourier analysis (see MHGausEvents). The signals are moreover averaged on an 
// event-by-event basis and written into the corresponding average pixels.
//
// The histograms are fitted to a Gaussian, mean and sigma with its errors 
// and the fit probability are extracted. If none of these values are NaN's and 
// if the probability is bigger than MHGausEvents::fProbLimit (default: 0.5%), 
// the fit is declared valid.
// Otherwise, the fit is repeated within ranges of the previous mean 
// +- MHCalibrationPix::fPickupLimit (default: 5) sigma (see MHCalibrationPix::RepeatFit())
// In case this does not make the fit valid, the histogram means and RMS's are 
// taken directly (see MHCalibrationPix::BypassFit()) and the following flags are set:
// - MBadPixelsPix::SetUncalibrated( MBadPixelsPix::kRelTimeNotFitted ) and
// - MBadPixelsPix::SetUnsuitable(   MBadPixelsPix::kUnreliableRun    ) 
// 
// Outliers of more than MHCalibrationPix::fPickupLimit (default: 5) sigmas 
// from the mean are counted as Pickup events (stored in MHCalibrationPix::fPickup) 
//
// The class also fills arrays with the signal vs. event number, creates a fourier 
// spectrum (see MHGausEvents::CreateFourierSpectrum()) and investigates if the 
// projected fourier components follow an exponential distribution. 
// In case that the probability of the exponential fit is less than 
// MHGausEvents::fProbLimit (default: 0.5%), the following flags are set:
// - MBadPixelsPix::SetUncalibrated( MBadPixelsPix::kRelTimeOscillating ) and
// - MBadPixelsPix::SetUnsuitable(   MBadPixelsPix::kUnreliableRun      )
// 
// This same procedure is performed for the average pixels.
//
// The following results are written into MCalibrationRelTimeCam:
//
// - MCalibrationPix::SetMean()
// - MCalibrationPix::SetMeanErr()
// - MCalibrationPix::SetSigma()
// - MCalibrationPix::SetSigmaErr()
// - MCalibrationPix::SetProb()
// - MCalibrationPix::SetNumPickup()
//
// For all averaged areas, the fitted sigma is multiplied with the square root of 
// the number involved pixels in order to be able to compare it to the average of 
// sigmas in the camera.
//                                                                         
/////////////////////////////////////////////////////////////////////////////
#include "MHCalibrationRelTimeCam.h"
#include "MHCalibrationPix.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MCalibrationIntensityRelTimeCam.h"

#include "MCalibrationRelTimeCam.h"
#include "MCalibrationRelTimePix.h"
#include "MCalibrationPix.h"

#include "MArrivalTimeCam.h"
#include "MArrivalTimePix.h"

#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MBadPixelsCam.h"
#include "MBadPixelsPix.h"

ClassImp(MHCalibrationRelTimeCam);

using namespace std;

const Float_t MHCalibrationRelTimeCam::fgNumHiGainSaturationLimit = 0.25;
const UInt_t  MHCalibrationRelTimeCam::fgReferencePixel = 1;
const Int_t   MHCalibrationRelTimeCam::fgNbins    = 900;
const Axis_t  MHCalibrationRelTimeCam::fgFirst    = -5.;
const Axis_t  MHCalibrationRelTimeCam::fgLast     =  5.;
const TString MHCalibrationRelTimeCam::gsHistName   = "RelTime";
const TString MHCalibrationRelTimeCam::gsHistTitle  = "Rel. Arr. Times";
const TString MHCalibrationRelTimeCam::gsHistXTitle = "Rel. Arr. Time [FADC slices]";
const TString MHCalibrationRelTimeCam::gsHistYTitle = "Nr. events";
// --------------------------------------------------------------------------
//
// Default Constructor.
//
// Sets: 
// - fReferencePixel to fgReferencePixel
// - fNbins to fgNbins
// - fFirst to fgFirst
// - fLast  to fgLast 
//
// - fHistName   to gsHistName  
// - fHistTitle  to gsHistTitle 
// - fHistXTitle to gsHistXTitle
// - fHistYTitle to gsHistYTitle
//
MHCalibrationRelTimeCam::MHCalibrationRelTimeCam(const char *name, const char *title) 
{

  fName  = name  ? name  : "MHCalibrationRelTimeCam";
  fTitle = title ? title : "Histogram class for the relative time calibration of the camera";

  SetNumHiGainSaturationLimit(fgNumHiGainSaturationLimit);

  SetReferencePixel();

  SetNbins(fgNbins);
  SetFirst(fgFirst);
  SetLast (fgLast );

  SetHistName  (gsHistName  .Data());
  SetHistTitle (gsHistTitle .Data());
  SetHistXTitle(gsHistXTitle.Data());
  SetHistYTitle(gsHistYTitle.Data());

}

// --------------------------------------------------------------------------
//
// Gets or creates the pointers to:
// - MCalibrationRelTimeCam
//
// Searches pointer to:
// - MArrivalTimeCam
//
// Calls:
// - MHCalibrationCam::InitHiGainArrays()
// - MHCalibrationCam::InitLoGainArrays()
// 
// Sets:
// - fSumareahi   to nareas
// - fSumarealo   to nareas
// - fSumsectorhi to nareas
// - fSumsectorlo to nareas
// - fNumareahi   to nareas
// - fNumarealo   to nareas
// - fNumsectorhi to nareas
// - fNumsectorlo to nareas
//
Bool_t MHCalibrationRelTimeCam::ReInitHists(MParList *pList)
{

  fIntensCam = (MCalibrationIntensityCam*)pList->FindObject(AddSerialNumber("MCalibrationIntensityRelTimeCam"));
  if (fIntensCam)
    *fLog << inf << "Found MCalibrationIntensityRelTimeCam ... " << endl;
  else
    {
      fCam = (MCalibrationCam*)pList->FindObject(AddSerialNumber("MCalibrationRelTimeCam"));
      if (!fCam)
        {
          fCam = (MCalibrationCam*)pList->FindCreateObj(AddSerialNumber("MCalibrationRelTimeCam"));
          if (!fCam)
            {
              *fLog << err << "Cannot find nor create MCalibrationRelTimeCam ... abort." << endl;
              return kFALSE;
            }
          fCam->Init(*fGeom);
        }
    }

  MArrivalTimeCam *signal = (MArrivalTimeCam*)pList->FindObject("MArrivalTimeCam");
  if (!signal)
  {
      *fLog << err << "MArrivalTimeCam not found... abort." << endl;
      return kFALSE;
  }

  const Int_t npixels  = fGeom->GetNumPixels();
  const Int_t nsectors = fGeom->GetNumSectors();
  const Int_t nareas   = fGeom->GetNumAreas();

  InitHiGainArrays(npixels,nareas,nsectors);
  InitLoGainArrays(npixels,nareas,nsectors);

  fSumareahi  .Set(nareas); 
  fSumarealo  .Set(nareas);
  fSumsectorhi.Set(nsectors); 
  fSumsectorlo.Set(nsectors);
  fNumareahi  .Set(nareas); 
  fNumarealo  .Set(nareas);
  fNumsectorhi.Set(nsectors); 
  fNumsectorlo.Set(nsectors);

  return kTRUE;
}


// -------------------------------------------------------------------------------
//
// Retrieves pointer to MArrivalTimeCam:
//
// Retrieves from MGeomCam:
// - number of pixels
// - number of pixel areas
// - number of sectors
//
// Fills HiGain or LoGain histograms (MHGausEvents::FillHistAndArray()), respectively
// depending on MArrivalTimePix::IsLoGainUsed(), with:
// - MArrivalTimePix::GetArrivalTime(pixid) - MArrivalTimePix::GetArrivalTime(1);
//   (i.e. the time difference between pixel i and pixel 1 (hardware number: 2) )
//
Bool_t MHCalibrationRelTimeCam::FillHists(const MParContainer *par, const Stat_t w)
{

  MArrivalTimeCam *arrtime = (MArrivalTimeCam*)par;
  if (!arrtime)
    {
      gLog << err << "No argument in MArrivalTime::Fill... abort." << endl;
      return kFALSE;
    }
  
  const Int_t npixels  = fGeom->GetNumPixels();
  const Int_t nareas   = fGeom->GetNumAreas();
  const Int_t nsectors = fGeom->GetNumSectors();

  fSumareahi  .Reset();
  fSumarealo  .Reset();
  fSumsectorhi.Reset(); 
  fSumsectorlo.Reset();
  fNumareahi  .Reset(); 
  fNumarealo  .Reset();
  fNumsectorhi.Reset(); 
  fNumsectorlo.Reset();

  const MArrivalTimePix &refpix = (*arrtime)[fReferencePixel];
  const Float_t reftime = refpix.IsLoGainUsed() 
    ? refpix.GetArrivalTimeLoGain() : refpix.GetArrivalTimeHiGain();

  for (Int_t i=0; i<npixels; i++)
    {

      MHCalibrationPix &histhi = (*this)[i];
      MHCalibrationPix &histlo = (*this)(i);

      if (histhi.IsExcluded())
	continue;

      const MArrivalTimePix &pix = (*arrtime)[i];
      const Int_t aidx   = (*fGeom)[i].GetAidx();
      const Int_t sector = (*fGeom)[i].GetSector();

      if (pix.IsLoGainUsed() && IsLoGain())
        { 
          const Float_t reltime = pix.GetArrivalTimeLoGain() - reftime;
          histhi.SetSaturated(1); 
          histlo.FillHistAndArray(reltime);
          fSumarealo  [aidx]   += reltime;
          fNumarealo  [aidx]   ++;
          fSumsectorlo[sector] += reltime;
          fNumsectorlo[sector] ++;
        }
      else
        {
          const Float_t reltime = pix.GetArrivalTimeHiGain() - reftime;

          histhi.FillHistAndArray(reltime) ;
          fSumareahi  [aidx]   += reltime;
          fNumareahi  [aidx]   ++;
          fSumsectorhi[sector] += reltime;
          fNumsectorhi[sector] ++;
        }
    }
  
  for (Int_t j=0; j<nareas; j++)
    {
      MHCalibrationPix &histhi = GetAverageHiGainArea(j);
      histhi.FillHistAndArray(fNumareahi[j] == 0 ? 0. : fSumareahi[j]/fNumareahi[j]);

      if (IsLoGain())
        {
          MHCalibrationPix &histlo = GetAverageLoGainArea(j);
          histlo.FillHistAndArray(fNumarealo[j] == 0 ? 0. : fSumarealo[j]/fNumarealo[j]);
        }
    }
  
  for (Int_t j=0; j<nsectors; j++)
    {
      MHCalibrationPix &histhi = GetAverageHiGainSector(j);
      histhi.FillHistAndArray(fNumsectorhi[j] == 0 ? 0. : fSumsectorhi[j]/fNumsectorhi[j]);

      if (IsLoGain())
        {
          MHCalibrationPix &histlo = GetAverageLoGainSector(j);
          histlo.FillHistAndArray(fNumsectorlo[j] == 0 ? 0. : fSumsectorlo[j]/fNumsectorlo[j]);
        }
    }

  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calls:
// - MHCalibrationCam::FitHiGainArrays() with flags:
//   MBadPixelsPix::kRelTimeNotFitted and MBadPixelsPix::kRelTimeOscillating
// - MHCalibrationCam::FitLoGainArrays() with flags:
//   MBadPixelsPix::kRelTimeNotFitted and MBadPixelsPix::kRelTimeOscillating
// 
Bool_t MHCalibrationRelTimeCam::FinalizeHists()
{
  for (Int_t i=0; i<fHiGainArray->GetSize(); i++)
    {
      
      MHCalibrationPix &histhi = (*this)[i];
      
      if (histhi.IsExcluded())
        continue;
      
      MCalibrationRelTimePix &pix = fIntensCam 
        ? (MCalibrationRelTimePix&)(*fIntensCam)[i] 
        : (MCalibrationRelTimePix&)(*fCam)[i];

      if (histhi.GetSaturated() > fNumHiGainSaturationLimit*histhi.GetHGausHist()->GetEntries())
        {
          pix.SetHiGainSaturation();
          histhi.SetExcluded();
        }
      else
        if (IsLoGain())
          (*this)(i).SetExcluded();

      Stat_t overflow = histhi.GetHGausHist()->GetBinContent(histhi.GetHGausHist()->GetNbinsX()+1);
      if (overflow > 0.1)
        {
          *fLog << warn << GetDescriptor()
                << ": HiGain Histogram Overflow occurred " << overflow 
                << " times in pixel: " << i << " (without saturation!) " << endl;
          //          bad.SetUncalibrated( MBadPixelsPix::kHiGainOverFlow ); 
        }

      overflow = histhi.GetHGausHist()->GetBinContent(0);
      if (overflow > 0.1)
        {
          *fLog << warn << GetDescriptor()
                << ": HiGain Histogram Underflow occurred " << overflow
                << " times in pixel: " << i << " (without saturation!) " << endl;
          //          bad.SetUncalibrated( MBadPixelsPix::kHiGainOverFlow ); 
        }
    }

  for (Int_t j=0; j<fAverageHiGainAreas->GetSize(); j++)
    {
      
      MHCalibrationPix &histhi = GetAverageHiGainArea(j);      

      if (histhi.GetSaturated() > fNumHiGainSaturationLimit*histhi.GetHGausHist()->GetEntries())
        {
          MCalibrationRelTimePix  &pix    = fIntensCam
            ? (MCalibrationRelTimePix&)fIntensCam->GetAverageArea(j)
            : (MCalibrationRelTimePix&)fCam->GetAverageArea(j);
          pix.SetHiGainSaturation();
          histhi.SetExcluded();
        }
      else
        if (IsLoGain())
          GetAverageLoGainArea(j).SetExcluded();

   }
  
  for (Int_t j=0; j<fAverageHiGainSectors->GetSize(); j++)
    {
      
      MHCalibrationPix &histhi = GetAverageHiGainSector(j);      
      MCalibrationRelTimePix  &pix = fIntensCam
        ? (MCalibrationRelTimePix&)fIntensCam->GetAverageSector(j)  
        : (MCalibrationRelTimePix&)fCam->GetAverageSector(j);

      if (histhi.GetSaturated() > fNumHiGainSaturationLimit*histhi.GetHGausHist()->GetEntries())
        {
          pix.SetHiGainSaturation();
          histhi.SetExcluded();
        }
      else
        if (IsLoGain())        
          GetAverageLoGainSector(j).SetExcluded();
    }

  FitHiGainArrays(fIntensCam ? (MCalibrationCam&)(*fIntensCam->GetCam()) : (MCalibrationCam&)(*fCam),
                  *fBadPixels,
                  MBadPixelsPix::kRelTimeNotFitted,
                  MBadPixelsPix::kRelTimeOscillating);
  
  if (IsLoGain())        
    FitLoGainArrays(fIntensCam ? (MCalibrationCam&)(*fIntensCam->GetCam()) : (MCalibrationCam&)(*fCam),
                    *fBadPixels,
                    MBadPixelsPix::kRelTimeNotFitted,
                    MBadPixelsPix::kRelTimeOscillating);

  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Sets all pixels to MBadPixelsPix::kUnreliableRun, if following flags are set:
// - MBadPixelsPix::kRelTimeNotFitted
// - MBadPixelsPix::kRelTimeOscillating
//
void MHCalibrationRelTimeCam::FinalizeBadPixels()
{

  for (Int_t i=0; i<fBadPixels->GetSize(); i++)
    {
      
      MBadPixelsPix          &bad    = (*fBadPixels)[i];

      if (bad.IsUncalibrated( MBadPixelsPix::kRelTimeNotFitted ))
        bad.SetUnsuitable(   MBadPixelsPix::kUnreliableRun    );

      if (bad.IsUncalibrated( MBadPixelsPix::kRelTimeOscillating))
        bad.SetUnsuitable(   MBadPixelsPix::kUnreliableRun    );
      
    }
}

// --------------------------------------------------------------------------
//
// The types are as follows:
// 
// Fitted values:
// ============== 
//
// 0: Fitted Mean Relative Arrival Time in FADC slices  (MHGausEvents::GetMean()
// 1: Error Mean Relative Arrival Time in FADC slices   (MHGausEvents::GetMeanErr()
// 2: Sigma fitted Relative Arrival Time in FADC slices (MHGausEvents::GetSigma()
// 3: Error Sigma Relative Arrival Time in FADC slices  (MHGausEvents::GetSigmaErr()
//
// Useful variables derived from the fit results:
// =============================================
//
// 4: Returned probability of Gauss fit              (calls: MHGausEvents::GetProb())
//
// Localized defects:
// ==================
//
// 5: Gaus fit not OK                               (calls: MHGausEvents::IsGausFitOK())
// 6: Fourier spectrum not OK                       (calls: MHGausEvents::IsFourierSpectrumOK())
//
Bool_t MHCalibrationRelTimeCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (fHiGainArray->GetSize() <= idx)
    return kFALSE;

  const MHCalibrationPix &pix = (*this)[idx];

  switch (type)
    {
    case 0:
      val = pix.GetMean();
      break;
    case 1:
      val = pix.GetMeanErr();
      break;
    case 2:
      val = pix.GetSigma();
      break;
    case 3:
      val = pix.GetSigmaErr();
      break;
    case 4:
      val = pix.GetProb();
      break;
    case 5:
      if (!pix.IsGausFitOK())
        val = 1.;
      break;
    case 6:
      if (!pix.IsFourierSpectrumOK())
        val = 1.;
      break;
    default:
      return kFALSE;
    }
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calls MHCalibrationPix::DrawClone() for pixel idx
//
void MHCalibrationRelTimeCam::DrawPixelContent(Int_t idx) const
{
 (*this)[idx].DrawClone();
}
