/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz   12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!              Markus Gaug    02/2004 <mailto:markus@ifae.es>
!              Florian Goebel 06/2004 <mailto:fgoebel@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MPedestalCam                                                            //
//                                                                         //
// Hold the Pedestal information for all pixels in the camera              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MPedestalCam.h"
#include "MPedestalPix.h"

#include <TArrayI.h>
#include <TArrayF.h>
#include <TArrayD.h>

#include <TClonesArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MBadPixelsCam.h"
#include "MBadPixelsPix.h"

ClassImp(MPedestalCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Creates a TClonesArray of MPedestalPix containers, initialized to 1 entry, destinated 
// to hold one container per pixel. Later, a call to MPedestalCam::InitSize() 
// has to be performed (in MGeomApply). 
//
// Creates a TClonesArray of MPedestalPix containers, initialized to 1 entry, destinated 
// to hold one container per pixel AREA. Later, a call to MPedestalCam::InitAreas() 
// has to be performed (in MGeomApply). 
//
// Creates a TClonesArray of MPedestalPix containers, initialized to 1 entry, destinated
// to hold one container per camera SECTOR. Later, a call to MPedestalCam::InitSectors() 
// has to be performed (in MGeomApply). 
//
MPedestalCam::MPedestalCam(const char *name, const char *title) 
    : fTotalEntries(0)
{
  fName  = name  ? name  : "MPedestalCam";
  fTitle = title ? title : "Storage container for all Pedestal Information in the camera";
  
  fArray            = new TClonesArray("MPedestalPix", 1);
  fAverageAreas     = new TClonesArray("MPedestalPix", 1);
  fAverageSectors   = new TClonesArray("MPedestalPix", 1);
}

// --------------------------------------------------------------------------
//
// Deletes the following TClonesArray's of MPedestalPix containers (if exist):
// - fArray
// - fAverageAreas
// - fAverageSectors
//  
MPedestalCam::~MPedestalCam()
{
  delete fArray;
  delete fAverageAreas;
  delete fAverageSectors;
}

// --------------------------------------------------------------------------
//
// Copy 'constructor'
//
void MPedestalCam::Copy(TObject &object) const
{

  MPedestalCam &cam = (MPedestalCam&)object;
  
  const Int_t n = GetSize();
  
  if (n==0)
    return;
  
  cam.InitSize(n);
  for (int i=0; i<n; i++)
    (*this)[i].Copy(cam[i]);
}

// --------------------------------------------------------------------------
//
// Set the size of the camera
//
void MPedestalCam::InitSize(const UInt_t i)
{
    fArray->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls TClonesArray::ExpandCreate() for:
// - fAverageAreas
//
void MPedestalCam::InitAverageAreas(const UInt_t i)
{
  fAverageAreas->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls TClonesArray::ExpandCreate() for:
// - fAverageSectors
//
void MPedestalCam::InitAverageSectors(const UInt_t i)
{
  fAverageSectors->ExpandCreate(i);
}

// -------------------------------------------------------------------
//
// Calls:
// - InitSize()
// - InitAverageAreas()
// - InitAverageSectors()
//
void MPedestalCam::Init(const MGeomCam &geom)
{
  InitSize          (geom.GetNumPixels() );
  InitAverageAreas  (geom.GetNumAreas()  );
  InitAverageSectors(geom.GetNumSectors());
}

// --------------------------------------------------------------------------
//
// This function returns the current size of the TClonesArray 
// independently if the MPedestalPix is filled with values or not.
//
// Get the size of the MPedestalCam
//
Int_t MPedestalCam::GetSize() const
{
    return fArray->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Returns the current size of the TClonesArray fAverageAreas
// independently if the MPedestalPix is filled with values or not.
//
const Int_t MPedestalCam::GetAverageAreas() const
{
  return fAverageAreas->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Returns the current size of the TClonesArray fAverageSectors
// independently if the MPedestalPix is filled with values or not.
//
const Int_t MPedestalCam::GetAverageSectors() const
{
  return fAverageSectors->GetEntriesFast();
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
MPedestalPix &MPedestalCam::operator[](Int_t i)
{
    return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th pixel (pixel number)
//
const MPedestalPix &MPedestalCam::operator[](Int_t i) const
{
    return *static_cast<MPedestalPix*>(fArray->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
MPedestalPix &MPedestalCam::GetAverageArea(UInt_t i)
{
  return *static_cast<MPedestalPix*>(fAverageAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (area number)
//
const MPedestalPix &MPedestalCam::GetAverageArea(UInt_t i) const 
{
  return *static_cast<MPedestalPix*>(fAverageAreas->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
MPedestalPix &MPedestalCam::GetAverageSector(UInt_t i)
{
  return *static_cast<MPedestalPix*>(fAverageSectors->UncheckedAt(i));
}

// --------------------------------------------------------------------------
//
// Get i-th average pixel (sector number)
//
const MPedestalPix &MPedestalCam::GetAverageSector(UInt_t i) const 
{
  return *static_cast<MPedestalPix*>(fAverageSectors->UncheckedAt(i));
}

// --------------------------------------
//
// Calls the ForEach macro for the TClonesArray fArray with the argument Clear()
// 
// Loops over the fAverageAreas, calling the function Clear() for 
// every entry in fAverageAreas
//
// Loops over the fAverageSectors, calling the function Clear() for 
// every entry in fAverageSectors
// 
void MPedestalCam::Clear(Option_t *o)
{
  fArray->ForEach(TObject, Clear)();
  
  //
  // another ForEach does not compile, thus have to do the loop ourselves:
  //
  for (Int_t i=0;i<GetAverageAreas();i++)
    fAverageAreas[i].Clear();


  //
  // another ForEach does not compile, thus have to do the loop ourselves:
  //
  for (Int_t i=0;i<GetAverageSectors();i++)
    fAverageSectors[i].Clear();
  
  fTotalEntries = 0;
}

void MPedestalCam::Print(Option_t *o) const
{
    *fLog << all << GetDescriptor() << ":" << endl;
    int id = 0;

    TIter Next(fArray);
    MPedestalPix *pix;
    while ((pix=(MPedestalPix*)Next()))
    {
        id++;

        if (!pix->IsValid())
            continue;

        *fLog << id-1 << ": ";
        *fLog << pix->GetPedestal() << " " << pix->GetPedestalRms() << endl;
    }
}

Float_t MPedestalCam::GetPedestalMin(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t minval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval < minval)
            minval = testval;
    }
    return minval;
}

Float_t MPedestalCam::GetPedestalMax(const MGeomCam *geom) const
{
    if (fArray->GetEntries() <= 0)
        return 50.;

    Float_t maxval = (*this)[0].GetPedestalRms();

    for (Int_t i=1; i<fArray->GetEntries(); i++)
    {
        const MPedestalPix &pix = (*this)[i];

        Float_t testval = pix.GetPedestalRms();

        if (geom)
            testval *= geom->GetPixRatio(i);

        if (testval > maxval)
            maxval = testval;
    }
    return maxval;
}

// --------------------------------------------------------------------------
//
// Calculates the average pedestal for pixel sizes. 
// The geometry container is used to get the necessary
// geometry information (area index) If the bad pixel
// container is given all pixels which have the flag 'kUnsuitableRun' are ignored
// in the calculation of the size average.
//
// Returns a TArrayF of dimension 2: 
// arr[0]: averaged pedestal (default: -1.)
// arr[1]: Error (rms) of averaged pedestal (default: 0.)
//
// ATTENTION: THE USER HAS TO DELETE THE RETURNED TARRAYF ACCORDINGLY
//
TArrayF *MPedestalCam::GetAveragedPedPerArea(const MGeomCam &geom, const UInt_t ai, MBadPixelsCam *bad)
{

  const Int_t np = GetSize();

  Double_t mean  = 0.;
  Double_t mean2 = 0.;
  Int_t    nr    = 0;

  for (int i=0; i<np; i++)
    {
      if (bad && (*bad)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        continue; 
      
      const UInt_t aidx = geom[i].GetAidx();
      
      if (ai != aidx)
        continue;

      const MPedestalPix &pix = (*this)[i];
      
      mean  += pix.GetPedestal();
      mean2 += pix.GetPedestal()*pix.GetPedestal();
      nr    ++;
      
    }

  TArrayF *arr = new TArrayF(2);
  arr->AddAt(nr   ? mean/nr : -1.,0);
  arr->AddAt(nr>1 ? TMath::Sqrt((mean2 - mean*mean/nr)/(nr-1)) : 0. ,1);

  return arr;
}

// --------------------------------------------------------------------------
//
// Calculates the average pedestal rms for pixel sizes. 
// The geometry container is used to get the necessary
// geometry information (area index) If the bad pixel
// container is given all pixels which have the flag 'kUnsuitableRun' are ignored
// in the calculation of the size average.
//
// Returns a TArrayF of dimension 2: 
// arr[0]: averaged pedestal RMS (default: -1.)
// arr[1]: Error (rms) of averaged pedestal RMS (default: 0.)
//
// ATTENTION: THE USER HAS TO DELETE THE RETURNED TARRAYF ACCORDINGLY
//
TArrayF *MPedestalCam::GetAveragedRmsPerArea(const MGeomCam &geom, const UInt_t ai, MBadPixelsCam *bad)
{

  const Int_t np = GetSize();

  Double_t rms  = 0.;
  Double_t rms2 = 0.;
  Int_t    nr   = 0;

  for (int i=0; i<np; i++)
    {
      if (bad && (*bad)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        continue; 
      
      const UInt_t aidx = geom[i].GetAidx();
      
      if (ai != aidx)
        continue;

      const MPedestalPix &pix = (*this)[i];
      
      rms  += pix.GetPedestalRms();
      rms2 += pix.GetPedestalRms()*pix.GetPedestalRms();
      nr   ++;
    }

  TArrayF *arr = new TArrayF(2);
  arr->AddAt(nr   ? rms/nr : -1.,0);
  arr->AddAt(nr>1 ? TMath::Sqrt((rms2 - rms*rms/nr)/(nr-1)) : 0. ,1);

  return arr;
}

// --------------------------------------------------------------------------
//
// Calculates the average pedestal for camera sectors. 
// The geometry container is used to get the necessary
// geometry information (area index) If the bad pixel
// container is given all pixels which have the flag 'kUnsuitableRun' are ignored
// in the calculation of the size average.
//
// Returns a TArrayF of dimension 2: 
// arr[0]: averaged pedestal (default: -1.)
// arr[1]: Error (rms) of averaged pedestal (default: 0.)
//
// ATTENTION: THE USER HAS TO DELETE THE RETURNED TARRAYF ACCORDINGLY
//
TArrayF *MPedestalCam::GetAveragedPedPerSector(const MGeomCam &geom, const UInt_t sec, MBadPixelsCam *bad)
{

  const Int_t np = GetSize();

  Double_t mean = 0.;
  Double_t mean2 = 0.;
  Int_t    nr   = 0;

  for (int i=0; i<np; i++)
    {
      if (bad && (*bad)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        continue;
      
      const UInt_t sector = geom[i].GetSector();
      
      if (sec != sector)
        continue;

      const MPedestalPix &pix = (*this)[i];
      
      mean  += pix.GetPedestal();
      mean2 += pix.GetPedestal()*pix.GetPedestal();
      nr    ++;
      
    }

  TArrayF *arr = new TArrayF(2);
  arr->AddAt(nr   ? mean/nr : -1.,0);
  arr->AddAt(nr>1 ? TMath::Sqrt((mean2 - mean*mean/nr)/(nr-1)) : 0. ,1);

  return arr;
}

// --------------------------------------------------------------------------
//
// Calculates the average pedestal rms for camera sectors. 
// The geometry container is used to get the necessary
// geometry information (area index) If the bad pixel
// container is given all pixels which have the flag 'kUnsuitableRun' are ignored
// in the calculation of the size average.
//
// Returns a TArrayF of dimension 2: 
// arr[0]: averaged pedestal RMS (default: -1.)
// arr[1]: Error (rms) of averaged pedestal RMS (default: 0.)
//
// ATTENTION: THE USER HAS TO DELETE THE RETURNED TARRAYF ACCORDINGLY
//
TArrayF *MPedestalCam::GetAveragedRmsPerSector(const MGeomCam &geom, const UInt_t sec, MBadPixelsCam *bad)
{

  const Int_t np = GetSize();

  Double_t rms  = 0.;
  Double_t rms2 = 0.;
  Int_t    nr   = 0;

  for (int i=0; i<np; i++)
    {
      if (bad && (*bad)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
        continue;
      
      const UInt_t sector = geom[i].GetSector();
      
      if (sec != sector)
        continue;

      const MPedestalPix &pix = (*this)[i];
      
      rms  += pix.GetPedestalRms();
      rms2 += pix.GetPedestalRms()*pix.GetPedestalRms();
      nr   ++;
      
    }

  TArrayF *arr = new TArrayF(2);
  arr->AddAt(nr   ? rms/nr : -1.,0);
  arr->AddAt(nr>1 ? TMath::Sqrt((rms2 - rms*rms/nr)/(nr-1)) : 0. ,1);

  return arr;
  
}


// --------------------------------------------------------------------------
//
// Calculates the avarage pedestal and pedestal rms for all sectors
// and pixel sizes. The geometry container is used to get the necessary
// geometry information (sector number, size index) If the bad pixel
// container is given all pixels which have the flag 'kUnsuitableRun' are ignored
// in the calculation of the sector and size average.
//
void MPedestalCam::ReCalc(const MGeomCam &geom, MBadPixelsCam *bad)
{
    const Int_t np = GetSize();
    const Int_t ns = geom.GetNumSectors();
    const Int_t na = geom.GetNumAreas();

    TArrayI acnt(na);
    TArrayI scnt(ns);
    TArrayD asumx(na);
    TArrayD ssumx(ns);
    TArrayD asumr(na);
    TArrayD ssumr(ns);

    for (int i=0; i<np; i++)
    {
        if (bad && (*bad)[i].IsUnsuitable(MBadPixelsPix::kUnsuitableRun))
            continue; //was: .IsBad()

        // Create sums for areas and sectors
        const UInt_t aidx = geom[i].GetAidx();
        const UInt_t sect = geom[i].GetSector();

        const MPedestalPix &pix = (*this)[i];

        const UInt_t  ne   = pix.GetNumEvents();
        const Float_t mean = pix.GetPedestal();
        const Float_t rms  = pix.GetPedestalRms();

        asumx[aidx] += ne*mean;
        asumr[aidx] += ne*rms;
        acnt[aidx]  += ne;

        ssumx[sect] += ne*mean;
        ssumr[sect] += ne*rms;
        scnt[sect]  += ne;
    }

    for (int i=0; i<ns; i++)
        if (scnt[i]>0)
            GetAverageSector(i).Set(ssumx[i]/scnt[i], ssumr[i]/scnt[i], 0, scnt[i]);
        else
            GetAverageSector(i).Clear();

    for (int i=0; i<na; i++)
        if (acnt[i]>0)
            GetAverageArea(i).Set(asumx[i]/acnt[i], asumr[i]/acnt[i], 0, acnt[i]);
        else
            GetAverageArea(i).Clear();
}

Bool_t MPedestalCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{
    if (GetSize() <= idx)
        return kFALSE;

    if (!(*this)[idx].IsValid())
        return kFALSE;

    const Float_t ped      = (*this)[idx].GetPedestal();
    const Float_t rms      = (*this)[idx].GetPedestalRms();

    switch (type)
    {
    case 0:
        val = ped;
        break;
    case 1:
        val = fTotalEntries > 0 ?
            rms/TMath::Sqrt((Float_t)fTotalEntries)
          : (*this)[idx].GetPedestalError();
        break;
    case 2:
        val = rms;
        break;
    case 3:
        val = fTotalEntries > 0 ?
          rms/TMath::Sqrt((Float_t)fTotalEntries*2.)
          : (*this)[idx].GetPedestalRmsError();
        break;
    default:
        return kFALSE;
    }
    return kTRUE;
}

void MPedestalCam::DrawPixelContent(Int_t idx) const
{
    *fLog << warn << "MPedestalCam::DrawPixelContent - not available." << endl;
}
