/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Josep Flix   09/2004 <mailto:jflix@ifae.es>
!
!   Copyright: MAGIC Software Development, 2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MTopologyCalc
//
// Taks that evaluates the Topological Parameters of images after image cleaning
//
// fDistance = SUM_(i,j) D_ij , where i,j<UsedPixels and d_ij are distances
//                              between pixels. This characterizes topology.
//
// fUsed = Used Pixels after image cleaning.
//
/////////////////////////////////////////////////////////////////////////////

#include "MTopologyCalc.h"

#include "MParList.h"

#include "MTopology.h"

#include "MCerPhotEvt.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MTopologyCalc);

using namespace std;

static const TString gsDefName  = "MTopologyCalc";
static const TString gsDefTitle = "Calculate Topology related parameters";

// -------------------------------------------------------------------------
//
// Default constructor.
//
MTopologyCalc::MTopologyCalc(const char *name, const char *title)
    : fTopology(NULL)
{
    fName  = name  ? name  : gsDefName.Data();
    fTitle = title ? title : gsDefTitle.Data();

}

// -------------------------------------------------------------------------
//
Int_t MTopologyCalc::PreProcess(MParList *pList)
{

    fCerPhotEvt = (MCerPhotEvt*)pList->FindObject("MCerPhotEvt");
    if (!fCerPhotEvt)
    {
        *fLog << err << dbginf << "MCerPhotEvt not found... aborting." << endl;
        return kFALSE;
    }

    fGeomCam = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!fGeomCam)
    {
        *fLog << err << dbginf << "MGeomCam not found... aborting." << endl;
        return kFALSE;
    }

    fTopology = (MTopology*)pList->FindCreateObj("MTopology");
    if (!fTopology)
        return kFALSE;
 
    return kTRUE;

}

// -------------------------------------------------------------------------
//
Int_t MTopologyCalc::Process()
{

    Int_t rc = fTopology->Calc(*fGeomCam, *fCerPhotEvt);
    
    if (rc == 1)
	fErrors++;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  Prints some statistics about the Topology calculation. The percentage
//  is calculated with respect to the number of executions of this task.
//
Int_t MTopologyCalc::PostProcess()
{
    if (GetNumExecutions()==0)
        return kTRUE;

    *fLog << inf << endl;
    *fLog << GetDescriptor() << " execution statistics:" << endl;
    *fLog << dec << setfill(' ');
    *fLog << " " << fErrors << " (" << (int)(fErrors*100/GetNumExecutions()) 
	  << "%) Evts skipped due to UsedPixels == -1" << endl;
    *fLog << endl;

    /*
    delete fHillas;
    delete fHillasSrc;
    delete fSrcPos;
    delete fTopology;
    */

    return kTRUE;
}
