/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 12/2004 <mailto:markus@ifae.es>
!         
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
//////////////////////////////////////////////////////////////////////////////
//
//  MCalibCalcFromPast
//
//  Steers the occurrance of interlaced calibration events in one data run
// 
//  Input Containers:
//   MRawEvtHeader
//   MParList
//   MCalibrationIntensityChargeCam
//   MCalibrationIntensityRelTimeCam
//   MBadPixelsIntensityCam 
//
//  Output Containers:
//   MCalibrationIntensityChargeCam
//   MCalibrationIntensityRelTimeCam
//   MBadPixelsIntensityCam 
//
//////////////////////////////////////////////////////////////////////////////
#include "MCalibCalcFromPast.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MTaskList.h"

#include "MHCalibrationCam.h"

#include "MCalibrationIntensityChargeCam.h"
#include "MCalibrationIntensityBlindCam.h"
#include "MCalibrationIntensityQECam.h"
#include "MCalibrationIntensityRelTimeCam.h"
#include "MCalibrationIntensityHiLoCam.h"

#include "MBadPixelsIntensityCam.h"

#include "MCalibrationChargeCalc.h"
#include "MCalibrationRelTimeCalc.h"

#include "MRawRunHeader.h"
#include "MRawEvtHeader.h"

#include "MGeomCam.h"

ClassImp(MCalibCalcFromPast);

using namespace std;

// --------------------------------------------------------------------------
//
//  Default constructor. 
//
MCalibCalcFromPast::MCalibCalcFromPast(const char *name, const char *title)
    : fHeader(NULL), fGeom(NULL), fParList(NULL), 
      fIntensCharge(NULL), fIntensRelTime(NULL), fIntensBad(NULL),
      fChargeCalc(NULL), fRelTimeCalc(NULL),
      fPattern(0)
{

  fName  = name  ? name  : "MCalibCalcFromPast";
  fTitle = title ? title : "Task to steer the processing of interlaced calibration events";

}

// -----------------------------------------------------------------------------------
//
// The following container are searched for and execution aborted if not in MParList:
//  - MRawEvtHeader
//  - MTaskList
//
Int_t MCalibCalcFromPast::PreProcess(MParList *pList)
{

  fHeader = (MRawEvtHeader*)pList->FindObject("MRawEvtHeader");
  if (!fHeader)
    {
      *fLog << err << "MRawEvtHeader not found... abort." << endl;
      return kFALSE;
    }

  fRunHeader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
  if (!fHeader)
    {
      *fLog << err << "MRawRunHeader not found... abort." << endl;
      return kFALSE;
    }

  fGeom = (MGeomCam*)pList->FindObject("MGeomCam");
  if (!fGeom)
    {
      *fLog << err << "MGeomCam not found... abort." << endl;
      return kFALSE;
    }

  fParList = pList;
  if (!fParList)
    {
      *fLog << err << "MParList not found... abort." << endl;
      return kFALSE;
    }

  MTaskList *tlist = (MTaskList*)pList->FindObject("MTaskList");
  if (!tlist)
    {
      *fLog << err << "MTaskList not found... abort." << endl;
      return kFALSE;
    }

  // 
  // Look for the MBadPixels Intensity Cam
  //
  fIntensBad = (MBadPixelsIntensityCam*)pList->FindCreateObj("MBadPixelsIntensityCam");
  if (fIntensBad)
    *fLog << inf << "Found MBadPixelsIntensityCam ... " << flush;
  else
    return kFALSE;
  
  // 
  // Look for the MCalibrationIntensityBlindCam
  //
  fIntensBlind = (MCalibrationIntensityBlindCam*)pList->FindCreateObj("MCalibrationIntensityBlindCam");
  if (fIntensBlind)
    *fLog << inf << "Found MCalibrationIntensityBlindCam ... " << flush;
  else
    return kFALSE;
  
  // 
  // Look for the MFillH name "FillChargeCam". In case yes, initialize the 
  // corresponding IntensityCam
  //
  if (pList->FindObject(AddSerialNumber("MHCalibrationChargeCam")))
  {

    fIntensCharge = (MCalibrationIntensityChargeCam*)pList->FindCreateObj("MCalibrationIntensityChargeCam");
    fIntensQE     = (MCalibrationIntensityQECam*)    pList->FindCreateObj("MCalibrationIntensityQECam");

    fChargeCalc   = (MCalibrationChargeCalc*)tlist->FindObject("MCalibrationChargeCalc");

    *fLog << inf << "Found MHCalibrationChargeCam ... " << flush;

    if (!fIntensCharge)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensityChargeCam abort... " << endl;
        return kFALSE;
      }
    
    if (!fIntensQE)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensityQECam abort... " << endl;
        return kFALSE;
      }
    
    if (!fChargeCalc)
      {
        *fLog << err << "Could not find MCalibrationChargeCalc abort... " << endl;
        return kFALSE;
      }
  }
      
  // 
  // Look for the MFillH name "FillRelTimeCam". In case yes, initialize the 
  // corresponding IntensityCam
  //
  if (pList->FindObject(AddSerialNumber("MHCalibrationRelTimeCam")))
  {

    fIntensRelTime = (MCalibrationIntensityRelTimeCam*)pList->FindCreateObj("MCalibrationIntensityRelTimeCam");
    fRelTimeCalc   = (MCalibrationRelTimeCalc*)tlist->FindObject(AddSerialNumber("MCalibrationRelTimeCalc"));

    *fLog << inf << "Found MHCalibrationRelTimeCam ... " << flush;

    if (!fIntensRelTime)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensityRelTimeCam abort... " << endl;
        return kFALSE;
      }

    if (!fRelTimeCalc)
      {
        *fLog << err << "Could not find MCalibrationRelTimeCalc abort... " << endl;
        return kFALSE;
      }
  }
      
  // 
  // Look for the MFillH name "FillRelTimeCam". In case yes, initialize the 
  // corresponding IntensityCam
  //
  if (pList->FindObject(AddSerialNumber("MHCalibrationHiLoCam")))
  {

    fIntensHiLo = (MCalibrationIntensityHiLoCam*)pList->FindCreateObj("MCalibrationIntensityHiLoCam");

    *fLog << inf << "Found MHCalibrationHiLoCam ... " << flush;

    if (!fIntensHiLo)
      {
        *fLog << err << "Could not find nor create MCalibrationIntensityHiLoCam abort... " << endl;
        return kFALSE;
      }
  }
      
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Reads the pattern from MRawEvtHeader and initializes new containers in the 
// Intensity Cams, if the pattern has changed. Executes CallPostProcess of the 
// MCalibration*Calc classes in that case.
//
Int_t MCalibCalcFromPast::Process()
{

  const UInt_t pattern = fHeader->GetCalibrationPattern();

  if (pattern == fPattern)
    return kTRUE;

  if (fPattern == 0)
    {
      fPattern = pattern;
      return kTRUE;
    }

  fPattern = pattern;
  
  //
  // Finalize Possible calibration histogram classes...
  //
  *fLog << inf << GetDescriptor() << " : Finalize calibration histograms..." << flush;
  if (Finalize("MHCalibrationChargeCam"))      *fLog << "MHCalibrationChargeCam";
  if (Finalize("MHCalibrationChargeBlindCam")) *fLog << "MHCalibrationChargeBlindCam";
  if (Finalize("MHCalibrationRelTimeCam"))     *fLog << "MHCalibrationRelTimeCam";  
  if (Finalize("MHCalibrationHiLoCam"))        *fLog << "MHCalibrationHiLoCam";  

  //
  // Finalize possible calibration calculation tasks
  //
  *fLog << endl;
  *fLog << inf << GetDescriptor() << " : Finalize calibration calculations..." << flush;
  if (fChargeCalc)
    fChargeCalc->CallPostProcess();
  if (fRelTimeCalc)
    fRelTimeCalc->CallPostProcess();

  ReInitialize();

  return kTRUE;
}


// --------------------------------------------------------------------------
//
// Searches for name in the MParList and calls, if existing: 
// - MHCalibrationCam::Finalize()
// - MHCalibrationCam::ResetHists()
//
Bool_t MCalibCalcFromPast::Finalize(const char* name)
{

  MHCalibrationCam *hist = (MHCalibrationCam*)fParList->FindObject(name);
  if (hist)
    {
      hist->Finalize();
      hist->ResetHists();
      return kTRUE;
    }

  return kFALSE;
  
}

// --------------------------------------------------------------------------
//
// Re-Intitializes new containers inside the Intensity Cams. 
// From now on, a call to the IntensityCam functions returns pointers 
// to the newly created Containers.
//
Bool_t MCalibCalcFromPast::ReInitialize()
{

  *fLog << endl;
  
  if (fIntensBad)
    {
      fIntensBad->AddToList(Form("MBadPixelsCam%s",GetNamePattern()),*fGeom);
      *fLog << inf << "New MBadPixelsCam with " << GetNamePattern() << endl;
    }
  if (fIntensCharge)
    {
      fIntensCharge->AddToList(Form("MCalibrationChargeCam%s",GetNamePattern()),*fGeom);
      *fLog << inf << "New MCalibrationChargeCam with " << GetNamePattern() << endl;
    }
  if (fIntensQE)
    {
      fIntensQE->AddToList(Form("MCalibrationQECam%s",GetNamePattern()),*fGeom);
      *fLog << inf << "New MCalibrationQECam with " << GetNamePattern() << endl;
    }
  if (fIntensBlind)
    {
      fIntensBlind->AddToList(Form("MCalibrationBlindCam%s",GetNamePattern()),*fGeom);
      *fLog << inf << "New MCalibrationBlindCam with " << GetNamePattern() << endl;
    }

  return kTRUE;

}

const char* MCalibCalcFromPast::GetNamePattern()
{

  Float_t number[MCalibrationCam::gkNumPulserColors];
  memset(number,0,MCalibrationCam::gkNumPulserColors*sizeof(Float_t));

  enum ColorCode_t 
    {
      k5LedGreen   = BIT(0 ),
      k2LedGreen   = BIT(1 ),
      k5LedBlue2   = BIT(2 ),      
      k1LedUV      = BIT(3 ),
      k2LedUV      = BIT(4 ),
      k5LedBlue3   = BIT(5 ),
      k5LedBlue4   = BIT(6 ),
      k2LedBlue    = BIT(7 ),
      k01LedBlue   = BIT(8 ),
      k1LedBlue    = BIT(10),
      k5LedUV1     = BIT(11),
      k5LedUV2     = BIT(12),
      k5LedBlue1   = BIT(13),
      k1LedGreen   = BIT(14),
      k01LedGreen  = BIT(15),
      kCT1Pulser   = BIT(16)
    };

  if (fPattern & k5LedGreen)
    number[MCalibrationCam::kGREEN] += 5;
  if (fPattern & k2LedGreen)
    number[MCalibrationCam::kGREEN] += 2;
  if (fPattern & k5LedBlue2)
    number[MCalibrationCam::kBLUE]  += 2;
  if (fPattern & k1LedUV)           
    number[MCalibrationCam::kUV]    += 1;
  if (fPattern & k2LedUV)           
    number[MCalibrationCam::kUV]    += 2;
  if (fPattern & k5LedBlue3)        
    number[MCalibrationCam::kBLUE]  += 5;
  if (fPattern & k5LedBlue4)        
    number[MCalibrationCam::kBLUE]  += 5;
  if (fPattern & k2LedBlue)         
    number[MCalibrationCam::kBLUE]  += 2;
  if (fPattern & k01LedBlue)        
    number[MCalibrationCam::kBLUE]  += 0.5;
  if (fPattern & k1LedBlue)         
    number[MCalibrationCam::kBLUE]  += 1;
  if (fPattern & k5LedUV1)          
    number[MCalibrationCam::kUV]    += 5;
  if (fPattern & k5LedUV2)        
    number[MCalibrationCam::kUV]    += 5;
  if (fPattern & k5LedBlue1)        
    number[MCalibrationCam::kBLUE]  += 5;
  if (fPattern & k1LedGreen)
    number[MCalibrationCam::kGREEN] += 1;
  if (fPattern & k01LedGreen)
    number[MCalibrationCam::kGREEN] += 0.5;
  if (fPattern & kCT1Pulser)
    number[MCalibrationCam::kCT1]   += 1;

  TString result;
  
  for (Int_t i=0; i<MCalibrationCam::gkNumPulserColors; i++)
    {
      switch (i)
        {
        case MCalibrationCam::kGREEN:
          if (number[i] > 0.1)
            result = Form("%2.1f%s",number[i],"GREEN");
          break;
        case MCalibrationCam::kBLUE:
          if (number[i] > 0.1)
            result = Form("%2.1f%s",number[i],"BLUE");
          break;
        case MCalibrationCam::kUV:
          if (number[i] > 0.1)
            result = Form("%2.1f%s",number[i],"UV");
          break;
        case MCalibrationCam::kCT1:
          if (number[i] > 0.1)
            result = Form("%2.1f%s",number[i],"CT1");
          break;
        }
    }
  return result.Data();
}
