/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug   02/2004 <mailto:markus@ifae.es>
!              
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//                                                                        
// MHCalibrationRelTimeCam                                                
//                                                                        
// Fills the extracted relative arrival times of MArrivalTimeCam into 
// the MHCalibrationPix-classes MHCalibrationPix for every:
//
// - Pixel, stored in the TObjArray's MHCalibrationCam::fHiGainArray  
//   or MHCalibrationCam::fHiGainArray, respectively, depending if 
//   MArrivalTimePix::IsLoGainUsed() is set.
//
// - Average pixel per AREA index (e.g. inner and outer for the MAGIC camera), 
//   stored in the TObjArray's MHCalibrationCam::fAverageHiGainAreas and 
//   MHCalibrationCam::fAverageHiGainAreas
//
// - Average pixel per camera SECTOR (e.g. sectors 1-6 for the MAGIC camera), 
//   stored in the TObjArray's MHCalibrationCam::fAverageHiGainSectors 
//   and MHCalibrationCam::fAverageHiGainSectors 
//
// Every relative time is calculated as the difference between the individual 
// pixel arrival time and the one of pixel 1 (hardware number: 2). 
// The relative times are filled into a histogram and an array, in order to perform 
// a Fourier analysis (see MHGausEvents). The signals are moreover averaged on an 
// event-by-event basis and written into the corresponding average pixels.
//
// The histograms are fitted to a Gaussian, mean and sigma with its errors 
// and the fit probability are extracted. If none of these values are NaN's and 
// if the probability is bigger than MHGausEvents::fProbLimit (default: 0.5%), 
// the fit is declared valid.
// Otherwise, the fit is repeated within ranges of the previous mean 
// +- MHCalibrationPix::fPickupLimit (default: 5) sigma (see MHCalibrationPix::RepeatFit())
// In case this does not make the fit valid, the histogram means and RMS's are 
// taken directly (see MHCalibrationPix::BypassFit()) and the following flags are set:
// - MBadPixelsPix::SetUncalibrated( MBadPixelsPix::kRelTimeNotFitted ) and
// - MBadPixelsPix::SetUnsuitable(   MBadPixelsPix::kUnreliableRun    ) 
// 
// Outliers of more than MHCalibrationPix::fPickupLimit (default: 5) sigmas 
// from the mean are counted as Pickup events (stored in MHCalibrationPix::fPickup) 
//
// The class also fills arrays with the signal vs. event number, creates a fourier 
// spectrum (see MHGausEvents::CreateFourierSpectrum()) and investigates if the 
// projected fourier components follow an exponential distribution. 
// In case that the probability of the exponential fit is less than 
// MHGausEvents::fProbLimit (default: 0.5%), the following flags are set:
// - MBadPixelsPix::SetUncalibrated( MBadPixelsPix::kRelTimeOscillating ) and
// - MBadPixelsPix::SetUnsuitable(   MBadPixelsPix::kUnreliableRun      )
// 
// This same procedure is performed for the average pixels.
//
// The following results are written into MCalibrationRelTimeCam:
//
// - MCalibrationPix::SetMean()
// - MCalibrationPix::SetMeanErr()
// - MCalibrationPix::SetSigma()
// - MCalibrationPix::SetSigmaErr()
// - MCalibrationPix::SetProb()
// - MCalibrationPix::SetNumPickup()
//
// For all averaged areas, the fitted sigma is multiplied with the square root of 
// the number involved pixels in order to be able to compare it to the average of 
// sigmas in the camera.
//                                                                         
/////////////////////////////////////////////////////////////////////////////
#include "MHCalibrationRelTimeCam.h"
#include "MHCalibrationPix.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MCalibrationIntensityRelTimeCam.h"

#include "MCalibrationRelTimeCam.h"
#include "MCalibrationRelTimePix.h"
#include "MCalibrationPix.h"

#include "MArrivalTimeCam.h"
#include "MArrivalTimePix.h"

#include "MGeomCam.h"
#include "MGeomPix.h"

#include "MBadPixelsIntensityCam.h"
#include "MBadPixelsCam.h"
#include "MBadPixelsPix.h"

#include <TOrdCollection.h>
#include <TPad.h>
#include <TVirtualPad.h>
#include <TCanvas.h>
#include <TStyle.h>
#include <TF1.h>
#include <TLine.h>
#include <TLatex.h>
#include <TLegend.h>
#include <TGraph.h>

ClassImp(MHCalibrationRelTimeCam);

using namespace std;

const Float_t MHCalibrationRelTimeCam::fgNumHiGainSaturationLimit = 0.25;
const UInt_t  MHCalibrationRelTimeCam::fgReferencePixel = 1;
const Int_t   MHCalibrationRelTimeCam::fgNbins      = 200;
const Axis_t  MHCalibrationRelTimeCam::fgFirst      = -5.;
const Axis_t  MHCalibrationRelTimeCam::fgLast       =  5.;
const Float_t MHCalibrationRelTimeCam::fgProbLimit  = 0.;
const Float_t MHCalibrationRelTimeCam::fgNumOverflowLimit = 0.005;
const TString MHCalibrationRelTimeCam::gsHistName   = "RelTime";
const TString MHCalibrationRelTimeCam::gsHistTitle  = "Rel. Arr. Times";
const TString MHCalibrationRelTimeCam::gsHistXTitle = "Rel. Arr. Time [FADC slices]";
const TString MHCalibrationRelTimeCam::gsHistYTitle = "Nr. events";
// --------------------------------------------------------------------------
//
// Default Constructor.
//
// Sets: 
// - fReferencePixel to fgReferencePixel
// - fNbins to fgNbins
// - fFirst to fgFirst
// - fLast  to fgLast 
//
// - fHistName   to gsHistName  
// - fHistTitle  to gsHistTitle 
// - fHistXTitle to gsHistXTitle
// - fHistYTitle to gsHistYTitle
//
MHCalibrationRelTimeCam::MHCalibrationRelTimeCam(const char *name, const char *title) 
{

  fName  = name  ? name  : "MHCalibrationRelTimeCam";
  fTitle = title ? title : "Histogram class for the relative time calibration of the camera";

  SetNumHiGainSaturationLimit(fgNumHiGainSaturationLimit);

  SetReferencePixel();

  SetNbins(fgNbins);
  SetFirst(fgFirst);
  SetLast (fgLast );

  SetNumOverflowLimit(fgNumOverflowLimit);
  SetProbLimit(fgProbLimit);

  SetHistName  (gsHistName  .Data());
  SetHistTitle (gsHistTitle .Data());
  SetHistXTitle(gsHistXTitle.Data());
  SetHistYTitle(gsHistYTitle.Data());

}

// --------------------------------------------------------------------------
//
// Creates new MHCalibrationRelTimeCam only with the averaged areas:
// the rest has to be retrieved directly, e.g. via: 
//     MHCalibrationRelTimeCam *cam = MParList::FindObject("MHCalibrationRelTimeCam");
//  -  cam->GetAverageSector(5).DrawClone();
//  -  (*cam)[100].DrawClone()
//
TObject *MHCalibrationRelTimeCam::Clone(const char *) const
{

  MHCalibrationRelTimeCam *cam = new MHCalibrationRelTimeCam();

  //
  // Copy the data members
  //
  cam->fColor                  = fColor;
  cam->fRunNumbers             = fRunNumbers;
  cam->fPulserFrequency        = fPulserFrequency;
  cam->fFlags                  = fFlags;
  cam->fNbins                  = fNbins;
  cam->fFirst                  = fFirst;
  cam->fLast                   = fLast;

  //
  // Copy the MArrays
  //
  cam->fAverageAreaRelSigma    = fAverageAreaRelSigma;
  cam->fAverageAreaRelSigmaVar = fAverageAreaRelSigmaVar;
  cam->fAverageAreaSat         = fAverageAreaSat;
  cam->fAverageAreaSigma       = fAverageAreaSigma;
  cam->fAverageAreaSigmaVar    = fAverageAreaSigmaVar;
  cam->fAverageAreaNum         = fAverageAreaNum;
  cam->fAverageSectorNum       = fAverageSectorNum;

  if (!IsAverageing())
    return cam;

  const Int_t navhi   =  fAverageHiGainAreas->GetSize();

  for (int i=0; i<navhi; i++)
    cam->fAverageHiGainAreas->AddAt(GetAverageHiGainArea(i).Clone(),i);

  if (IsLoGain())
    {
      
      const Int_t navlo = fAverageLoGainAreas->GetSize();
      for (int i=0; i<navlo; i++)
        cam->fAverageLoGainAreas->AddAt(GetAverageLoGainArea(i).Clone(),i);

    }

  return cam;
}

// --------------------------------------------------------------------------
//
// Gets or creates the pointers to:
// - MCalibrationRelTimeCam
//
// Searches pointer to:
// - MArrivalTimeCam
//
// Calls:
// - MHCalibrationCam::InitHiGainArrays()
// - MHCalibrationCam::InitLoGainArrays()
// 
// Sets:
// - fSumareahi   to nareas
// - fSumarealo   to nareas
// - fSumsectorhi to nareas
// - fSumsectorlo to nareas
// - fNumareahi   to nareas
// - fNumarealo   to nareas
// - fNumsectorhi to nareas
// - fNumsectorlo to nareas
//
Bool_t MHCalibrationRelTimeCam::ReInitHists(MParList *pList)
{

  fIntensCam = (MCalibrationIntensityCam*)pList->FindObject(AddSerialNumber("MCalibrationIntensityRelTimeCam"));
  if (fIntensCam)
    *fLog << inf << "Found MCalibrationIntensityRelTimeCam ... " << endl;
  else
    {
      fCam = (MCalibrationCam*)pList->FindObject(AddSerialNumber("MCalibrationRelTimeCam"));
      if (!fCam)
        {
          fCam = (MCalibrationCam*)pList->FindCreateObj(AddSerialNumber("MCalibrationRelTimeCam"));
          if (!fCam)
            {
              *fLog << err << "Cannot find nor create MCalibrationRelTimeCam ... abort." << endl;
              return kFALSE;
            }
          fCam->Init(*fGeom);
        }
    }

  MArrivalTimeCam *signal = (MArrivalTimeCam*)pList->FindObject("MArrivalTimeCam");
  if (!signal)
  {
      *fLog << err << "MArrivalTimeCam not found... abort." << endl;
      return kFALSE;
  }

  const Int_t npixels  = fGeom->GetNumPixels();
  const Int_t nsectors = fGeom->GetNumSectors();
  const Int_t nareas   = fGeom->GetNumAreas();

  InitHiGainArrays(npixels,nareas,nsectors);
  InitLoGainArrays(npixels,nareas,nsectors);

  fSumareahi  .Set(nareas); 
  fSumarealo  .Set(nareas);
  fSumsectorhi.Set(nsectors); 
  fSumsectorlo.Set(nsectors);
  fNumareahi  .Set(nareas); 
  fNumarealo  .Set(nareas);
  fNumsectorhi.Set(nsectors); 
  fNumsectorlo.Set(nsectors);

  return kTRUE;
}


// -------------------------------------------------------------------------------
//
// Retrieves pointer to MArrivalTimeCam:
//
// Retrieves from MGeomCam:
// - number of pixels
// - number of pixel areas
// - number of sectors
//
// Fills HiGain or LoGain histograms (MHGausEvents::FillHistAndArray()), respectively
// depending on MArrivalTimePix::IsLoGainUsed(), with:
// - MArrivalTimePix::GetArrivalTime(pixid) - MArrivalTimePix::GetArrivalTime(1);
//   (i.e. the time difference between pixel i and pixel 1 (hardware number: 2) )
//
Bool_t MHCalibrationRelTimeCam::FillHists(const MParContainer *par, const Stat_t w)
{

  MArrivalTimeCam *arrtime = (MArrivalTimeCam*)par;
  if (!arrtime)
    {
      gLog << err << "No argument in MArrivalTime::Fill... abort." << endl;
      return kFALSE;
    }
  
  const Int_t npixels  = fGeom->GetNumPixels();
  const Int_t nareas   = fGeom->GetNumAreas();
  const Int_t nsectors = fGeom->GetNumSectors();

  fSumareahi  .Reset();
  fSumarealo  .Reset();
  fSumsectorhi.Reset(); 
  fSumsectorlo.Reset();
  fNumareahi  .Reset(); 
  fNumarealo  .Reset();
  fNumsectorhi.Reset(); 
  fNumsectorlo.Reset();

  const MArrivalTimePix &refpix = (*arrtime)[fReferencePixel];
  const Float_t reftime = refpix.IsLoGainUsed() 
     ? refpix.GetArrivalTimeLoGain() : refpix.GetArrivalTimeHiGain();

  for (Int_t i=0; i<npixels; i++)
    {

      MHCalibrationPix &histhi = (*this)[i];

      if (histhi.IsExcluded())
	continue;

      const MArrivalTimePix &pix = (*arrtime)[i];
      const Int_t aidx   = (*fGeom)[i].GetAidx();
      const Int_t sector = (*fGeom)[i].GetSector();

      if (pix.IsLoGainUsed() && IsLoGain())
        { 
          const Float_t reltime = pix.GetArrivalTimeLoGain() - reftime;
          histhi.AddSaturated(1); 

          MHCalibrationPix &histlo = (*this)(i);
          histlo.FillHistAndArray(reltime);

          fSumarealo  [aidx]   += reltime;
          fNumarealo  [aidx]   ++;
          fSumsectorlo[sector] += reltime;
          fNumsectorlo[sector] ++;
        }
      else
        {
          const Float_t reltime = pix.GetArrivalTimeHiGain() - reftime;

          histhi.FillHistAndArray(reltime) ;
          fSumareahi  [aidx]   += reltime;
          fNumareahi  [aidx]   ++;
          fSumsectorhi[sector] += reltime;
          fNumsectorhi[sector] ++;
        }
    }
  
  for (Int_t j=0; j<nareas; j++)
    {
      MHCalibrationPix &histhi = GetAverageHiGainArea(j);
      histhi.FillHistAndArray(fNumareahi[j] == 0 ? 0. : fSumareahi[j]/fNumareahi[j]);

      if (IsLoGain())
        {
          MHCalibrationPix &histlo = GetAverageLoGainArea(j);
          histlo.FillHistAndArray(fNumarealo[j] == 0 ? 0. : fSumarealo[j]/fNumarealo[j]);
        }
    }
  
  for (Int_t j=0; j<nsectors; j++)
    {
      MHCalibrationPix &histhi = GetAverageHiGainSector(j);
      histhi.FillHistAndArray(fNumsectorhi[j] == 0 ? 0. : fSumsectorhi[j]/fNumsectorhi[j]);

      if (IsLoGain())
        {
          MHCalibrationPix &histlo = GetAverageLoGainSector(j);
          histlo.FillHistAndArray(fNumsectorlo[j] == 0 ? 0. : fSumsectorlo[j]/fNumsectorlo[j]);
        }
    }

  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calls:
// - MHCalibrationCam::FitHiGainArrays() with flags:
//   MBadPixelsPix::kRelTimeNotFitted and MBadPixelsPix::kRelTimeOscillating
// - MHCalibrationCam::FitLoGainArrays() with flags:
//   MBadPixelsPix::kRelTimeNotFitted and MBadPixelsPix::kRelTimeOscillating
// 
Bool_t MHCalibrationRelTimeCam::FinalizeHists()
{

  *fLog << endl;

  MCalibrationCam *relcam = fIntensCam ? fIntensCam->GetCam() : fCam;
  MBadPixelsCam   *badcam = fIntensBad ? fIntensBad->GetCam() : fBadPixels;

  const Int_t nareas = fAverageHiGainAreas->GetSize();
  const Int_t nsectors = fAverageHiGainSectors->GetSize();

  TArrayI satarea(nareas);
  TArrayI satsect(nsectors);
  fNumareahi  .Reset(); 
  fNumsectorhi.Reset(); 

  for (Int_t i=0; i<fHiGainArray->GetSize(); i++)
    {
      
      MHCalibrationPix &histhi = (*this)[i];

      if (histhi.IsExcluded())
        continue;
      
      const Int_t aidx   = (*fGeom)[i].GetAidx();
      const Int_t sector = (*fGeom)[i].GetSector();

      MCalibrationRelTimePix &pix = (MCalibrationRelTimePix&)(*relcam)[i] ;

      fNumareahi[aidx]++;
      fNumsectorhi[sector]++;
      //
      // Check saturation
      // 
      if (histhi.GetSaturated() > fNumHiGainSaturationLimit*histhi.GetHGausHist()->GetEntries())
        {
          pix.SetHiGainSaturation();
          histhi.SetExcluded();
          satarea[aidx]++;
          satsect[sector]++;
        }
      else
        if (IsLoGain())
          (*this)(i).SetExcluded();

      //
      // Check histogram overflow
      // 
      CheckOverflow(histhi);
      if (IsLoGain())
        CheckOverflow((*this)(i));

    }

  for (Int_t j=0; j<nareas; j++)
    {
      
      MHCalibrationPix     &histhi = GetAverageHiGainArea(j);      
      MCalibrationRelTimePix &pix = (MCalibrationRelTimePix&)relcam->GetAverageArea(j);

      if (satarea[j] > 0.5*fNumareahi[j])
        {
          pix.SetHiGainSaturation();
          histhi.SetExcluded();
        }
      else
        if (IsLoGain())
          GetAverageLoGainArea(j).SetExcluded();

      //
      // Check histogram overflow
      // 
      CheckOverflow(histhi);
      if (IsLoGain())
        CheckOverflow(GetAverageLoGainArea(j));
   }
  
  for (Int_t j=0; j<fAverageHiGainSectors->GetSize(); j++)
    {
      
      MHCalibrationPix &histhi = GetAverageHiGainSector(j);      
      MCalibrationRelTimePix  &pix = (MCalibrationRelTimePix&)relcam->GetAverageSector(j) ;

      if (satsect[j] > 0.5*fNumsectorhi[j])
        {
          pix.SetHiGainSaturation();
          histhi.SetExcluded();
        }
      else
        if (IsLoGain())        
          GetAverageLoGainSector(j).SetExcluded();

      //
      // Check histogram overflow
      // 
      CheckOverflow(histhi);
      if (IsLoGain())
        CheckOverflow(GetAverageLoGainSector(j));
    }

  FitHiGainArrays(*relcam,*badcam,
                  MBadPixelsPix::kRelTimeNotFitted,
                  MBadPixelsPix::kRelTimeOscillating);
  
  if (IsLoGain())        
    FitLoGainArrays(*relcam,*badcam,
                    MBadPixelsPix::kRelTimeNotFitted,
                    MBadPixelsPix::kRelTimeOscillating);

  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Sets all pixels to MBadPixelsPix::kUnreliableRun, if following flags are set:
// - MBadPixelsPix::kRelTimeNotFitted
// - MBadPixelsPix::kRelTimeOscillating
//
void MHCalibrationRelTimeCam::FinalizeBadPixels()
{

  MBadPixelsCam *badcam = fIntensBad ? fIntensBad->GetCam() : fBadPixels;

  for (Int_t i=0; i<badcam->GetSize(); i++)
    {
      MBadPixelsPix          &bad    = (*badcam)[i];

      if (bad.IsUncalibrated( MBadPixelsPix::kRelTimeNotFitted ))
        bad.SetUnsuitable(   MBadPixelsPix::kUnreliableRun    );

      if (bad.IsUncalibrated( MBadPixelsPix::kRelTimeOscillating))
        bad.SetUnsuitable(   MBadPixelsPix::kUnreliableRun    );
      
    }
}

// --------------------------------------------------------------------------
//
// The types are as follows:
// 
// Fitted values:
// ============== 
//
// 0: Fitted Mean Relative Arrival Time in FADC slices  (MHGausEvents::GetMean()
// 1: Error Mean Relative Arrival Time in FADC slices   (MHGausEvents::GetMeanErr()
// 2: Sigma fitted Relative Arrival Time in FADC slices (MHGausEvents::GetSigma()
// 3: Error Sigma Relative Arrival Time in FADC slices  (MHGausEvents::GetSigmaErr()
//
// Useful variables derived from the fit results:
// =============================================
//
// 4: Returned probability of Gauss fit              (calls: MHGausEvents::GetProb())
//
// Localized defects:
// ==================
//
// 5: Gaus fit not OK                               (calls: MHGausEvents::IsGausFitOK())
// 6: Fourier spectrum not OK                       (calls: MHGausEvents::IsFourierSpectrumOK())
//
Bool_t MHCalibrationRelTimeCam::GetPixelContent(Double_t &val, Int_t idx, const MGeomCam &cam, Int_t type) const
{

  if (fHiGainArray->GetSize() <= idx)
    return kFALSE;

  const MHCalibrationPix &pix = (*this)[idx];

  switch (type)
    {
    case 0:
      val = pix.GetMean();
      break;
    case 1:
      val = pix.GetMeanErr();
      break;
    case 2:
      val = pix.GetSigma();
      break;
    case 3:
      val = pix.GetSigmaErr();
      break;
    case 4:
      val = pix.GetProb();
      break;
    case 5:
      if (!pix.IsGausFitOK())
        val = 1.;
      break;
    case 6:
      if (!pix.IsFourierSpectrumOK())
        val = 1.;
      break;
    default:
      return kFALSE;
    }
  return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calls MHCalibrationPix::DrawClone() for pixel idx
//
void MHCalibrationRelTimeCam::DrawPixelContent(Int_t idx) const
{
 (*this)[idx].DrawClone();
}

// -----------------------------------------------------------------------------
// 
// Default draw:
//
// Displays the averaged areas, both High Gain and Low Gain 
//
// Calls the Draw of the fAverageHiGainAreas and fAverageLoGainAreas objects with options
//
void MHCalibrationRelTimeCam::Draw(const Option_t *opt)
{

  const Int_t nareas = fAverageHiGainAreas->GetSize();
  if (nareas == 0)
    return;

  TString option(opt);
  option.ToLower();

  if (!option.Contains("datacheck"))
    {
      MHCalibrationCam::Draw(opt);
      return;
    }

  // 
  // From here on , the datacheck - Draw
  //
  TVirtualPad *pad = gPad ? gPad : MH::MakeDefCanvas(this);  
  pad->SetBorderMode(0);
  pad->Divide(1,nareas);

  //
  // Loop over inner and outer pixels
  //  
  for (Int_t i=0; i<nareas;i++) 
     {
       
       pad->cd(i+1);
       
       MHCalibrationPix &hipix = GetAverageHiGainArea(i);
       //
       // Ask for Hi-Gain saturation
       //
       if (hipix.IsExcluded() && IsLoGain())
        {
           MHCalibrationPix &lopix = GetAverageLoGainArea(i);
           DrawDataCheckPixel(lopix,0.);
         }
       else
         DrawDataCheckPixel(hipix,0.);
    }      
}

void MHCalibrationRelTimeCam::CheckOverflow( MHCalibrationPix &pix )
{

  if (pix.IsExcluded())
    return;

  TH1F *hist = pix.GetHGausHist();
  
  Stat_t overflow = hist->GetBinContent(hist->GetNbinsX()+1);
  if (overflow > fNumOverflowLimit*hist->GetEntries())
    {
      *fLog << warn << "HiGain Hist-overflow " << overflow
            << " times in " << pix.GetName() << " (w/o saturation!) " << endl;
    }
  
  overflow = hist->GetBinContent(0);
  if (overflow > fNumOverflowLimit*hist->GetEntries())
    {
      *fLog << warn << "HiGain Hist-underflow " << overflow
            << " times in " << pix.GetName() << " (w/o saturation!) " << endl;
    }
}


// -----------------------------------------------------------------------------
// 
// Draw the average pixel for the datacheck:
//
// Displays the averaged areas, both High Gain and Low Gain 
//
// Calls the Draw of the fAverageHiGainAreas and fAverageLoGainAreas objects with options
//
void MHCalibrationRelTimeCam::DrawDataCheckPixel(MHCalibrationPix &pix, const Float_t refline)
{
  
  TVirtualPad *newpad = gPad;
  newpad->Divide(1,2);
  newpad->cd(1);
  
  gPad->SetTicks();
  if (!pix.IsEmpty() && !pix.IsOnlyOverflow() && !pix.IsOnlyUnderflow())
    gPad->SetLogy();

  TH1F *hist = pix.GetHGausHist();
  
  TH1F *null = new TH1F("Null",hist->GetTitle(),100,pix.GetFirst() < -3.0 ? -3.0 : pix.GetFirst(),
                        pix.GetLast() > 3.0 ? 3.0 : pix.GetLast());

  null->SetMaximum(1.1*hist->GetMaximum());
  null->SetDirectory(NULL);
  null->SetBit(kCanDelete);
  null->SetStats(kFALSE);
  //
  // set the labels bigger
  //
  TAxis *xaxe = null->GetXaxis();
  TAxis *yaxe = null->GetYaxis();
  xaxe->CenterTitle();
  yaxe->CenterTitle();    
  xaxe->SetTitleSize(0.07);
  yaxe->SetTitleSize(0.07);    
  xaxe->SetTitleOffset(0.7);
  yaxe->SetTitleOffset(0.55);    
  xaxe->SetLabelSize(0.06);
  yaxe->SetLabelSize(0.06);    

  xaxe->SetTitle(hist->GetXaxis()->GetTitle());
  yaxe->SetTitle(hist->GetYaxis()->GetTitle());  

  null->Draw();
  hist->Draw("same");

  gStyle->SetOptFit();

  TF1 *fit = pix.GetFGausFit();

  if (fit)
  {
    switch ( fColor )
      {
      case MCalibrationCam::kGREEN:
        fit->SetLineColor(kGreen);
        break;
      case MCalibrationCam::kBLUE:
        fit->SetLineColor(kBlue);
        break;
      case MCalibrationCam::kUV:  
        fit->SetLineColor(106);
        break;
      case MCalibrationCam::kCT1: 
        fit->SetLineColor(006);
        break;
      default:
        fit->SetLineColor(kRed);
      }
    fit->Draw("same");
  }

  //  DisplayRefLines(null,refline);

  gPad->Modified();
  gPad->Update();

  newpad->cd(2);
  gPad->SetTicks();

  TH1F *null2 = new TH1F("Null2",hist->GetTitle(),100,0.,pix.GetEvents()->GetSize()/pix.GetEventFrequency());

  null2->SetMinimum(pix.GetMean()-10.*pix.GetSigma());
  null2->SetMaximum(pix.GetMean()+10.*pix.GetSigma());
  null2->SetDirectory(NULL);
  null2->SetBit(kCanDelete);
  null2->SetStats(kFALSE);
  //
  // set the labels bigger
  //
  TAxis *xaxe2 = null2->GetXaxis();
  TAxis *yaxe2 = null2->GetYaxis();
  xaxe2->CenterTitle();
  yaxe2->CenterTitle();    
  xaxe2->SetTitleSize(0.07);
  yaxe2->SetTitleSize(0.07);    
  xaxe2->SetTitleOffset(0.7);
  yaxe2->SetTitleOffset(0.55);    
  xaxe2->SetLabelSize(0.06);
  yaxe2->SetLabelSize(0.06);    

  pix.CreateGraphEvents();
  TGraph *gr = pix.GetGraphEvents();

  xaxe2->SetTitle(gr->GetXaxis()->GetTitle());
  yaxe2->SetTitle(gr->GetYaxis()->GetTitle());  

  null2->Draw();

  pix.DrawEvents("same");

  return;
}

