/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Markus Gaug, 4/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJPedestal
//
/////////////////////////////////////////////////////////////////////////////
#include "MJPedestal.h"

// root classes
#include <TF1.h>
#include <TEnv.h>
#include <TFile.h>
#include <TLine.h>
#include <TLatex.h>
#include <TString.h>
#include <TCanvas.h>
#include <TSystem.h>
#include <TLegend.h>
#include <TPad.h>

// mars core
#include "MLog.h"
#include "MLogManip.h"

#include "MTaskEnv.h"
#include "MSequence.h"
#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MStatusDisplay.h"

// Other basic classes
#include "MExtractor.h"

// parameter containers
#include "MGeomCam.h"
#include "MHCamera.h"
#include "MPedestalCam.h"
#include "MPedestalPix.h"
#include "MBadPixelsCam.h"

#include "MCalibrationPedCam.h"
#include "MCalibrationPix.h"
#include "MHPedestalCam.h"
#include "MHPedestalPix.h"

// tasks
#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MGeomApply.h"
#include "MBadPixelsMerge.h"
#include "MFillH.h"
#include "MPedCalcPedRun.h"
#include "MPedCalcFromLoGain.h"

ClassImp(MJPedestal);

using namespace std;

const Double_t MJPedestal::fgPedestalMin = 4.;
const Double_t MJPedestal::fgPedestalMax = 16.;
const Double_t MJPedestal::fgPedRmsMin   = 0.;
const Double_t MJPedestal::fgPedRmsMax   = 20.;

const Float_t  MJPedestal::fgRefPedClosedLids       = 9.635;
const Float_t  MJPedestal::fgRefPedExtraGalactic    = 9.93;
const Float_t  MJPedestal::fgRefPedGalactic         = 10.03;
const Float_t  MJPedestal::fgRefPedRmsClosedLidsInner    = 1.7;
const Float_t  MJPedestal::fgRefPedRmsExtraGalacticInner = 5.6;
const Float_t  MJPedestal::fgRefPedRmsGalacticInner      = 6.92;
const Float_t  MJPedestal::fgRefPedRmsClosedLidsOuter    = 1.7;
const Float_t  MJPedestal::fgRefPedRmsExtraGalacticOuter = 3.35;
const Float_t  MJPedestal::fgRefPedRmsGalacticOuter      = 4.2;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets:
// - fRuns to 0, 
// - fExtractor to NULL, 
// - fDataCheck to kFALSE, 
// - fExtractType to kUsePedRun
// - fStorage to Normal Storage
// - fExtractorResolution to kFALSE
//
MJPedestal::MJPedestal(const char *name, const char *title) 
    : fRuns(0), fExtractor(NULL), fDisplayType(kNormalDisplay),
      fDataCheck(kFALSE), fExtractType(kUsePedRun), fExtractionType(kFundamental)
{
    fName  = name  ? name  : "MJPedestal";
    fTitle = title ? title : "Tool to create a pedestal file (MPedestalCam)";

    SetNormalStorage();
    SetUsePedRun();
    SetPathIn("");
}

MJPedestal::~MJPedestal()
{
    if (fExtractor)
        delete fExtractor;
}

const char* MJPedestal::GetOutputFile() const
{
    if (fSequence.IsValid())
        return Form("%s/pedest%08d.root", (const char*)fPathOut, fSequence.GetSequence());

    if (!fRuns)
        return "";

    return Form("%s/%s-F0.root", (const char*)fPathOut, (const char*)fRuns->GetRunsAsFileName());
}

//---------------------------------------------------------------------------------
//
// Try to read an existing MPedestalCam from a previously created output file. 
// If found, also an MBadPixelsCam and the corresponding display is read.
//
// In case of Storage type "kNoStorage" or if the file is not found or the 
// MPedestalCam cannot be read, return kFALSE, otherwise kTRUE.
//
Bool_t MJPedestal::ReadPedestalCam()
{
    const TString fname = GetOutputFile();

    *fLog << inf << "Reading pedestals from file: " << fname << endl;

    TFile file(fname, "READ");
    if (fPedestalCamIn.Read()<=0)
    {
        *fLog << err << "Unable to read incoming MPedestalCam from " << fname << endl;
        return kFALSE;
    }

    if (fPedestalCamOut.Read()<=0)
    {
        *fLog << err << "Unable to read outgoing MPedestalCam from " << fname << endl;
        return kFALSE;
    }

    if (file.FindKey("MBadPixelsCam"))
    {
        MBadPixelsCam bad;
        if (bad.Read()<=0)
        {
            *fLog << err << "Unable to read MBadPixelsCam from " << fname << endl;
            return kFALSE;
        }
        fBadPixels.Merge(bad);
    }

    if (fDisplay && !fDisplay->GetCanvas("Pedestals"))
        fDisplay->Read();

    return kTRUE;
}

MExtractor *MJPedestal::ReadCalibration() const
{
    const TString fname = Form("%s/calib%08d.root",fPathIn.Data(), fSequence.GetSequence());

    *fLog << inf << "Reading extractor from file: " << fname << endl;

    TFile file(fname, "READ");
    if (!file.IsOpen())
    {
        *fLog << err << dbginf << "ERROR - Could not open file " << fname << endl;
        return NULL;
    }

    TObject *o = file.Get("ExtractSignal");
    if (o && !o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractSignal read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return NULL;
    }

    return o ? (MExtractor*)o->Clone("ExtractSignal") : NULL;
}

//---------------------------------------------------------------------------------
//
// Display the results. 
// If Display type "kDataCheck" was chosen, also the reference lines are displayed.
//
void MJPedestal::DisplayResult(MParList &plist)
{
    if (!fDisplay)
        return;

    //
    // Update display
    //
    TString title = fDisplay->GetTitle();
    title += "--  Pedestal ";
    if (fSequence.IsValid())
        title += fSequence.GetName();
    else
        if (fRuns)  // FIXME: What to do if an environmentfile was used?
            title += fRuns->GetRunsAsString();
    title += "  --";
    fDisplay->SetTitle(title);

    //
    // Get container from list
    //
    MGeomCam           &geomcam   = *(MGeomCam*)plist.FindObject("MGeomCam");
    MCalibrationPedCam &calpedcam = *(MCalibrationPedCam*)plist.FindObject("MCalibrationPedCam");

    //
    // Create container to display
    //
    MHCamera disp0 (geomcam, "MPedestalCam;ped", "Mean Pedestal");
    MHCamera disp1 (geomcam, "MPedestalCam;RMS", "Pedestal RMS");
    MHCamera disp2 (geomcam, "MCalibPedCam;histmean", "Mean Pedestal (Hist.)");
    MHCamera disp3 (geomcam, "MCalibPedCam;histsigma", "Pedestal RMS (Hist.)");
    MHCamera disp4 (geomcam, "MCalibPedCam;ped", "Mean Pedestal");
    MHCamera disp5 (geomcam, "MCalibPedCam;RMS", "Pedestal RMS");
    MHCamera disp6 (geomcam, "MCalibDiffCam;ped", "Diff. Mean Pedestal (Hist.)");
    MHCamera disp7 (geomcam, "MCalibDiffCam;RMS", "Diff. Pedestal RMS (Hist.)");
    MHCamera disp8 (geomcam, "MCalibDiffCam;ped", "Diff. Mean Pedestal");
    MHCamera disp9 (geomcam, "MCalibDiffCam;AbsRMS", "Diff. Abs. Pedestal RMS");
    MHCamera disp10(geomcam, "MCalibDiffCam;RelRMS", "Diff. Rel. Pedestal RMS");

    disp0.SetCamContent(fPedestalCamOut, 0);
    disp0.SetCamError  (fPedestalCamOut, 1);

    disp1.SetCamContent(fPedestalCamOut, 2);
    disp1.SetCamError  (fPedestalCamOut, 3);

    if (fExtractType == kUseHists)
      {
        disp2.SetCamContent(calpedcam, 0);
        disp2.SetCamError  (calpedcam, 1);

        disp3.SetCamContent(calpedcam, 2);
        disp3.SetCamError  (calpedcam, 3);

        disp4.SetCamContent(calpedcam, 5);
        disp4.SetCamError  (calpedcam, 6);

        disp5.SetCamContent(calpedcam, 7);
        disp5.SetCamError  (calpedcam, 8);

        for (UInt_t i=0;i<geomcam.GetNumPixels();i++)
          {

            MPedestalPix    &ped  = fPedestalCamOut[i];
            MCalibrationPix &hist = calpedcam [i];
            MBadPixelsPix &bad  = fBadPixels[i];

            if (bad.IsUnsuitable())
              continue;

            disp6.Fill(i,ped.GetPedestal()-hist.GetHiGainMean());
            disp6.SetUsed(i);

            disp7.Fill(i,hist.GetHiGainSigma()-ped.GetPedestalRms());
            if (TMath::Abs(ped.GetPedestalRms()-hist.GetHiGainSigma()) < 4.0)
              disp7.SetUsed(i);

            disp8.Fill(i,ped.GetPedestal()-hist.GetLoGainMean());
            disp8.SetUsed(i);

            disp9.Fill(i,hist.GetLoGainSigma()-ped.GetPedestalRms());
            if (TMath::Abs(hist.GetLoGainSigma() - ped.GetPedestalRms()) < 4.0)
              disp9.SetUsed(i);
          }
      }
    
    if (fExtractionType!=kFundamental/*fExtractorResolution*/)
      {
        for (UInt_t i=0;i<geomcam.GetNumPixels();i++)
          {

            MPedestalPix &pedo  = fPedestalCamOut[i];
            MPedestalPix &pedi  = fPedestalCamIn[i];
            MBadPixelsPix &bad  = fBadPixels[i];

            if (bad.IsUnsuitable())
              continue;

            const Float_t diff = pedo.GetPedestalRms()-pedi.GetPedestalRms();
            const Float_t sum  = 0.5*(pedo.GetPedestalRms()+pedi.GetPedestalRms());

            disp9.Fill(i,pedo.GetPedestalRms()-pedi.GetPedestalRms());
            if (pedo.IsValid() && pedi.IsValid())
              disp9.SetUsed(i);

            disp10.Fill(i,sum == 0. ? 0. : diff/sum);
            if (pedo.IsValid() && pedi.IsValid() && sum != 0.)
              disp10.SetUsed(i);
          }
      }

    disp0.SetYTitle("P [cts/slice]");
    disp1.SetYTitle("P_{rms} [cts/slice]");
    disp2.SetYTitle("Hist. Mean  [cts/slice]");
    disp3.SetYTitle("Hist. Sigma [cts/slice]");
    disp4.SetYTitle("Calc. Mean  [cts/slice]");
    disp5.SetYTitle("Calc. RMS   [cts/slice]");
    disp6.SetYTitle("Diff. Mean  [cts/slice]");
    disp7.SetYTitle("Diff. RMS   [cts/slice]");
    disp8.SetYTitle("Diff. Mean  [cts/slice]");
    disp9.SetYTitle("Abs.Diff.RMS [cts/slice]");
    disp10.SetYTitle("Rel.Diff.RMS [1]");

    //
    // Display data
    //
    if (fDisplayType != kDataCheckDisplay && fExtractType != kUseHists && fExtractionType==kFundamental/*fExtractorResolution*/)
      {
        TCanvas &c3 = fDisplay->AddTab("Pedestals");
        c3.Divide(2,3);
        
        disp0.CamDraw(c3, 1, 2, 1);
        disp1.CamDraw(c3, 2, 2, 6);
        return;
      }
    
    if (fExtractType == kUseHists)
      {
        
        TCanvas &c3 = fDisplay->AddTab("Extractor Hist.");
        c3.Divide(2,3);

        disp2.CamDraw(c3, 1, 2, 1);
        disp3.CamDraw(c3, 2, 2, 5);

        TCanvas &c4 = fDisplay->AddTab("Extractor Calc.");
        c4.Divide(2,3);
        
        disp4.CamDraw(c4, 1, 2, 1);
        disp5.CamDraw(c4, 2, 2, 5);


        TCanvas &c5 = fDisplay->AddTab("Difference Hist.");
        c5.Divide(2,3);
        
        disp6.CamDraw(c5, 1, 2, 1);
        disp7.CamDraw(c5, 2, 2, 5);

        TCanvas &c6 = fDisplay->AddTab("Difference Calc.");
        c6.Divide(2,3);
        
        disp8.CamDraw(c6, 1, 2, 1);
        disp9.CamDraw(c6, 2, 2, 5);
        return;
      }

    if (fDisplayType == kDataCheckDisplay)
      {

        TCanvas &c3 = fDisplay->AddTab(fExtractionType!=kFundamental/*fExtractorResolution*/ ? "PedExtrd" : "Ped");
        c3.Divide(2,3);
        
        c3.cd(1);
        gPad->SetBorderMode(0);
        gPad->SetTicks();
        MHCamera *obj1=(MHCamera*)disp0.DrawCopy("hist");
        //
        // for the datacheck, fix the ranges!!
        //
        if (fExtractionType==kFundamental/*!fExtractorResolution*/)
          {
            obj1->SetMinimum(fgPedestalMin);
            obj1->SetMaximum(fgPedestalMax);
            //
            // Set the datacheck sizes:
            //
            FixDataCheckHist((TH1D*)obj1);
            //
            // set reference lines
            //
            DisplayReferenceLines(obj1,0);
          }
        
        //
        // end reference lines
        //
        c3.cd(3);
        gPad->SetBorderMode(0);
        obj1->SetPrettyPalette();
        obj1->Draw();
        
        c3.cd(5);
        gPad->SetBorderMode(0);
        gPad->SetTicks();
        TH1D *obj2 = (TH1D*)obj1->Projection(obj1->GetName());
        obj2->Draw();
        obj2->SetBit(kCanDelete);
        obj2->Fit("gaus","Q");
        obj2->GetFunction("gaus")->SetLineColor(kYellow);
        //
        // Set the datacheck sizes:
        //
        FixDataCheckHist(obj2);
        obj2->SetStats(1);
        
        c3.cd(2);
        gPad->SetBorderMode(0);
        gPad->SetTicks();
        MHCamera *obj3=(MHCamera*)disp1.DrawCopy("hist");
        //
        // for the datacheck, fix the ranges!!
        //
        obj3->SetMinimum(fgPedRmsMin);
        obj3->SetMaximum(fgPedRmsMax);
        //
        // Set the datacheck sizes:
        //
        FixDataCheckHist((TH1D*)obj3);
        //
        // set reference lines
        //
        DisplayReferenceLines(obj3,1);
        
        c3.cd(4);
        gPad->SetBorderMode(0);
        obj3->SetPrettyPalette();
        obj3->Draw();

        c3.cd(6);
        gPad->SetBorderMode(0);
        
        if (geomcam.InheritsFrom("MGeomCamMagic"))
          {
            TArrayI inner(1);
            inner[0] = 0;

            TArrayI outer(1);
            outer[0] = 1;
            
            TArrayI s0(6);
            s0[0] = 6;
            s0[1] = 1;
            s0[2] = 2;
            s0[3] = 3;
            s0[4] = 4;
            s0[5] = 5;
            
            TArrayI s1(3);
            s1[0] = 6;
            s1[1] = 1;
            s1[2] = 2;
            
            TArrayI s2(3);
            s2[0] = 3;
            s2[1] = 4;
            s2[2] = 5;
            
            TVirtualPad *pad = gPad;
            pad->Divide(2,1);
            
            TH1D *inout[2];
            inout[0] = disp1.ProjectionS(s0, inner, "Inner");
            inout[1] = disp1.ProjectionS(s0, outer, "Outer");
            FixDataCheckHist(inout[0]);
            FixDataCheckHist(inout[1]);
            
            inout[0]->SetTitle(Form("%s %s",disp1.GetTitle(),"Inner"));
            inout[1]->SetTitle(Form("%s %s",disp1.GetTitle(),"Outer"));
            
            
            for (int i=0; i<2; i++)
              {
                pad->cd(i+1);
                gPad->SetBorderMode(0);
                gPad->SetTicks();
                
                inout[i]->SetDirectory(NULL);
                inout[i]->SetLineColor(kRed+i);
                inout[i]->SetBit(kCanDelete);
                inout[i]->Draw();
                inout[i]->Fit("gaus", "Q");
                
                TLegend *leg2 = new TLegend(0.6,0.2,0.9,0.55);
                leg2->SetHeader(inout[i]->GetName());
                leg2->AddEntry(inout[i], inout[i]->GetName(), "l");
                
                //
                // Display the outliers as dead and noisy pixels
                //
                DisplayOutliers(inout[i]);
                
                //
                // Display the two half of the camera separately
                //
                TH1D *half[2];
                half[0] = disp1.ProjectionS(s1, i==0 ? inner : outer , "Sector 6-1-2");
                half[1] = disp1.ProjectionS(s2, i==0 ? inner : outer , "Sector 3-4-5");
                
                for (int j=0; j<2; j++)
                  {
                    half[j]->SetLineColor(kRed+i+2*j+1);
                    half[j]->SetDirectory(NULL);
                    half[j]->SetBit(kCanDelete);
                    half[j]->Draw("same");
                    leg2->AddEntry(half[j], half[j]->GetName(), "l");
                  }
                leg2->Draw();
                delete leg2;
              }
            return;
          }
      }
    
    if (fExtractionType!=kFundamental/*fExtractorResolution*/)
      {
            
          TCanvas &c3 = fDisplay->AddTab(fExtractionType==kWithExtractor?"PedExtrd":"PedRndm");
          c3.Divide(2,3);

          disp0.CamDraw(c3, 1, 2, 1);
          disp1.CamDraw(c3, 2, 2, 6);

          TCanvas &c13 = fDisplay->AddTab(fExtractionType==kWithExtractor?"PedDiff":"DiffRndm");
          c13.Divide(2,3);

          disp9.CamDraw(c13, 1, 2, 5);
          disp10.CamDraw(c13, 2, 2, 5);
          return;
      }
}


void  MJPedestal::DisplayReferenceLines(MHCamera *cam, const Int_t what) const
{

  Double_t x = cam->GetNbinsX();
  
  const MGeomCam *geom = cam->GetGeometry();

  if (geom->InheritsFrom("MGeomCamMagic"))
    x = what ? 397 : cam->GetNbinsX();

  TLine line;
  line.SetLineStyle(kDashed);
  line.SetLineWidth(3);
  
  line.SetLineColor(kBlue);
  TLine *l1 = line.DrawLine(0, what ? fgRefPedRmsGalacticInner : fgRefPedGalactic,
                            x, what ? fgRefPedRmsGalacticInner : fgRefPedGalactic);
  
  line.SetLineColor(kYellow);
  TLine *l2 = line.DrawLine(0, what ? fgRefPedRmsExtraGalacticInner : fgRefPedExtraGalactic,
                            x, what ? fgRefPedRmsExtraGalacticInner : fgRefPedExtraGalactic);
  
  line.SetLineColor(kMagenta);
  TLine *l3 = line.DrawLine(0, what ? fgRefPedRmsClosedLidsInner : fgRefPedClosedLids,
                            x, what ? fgRefPedRmsClosedLidsInner : fgRefPedClosedLids);
  
  if (geom->InheritsFrom("MGeomCamMagic"))
    if (what)
      {
        const Double_t x2 = cam->GetNbinsX();

        line.SetLineColor(kBlue);
        line.DrawLine(398, fgRefPedRmsGalacticOuter,
                      x2,  fgRefPedRmsGalacticOuter);
        
        line.SetLineColor(kYellow);
        line.DrawLine(398, fgRefPedRmsExtraGalacticOuter,
                      x2,  fgRefPedRmsExtraGalacticOuter);
        
        line.SetLineColor(kMagenta);
        line.DrawLine(398, fgRefPedRmsClosedLidsOuter,
                      x2,  fgRefPedRmsClosedLidsOuter);
      }
  
    
  TLegend *leg = new TLegend(0.4,0.75,0.7,0.99);
  leg->SetBit(kCanDelete);
  leg->AddEntry(l1, "Galactic Source","l");
  leg->AddEntry(l2, "Extra-Galactic Source","l");
  leg->AddEntry(l3, "Closed Lids","l");
  leg->Draw();
}

void  MJPedestal::DisplayOutliers(TH1D *hist) const
{
    const Float_t mean  = hist->GetFunction("gaus")->GetParameter(1);
    const Float_t lolim = mean - 3.5*hist->GetFunction("gaus")->GetParameter(2);
    const Float_t uplim = mean + 3.5*hist->GetFunction("gaus")->GetParameter(2);
    const Stat_t  dead  = hist->Integral(0,hist->FindBin(lolim)-1);
    const Stat_t  noisy = hist->Integral(hist->FindBin(uplim)+1,hist->GetNbinsX()+1);

    TLatex deadtex;
    deadtex.SetTextSize(0.06);
    deadtex.DrawLatex(0.1,hist->GetBinContent(hist->GetMaximumBin())/1.1,Form("%3i dead pixels",(Int_t)dead));

    TLatex noisytex;
    noisytex.SetTextSize(0.06);
    noisytex.DrawLatex(0.1,hist->GetBinContent(hist->GetMaximumBin())/1.2,Form("%3i noisy pixels",(Int_t)noisy));
}

void MJPedestal::FixDataCheckHist(TH1D *hist) const 
{
    hist->SetDirectory(NULL);
    hist->SetStats(0);

    //
    // set the labels bigger
    //
    TAxis *xaxe = hist->GetXaxis();
    TAxis *yaxe = hist->GetYaxis();
    
    xaxe->CenterTitle();
    yaxe->CenterTitle();    
    xaxe->SetTitleSize(0.06);
    yaxe->SetTitleSize(0.06);    
    xaxe->SetTitleOffset(0.8);
    yaxe->SetTitleOffset(0.5);    
    xaxe->SetLabelSize(0.05);
    yaxe->SetLabelSize(0.05);    
}

/*
Bool_t MJPedestal::WriteEventloop(MEvtLoop &evtloop) const
{
    if (fOutputPath.IsNull())
        return kTRUE;

    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, fOverwrite?"RECREATE":"NEW", "File created by MJPedestal", 9);
    if (!file.IsOpen())
    {
        *fLog << err << "ERROR - Couldn't open file " << oname << " for writing..." << endl;
        return kFALSE;
    }

    if (evtloop.Write(fName)<=0)
    {
        *fLog << err << "Unable to write MEvtloop to " << oname << endl;
        return kFALSE;
    }

    return kTRUE;
}
*/

void MJPedestal::SetExtractor(MExtractor* ext)
{
    if (ext)
    {
        if (fExtractor)
            delete fExtractor;
        fExtractor = ext ? (MExtractor*)ext->Clone(ext->GetName()) : NULL;
    }
    else
        fExtractor = 0;
}

// --------------------------------------------------------------------------
//
// The following resource options are available:
//
// Do a datacheck run (read raw-data and enable display)
//   Prefix.DataCheck:        Yes, No <default>
//
// Show data check display
//   Prefix.DataCheckDisplay: Yes, No <default>
//
// Use cosmic data instead of pedestal data (DatRuns)
//   Prefix.UseData:          Yes, No <default>
//
// Write an output file with pedestals and status-display
//   Prefix.DisableOutput:    Yes, No <default>
//
Bool_t MJPedestal::CheckEnvLocal()
{
    SetDataCheck(GetEnv("DataCheck", fDataCheck));

    if (HasEnv("DataCheckDisplay"))
        fDisplayType = GetEnv("DataCheckDisplay", kFALSE) ? kDataCheckDisplay : kNormalDisplay;

    if (HasEnv("UseData"))
        fExtractType = GetEnv("UseData",kFALSE) ? kUseData : kUsePedRun;

    if (HasEnv("UseHists"))
        if (GetEnv("UseHists",kFALSE))
            fExtractType = kUseHists;

    //fExtractorResolution = GetEnv("ExtractionType", fExtractorResolution);

    SetNoStorage(GetEnv("DisableOutput", IsNoStorage()));

    MTaskEnv tenv("ExtractSignal");
    tenv.SetDefault(fExtractor);

    if (tenv.ReadEnv(*GetEnv(), GetEnvPrefix()+".ExtractSignal", GetEnvDebug())==kERROR)
        return kFALSE;

    if (fExtractor==tenv.GetTask())
        return kTRUE;

    if (!tenv.GetTask()->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << "ERROR: ExtractSignal from resource file doesn't inherit from MExtractor.... abort." << endl;
        return kFALSE;
    }

    SetExtractor((MExtractor*)tenv.GetTask());

    return kTRUE;
}

//---------------------------------------------------------------------------------
//
// Try to write the created MPedestalCam in the output file. 
// If possible, also an MBadPixelsCam and the corresponding display is written.
//
// In case of Storage type "kNoStorage" or if any of the containers
// cannot be written, return kFALSE, otherwise kTRUE.
//
Bool_t MJPedestal::WriteResult()
{
    if (IsNoStorage())
        return kTRUE;

    if (fPathOut.IsNull())
        return kTRUE;

    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "UPDATE", "File created by MJPedestal", 9);
    if (!file.IsOpen())
    {
        *fLog << err << "ERROR - Couldn't open file " << oname << " for writing..." << endl;
        return kFALSE;
    }

    TObjArray cont;

    if (fDisplay)
        cont.Add(fDisplay);

    cont.Add(&fPedestalCamOut);
    cont.Add(&fBadPixels);

    return WriteContainer(cont);
}

Bool_t MJPedestal::Process()
{
    if (!ReadPedestalCam())
        return ProcessFile();

    return kTRUE;
}

Bool_t MJPedestal::ProcessFile()
{
    if (!fSequence.IsValid())
    {
        if (!fRuns)
        {
            *fLog << err << "Neither AddRuns nor SetSequence nor SetEnv was called... abort." << endl;
            return kFALSE;
        }
        if (fRuns && fRuns->GetNumRuns() != fRuns->GetNumEntries())
        {
            *fLog << err << "Number of files found doesn't match number of runs... abort." << endl;
            return kFALSE;
        }
    }

    if (!CheckEnv())
        return kFALSE;

    // --------------------------------------------------------------------------------

    const TString type = IsUseData() ? "data" : "pedestal";

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate MPedestalCam from " << type << "-runs ";
    if (fSequence.IsValid())
        *fLog << fSequence.GetName() << endl;
    else
        if (fRuns)
            *fLog << fRuns->GetRunsAsString() << endl;
        else
            *fLog << "in Resource File." << endl;
    *fLog << endl;

    // --------------------------------------------------------------------------------

    MParList  plist;
    MTaskList tlist;
    plist.AddToList(&tlist);
    plist.AddToList(this); // take care of fDisplay!

    MReadMarsFile read("Events");
    MRawFileRead rawread(NULL);

    MDirIter iter;
    if (fSequence.IsValid())
    {
        const Int_t n0 = IsUseData() ? fSequence.SetupDatRuns(iter, fPathData, "D", fDataCheck) : fSequence.SetupPedRuns(iter, fPathData, "P", fDataCheck);
        const Int_t n1 = IsUseData() ? fSequence.GetNumDatRuns() : fSequence.GetNumPedRuns();
        if (n0==0)
        {
            *fLog << err << "ERROR - No " << type << " input files of sequence found in " << (fPathData.IsNull()?"<defaul>":fPathData.Data()) << endl;
            return kFALSE;
        }
        if (n0!=n1)
        {
            *fLog << err << "ERROR - Number of " << type << " files found (" << n0 << ") in " << (fPathData.IsNull()?"<defaul>":fPathData.Data())  << " doesn't match number of files in sequence (" << n1 << ")" << endl;
            return kFALSE;
        }
    }

    if (fDataCheck)
    {
        if (fRuns || fSequence.IsValid())
            rawread.AddFiles(fSequence.IsValid() ? iter : *fRuns);
        tlist.AddToList(&rawread);
    }
    else
    {
        read.DisableAutoScheme();
        if (fRuns || fSequence.IsValid())
            read.AddFiles(fSequence.IsValid() ? iter : *fRuns);
	tlist.AddToList(&read);
    }

    // Setup Tasklist
    plist.AddToList(&fPedestalCamOut);
    plist.AddToList(&fBadPixels);
    
    MGeomApply geomapl;
    MBadPixelsMerge merge(&fBadPixels);

    MPedCalcPedRun pedcalc;
    pedcalc.SetPedestalUpdate(kFALSE);

    MPedCalcFromLoGain pedlogain;
    pedlogain.SetPedestalUpdate(kFALSE);

    MHPedestalCam hpedcam;
    hpedcam.SetRenorm(kTRUE);
    
    MFillH fillped(&hpedcam, "MExtractedSignalCam", "FillPedestalCam");
    fillped.SetBit(MFillH::kDoNotDisplay);

    tlist.AddToList(&geomapl);
    tlist.AddToList(&merge);

    if (!fPathIn.IsNull())
    {
        fExtractor = ReadCalibration();
        if (!fExtractor)
            return kFALSE;

        *fLog << all;
        *fLog << underline << "Signal Extractor found in calibration file:" << endl;
        fExtractor->Print();
        *fLog << endl;
    }

    // ----------------------------------------------------------------------
    //   Now we make sure, that in all cases the ranges are setup correctly
    // ----------------------------------------------------------------------
    MTaskEnv taskenv("ExtractPedestal");
    switch (fExtractType)
    {
    case kUsePedRun:
        // In case  other than 'fundamental' second argument is obsolete
        pedcalc.SetExtractWindow(0,14);  // kUsePedRun
        taskenv.SetDefault(&pedcalc);
        tlist.AddToList(&taskenv);
        break;

    case kUseData:
        // In case  other than 'fundamental' second argument is obsolete
        pedlogain.SetExtractWindow(15,14); // kUseData
        taskenv.SetDefault(&pedlogain);
        tlist.AddToList(&taskenv);
        break;

    case kUseHists:
        if (!fExtractor)
        {
            *fLog << err << GetDescriptor() << " - ERROR: ";
            *fLog << "Extraction Type is kUseHists, but no extractor was set" << endl;
            return kFALSE;
        }

        tlist.AddToList(fExtractor);
        tlist.AddToList(&fillped);
        break;
    }

    pedcalc.SetPedestalsIn(&fPedestalCamIn);
    pedlogain.SetPedestalsIn(&fPedestalCamIn);

     // kFundamental
    if (fExtractor)
    {
        if (!fExtractor->InheritsFrom("MExtractTimeAndCharge"))
        {
            const Float_t f = 0.5+fExtractor->GetHiGainFirst();
            const Float_t n = 0.5+fExtractor->GetNumHiGainSamples();
            pedcalc.SetExtractWindow((Int_t)f, (Int_t)n);
            pedlogain.SetExtractWindow((Int_t)(15+f), (Int_t)n);

            if (fExtractionType!=kFundamental)
            {
                *fLog << inf;
                *fLog << "Signal extractor doesn't inherit from MExtractTimeAndCharge..." << endl;
                *fLog << " --> falling back to fundamental pedestal extraction." << endl;
                fExtractionType=kFundamental;
            }
        }

        if (fExtractionType!=kFundamental)
        {
            pedcalc.SetRandomCalculation(fExtractionType==kWithExtractorRndm);
            pedlogain.SetRandomCalculation(fExtractionType==kWithExtractorRndm);

            pedcalc.SetExtractor((MExtractTimeAndCharge*)fExtractor);
            pedlogain.SetExtractor((MExtractTimeAndCharge*)fExtractor);
        }
    }

    /*
    switch (fExtractType)
    {
    case kUseData:
        *fLog << all << "TYPE: USEDATA " << fExtractor << endl;
        taskenv.SetDefault(&pedlogain);
        tlist.AddToList(&taskenv);

        if (!SetupExtractor(plist, pedlogain))
        {
            *fLog << all <<  "SETTING TO: " << fExtractor << " " << fExtractor->GetNumHiGainSamples() << endl;
            fExtractor->Print();
            pedlogain.SetExtractWindow(15, (Int_t)TMath::Nint(fExtractor->GetNumHiGainSamples()));
        }
        break;

    case kUsePedRun:
        *fLog << all <<  "TYPE: USEPEDRUN " << fExtractor << endl;
        taskenv.SetDefault(&pedcalc);
        tlist.AddToList(&taskenv);

        if (!SetupExtractor(plist, pedcalc))
            pedcalc.SetExtractWindow(fExtractor->GetHiGainFirst(), TMath::Nint(fExtractor->GetNumHiGainSamples()));
        break;

    case kUseHists:
        if (!fExtractor)
        {
            *fLog << err << GetDescriptor() << " - ERROR: ";
            *fLog << "Extraction Type is kUseHists, but no extractor was set" << endl;
            return kFALSE;
        }

        tlist.AddToList(fExtractor);
        tlist.AddToList(&fillped);
        break;
    } */

    /*
    if (!fPathIn.IsNull())
    {
        delete fExtractor;
        fExtractor = 0;
    }
    */
    //
    // Execute the eventloop
    //
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    //    if (!WriteEventloop(evtloop))
    //        return kFALSE;

    // Execute first analysis
    if (!evtloop.Eventloop(fMaxEvents))
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    DisplayResult(plist);

    if (!WriteResult())
        return kFALSE;

    *fLog << all << GetDescriptor() << ": Done." << endl;
    *fLog << endl << endl;

    return kTRUE;
}
