/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Markus Gaug, 02/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */
/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibration
//
//  Do one calibration loop over serious of runs with the same pulser 
//  colour and the same intensity. The following containers (rectangular) and 
//  tasks (ellipses) are called to produce an MCalibrationChargeCam and to 
//  update the MCalibrationQECam: (MCalibrate is not called from this class)
//
//Begin_Html
/*
<img src="images/CalibClasses.gif">
*/
//End_Html
//
// Different signal extractors can be set with the command SetExtractor()
// Only extractors deriving from MExtractor can be set, default is MExtractSlidingWindow
//
// Different arrival time extractors can be set with the command SetTimeExtractor()
// Only extractors deriving from MExtractTime can be set, default is MExtractTimeSpline
//
// At the end of the eventloop, plots and results are displayed, depending on 
// the flags set (see DisplayResult()) 
// 
// If the flag SetFullDisplay() is set, all MHCameras will be displayed. 
// if the flag SetDataCheckDisplay() is set, only the most important ones are displayed 
// Otherwise, (default: SetNormalDisplay()), a good selection of plots is given
//
// If the flag SetDataCheck() is set, the calibration is used as in the data check at 
// La Palma, which mean especially running on raw data files.
// 
// The absolute light calibration devices Blind Pixel and PIN Diode can be switched on
// and off with the commands:
//
// - SetUseBlindPixel(Bool_t )
// - SetUsePINDiode(Bool_t )
//
// See also: MHCalibrationChargePix, MHCalibrationChargeCam, MHGausEvents
//           MHCalibrationChargeBlindPix, MHCalibrationChargePINDiode
//           MCalibrationChargePix, MCalibrationChargeCam, MCalibrationChargeCalc
//           MCalibrationBlindPix, MCalibrationChargePINDiode,
//           MCalibrationQECam, MBadPixelsPix, MBadPixelsCam
//
// If the flag RelTimeCalibration() is set, a calibration of the relative arrival 
// times is also performed. The following containers (rectangular) and 
// tasks (ellipses) are called to produce an MCalibrationRelTimeCam used by  
// MCalibrateTime to correct timing offset between pixels: (MCalibrateTime is not 
// called from this class)
//
//Begin_Html
/*
<img src="images/RelTimeClasses.gif">
*/
//End_Html
//
// Different arrival time extractors can be set directly with the command SetTimeExtractor(MExtractor *)
//
// See also: MHCalibrationRelTimePix, MHCalibrationRelTimeCam, MHGausEvents
//           MCalibrationRelTimePix, MCalibrationRelTimeCam
//           MBadPixelsPix, MBadPixelsCam
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibration.h"

#include <TFile.h>
#include <TF1.h>
#include <TStyle.h>
#include <TCanvas.h>
#include <TSystem.h>
#include <TLine.h>
#include <TLatex.h>
#include <TLegend.h>
#include <TRegexp.h>
#include <TPaveText.h>
#include <TPaveStats.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MRunIter.h"
#include "MSequence.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MHCamera.h"
#include "MGeomCam.h"

#include "MPedestalCam.h"
#include "MCalibColorSteer.h"

#include "MCalibrationIntensityChargeCam.h"
#include "MCalibrationIntensityBlindCam.h"
#include "MCalibrationIntensityRelTimeCam.h"
#include "MCalibrationIntensityQECam.h"

#include "MCalibrationCam.h"
#include "MCalibrationQECam.h"
#include "MCalibrationQEPix.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationChargePix.h"
#include "MCalibrationChargePINDiode.h"
#include "MCalibrationBlindPix.h"
#include "MCalibrationBlindCam.h"
#include "MCalibrationBlindCamOneOldStyle.h"
#include "MCalibrationBlindCamTwoNewStyle.h"
#include "MCalibrationBlindCamThreeNewStyle.h"
#include "MCalibrationChargeCalc.h"
#include "MCalibColorSet.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationRelTimeCalc.h"

#include "MHGausEvents.h"
#include "MHCalibrationCam.h"
#include "MHCalibrationChargeCam.h"
#include "MHCalibrationChargeBlindCam.h"
#include "MHCalibrationChargePINDiode.h"
#include "MHCalibrationRelTimeCam.h"

#include "MReadMarsFile.h"
#include "MPedCalcPedRun.h"
#include "MRawFileRead.h"
#include "MGeomApply.h"
#include "MTaskEnv.h"
#include "MBadPixelsMerge.h"
#include "MBadPixelsCam.h"
#include "MExtractTime.h"
#include "MExtractor.h"
#include "MExtractPINDiode.h"
#include "MExtractBlindPixel.h"
#include "MExtractSlidingWindow.h"
#include "MExtractTimeFastSpline.h"
#include "MFCosmics.h"
#include "MContinue.h"
#include "MFillH.h"

#include "MArrivalTimeCam.h"

#include "MStatusDisplay.h"

ClassImp(MJCalibration);

using namespace std;

const Int_t MJCalibration::gkIFAEBoxInaugurationRun        = 20113;
const Int_t MJCalibration::gkSecondBlindPixelInstallation  = 31693;
const Int_t MJCalibration::gkSpecialPixelsContInstallation = 34057;
const Int_t MJCalibration::gkThirdBlindPixelInstallation   = 43308;

const Double_t MJCalibration::fgConvFADC2PheMin           = 0.;
const Double_t MJCalibration::fgConvFADC2PheMax           = 1.5;
const Double_t MJCalibration::fgConvFADC2PhotMin          = 0.; 
const Double_t MJCalibration::fgConvFADC2PhotMax          = 10.; 
const Double_t MJCalibration::fgQEMin                     = 0.;
const Double_t MJCalibration::fgQEMax                     = 0.3;
const Double_t MJCalibration::fgArrivalTimeMin            = 1.;
const Double_t MJCalibration::fgArrivalTimeMax            = 10.;

const Float_t  MJCalibration::fgRefConvFADC2PheInner      = 0.14;       
const Float_t  MJCalibration::fgRefConvFADC2PheOuter      = 0.4;       
const Float_t  MJCalibration::fgRefConvFADC2PhotInner     = 0.8; 
const Float_t  MJCalibration::fgRefConvFADC2PhotOuter     = 3.8; 
const Float_t  MJCalibration::fgRefQEInner                = 0.18;            
const Float_t  MJCalibration::fgRefQEOuter                = 0.12;
const Float_t  MJCalibration::fgRefArrivalTimeInner       = 4.5;            
const Float_t  MJCalibration::fgRefArrivalTimeOuter       = 5.0;
const Float_t  MJCalibration::fgRefArrivalTimeRmsInner    = 0.5;            
const Float_t  MJCalibration::fgRefArrivalTimeRmsOuter    = 0.5;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// - Sets fRuns to 0, fExtractor to NULL, fTimeExtractor to NULL, fColor to kNONE, 
//   fDisplay to kNormalDisplay, kRelTimes to kFALSE, kataCheck to kFALSE, kDebug to kFALSE,
//   kIntensity to kFALSE
// - SetUseBlindPixel()
// - SetUsePINDiode()
//
MJCalibration::MJCalibration(const char *name, const char *title) 
    : fRuns(0), fExtractor(NULL), fTimeExtractor(NULL), 
      fColor(MCalibrationCam::kNONE), fDisplayType(kDataCheckDisplay),
      fGeometry("MGeomCamMagic")
{

  fName  = name  ? name  : "MJCalibration";
  fTitle = title ? title : "Tool to create the calibration constants for one calibration run";
  
  SetUseBlindPixel();
  SetUsePINDiode();
  
  SetRelTimeCalibration();
  SetDataCheck(kFALSE);
  SetDebug(kFALSE);
  SetIntensity(kFALSE);

  SetNoStorage(kFALSE);
  SetHistsStorage(kFALSE);
}

void MJCalibration::DrawTab(MParList &plist, const char *cont, const char *name, Option_t *opt)
{
    TObject *obj = plist.FindObject(cont);
    if (!obj)
        return;

    fDisplay->AddTab(name);
    obj->DrawClone(opt);
}

// --------------------------------------------------------------------------
//
// Display the results in MStatusDisplay: 
//
// - Add "Calibration" to the MStatusDisplay title
// - Retrieve the MGeomCam from MParList
// - Initialize the following MHCamera's:
//   1)  MCalibrationPix::GetMean()
//   2)  MCalibrationPix::Sigma()
//   3)  MCalibrationChargePix::GetRSigma()
//   4)  MCalibrationChargePix::GetRSigmaPerCharge()
//   5)  MCalibrationChargePix::GetPheFFactorMethod()
//   6)  MCalibrationChargePix::GetMeanConvFADC2Phe()
//   7)  MCalibrationChargePix::GetMeanFFactorFADC2Phot()
//   8)  MCalibrationQEPix::GetQECascadesFFactor()
//   9)  MCalibrationQEPix::GetQECascadesBlindPixel()
//   10) MCalibrationQEPix::GetQECascadesPINDiode()
//   11) MCalibrationQEPix::GetQECascadesCombined()
//   12) MCalibrationQEPix::IsAverageQEFFactorAvailable()
//   13) MCalibrationQEPix::IsAverageQEBlindPixelAvailable()
//   14) MCalibrationQEPix::IsAverageQEPINDiodeAvailable()
//   15) MCalibrationQEPix::IsAverageQECombinedAvailable()
//   16) MCalibrationChargePix::IsHiGainSaturation()
//   17) MCalibrationPix::GetHiLoMeansDivided()
//   18) MCalibrationPix::GetHiLoSigmasDivided()
//   19) MCalibrationChargePix::GetHiGainPickup()
//   20) MCalibrationChargePix::GetLoGainPickup()
//   21) MCalibrationChargePix::GetHiGainBlackout()
//   22) MCalibrationChargePix::GetLoGainBlackout()
//   23) MCalibrationPix::IsExcluded()
//   24) MBadPixelsPix::IsUnsuitable(MBadPixelsPix::kUnsuitableRun)
//   25) MBadPixelsPix::IsUnsuitable(MBadPixelsPix::kUnreliableRun)
//   26) MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kHiGainOscillating)
//   27) MBadPixelsPix::IsUncalibrated(MBadPixelsPix::kLoGainOscillating)
//   28) MCalibrationChargePix::GetAbsTimeMean()
//   29) MCalibrationChargePix::GetAbsTimeRms()
//
// If the flag SetFullDisplay() is set, all MHCameras will be displayed. 
// if the flag SetDataCheckDisplay() is set, only the most important ones are displayed 
// and otherwise, (default: SetNormalDisplay()), a good selection of plots is given
//
void MJCalibration::DisplayResult(MParList &plist)
{
    if (!fDisplay)
        return;

    TString drawoption = "nonew ";
    if (fDisplayType == kDataCheckDisplay)
        drawoption += "datacheck";
    if (fDisplayType == kFullDisplay)
        drawoption += "all";

    if (IsUsePINDiode())
        DrawTab(plist, "MHCalibrationChargePINDiode", "PINDiode",  drawoption);
    if (IsUseBlindPixel())
        DrawTab(plist, "MHCalibrationChargeBlindCam", "BlindPix",  drawoption);
    if (IsRelTimes())
        DrawTab(plist, "MHCalibrationRelTimeCam",     "Time",      drawoption);
    DrawTab(plist, "MHCalibrationChargeCam",          "Charge",    drawoption);

    //
    // Update display
    //
    TString title = fDisplay->GetTitle();
    title += "--  Calibration ";
    title += fSequence.IsValid() ? Form("calib%08d", fSequence.GetSequence()) : (const char*)fRuns->GetRunsAsString();
    title += "  --";
    fDisplay->SetTitle(title);

    //
    // Get container from list
    //
    MGeomCam &geomcam = *(MGeomCam*)plist.FindObject("MGeomCam");

    // Create histograms to display
    MHCamera disp1 (geomcam, "Charge",            "Fitted Mean Signal (Charges)");
    MHCamera disp2 (geomcam, "SigmaCharge",       "Sigma of Fitted Signal");
    MHCamera disp3 (geomcam, "RSigma",            "Reduced Sigmas");
    MHCamera disp4 (geomcam, "RSigmaPerCharge",   "Reduced Sigma per Charge");
    MHCamera disp5 (geomcam, "NumPhes",           "Number Photo-electrons");
    MHCamera disp6 (geomcam, "ConvFADC2Phes",     "Conversion Factor to Phes");
    MHCamera disp7 (geomcam, "TotalFFactor",      "Total F-Factor(F-Factor Method)");
    MHCamera disp8 (geomcam, "CascadesQEFFactor", "Cascades QE (F-Factor Method)");
    MHCamera disp9 (geomcam, "CascadesQEBlindPix","Cascades QE (Blind Pixel Method)");
    MHCamera disp10(geomcam, "CascadesQEPINDiode","Cascades QE (PIN Diode Method)");
    MHCamera disp11(geomcam, "CascadesQECombined","Cascades QE (Combined Method)");
    MHCamera disp12(geomcam, "FFactorValid",      "Pixels with valid F-Factor calibration");
    MHCamera disp13(geomcam, "BlindPixelValid",   "Pixels with valid BlindPixel calibration");
    MHCamera disp14(geomcam, "PINdiodeValid",     "Pixels with valid PINDiode calibration");
    MHCamera disp15(geomcam, "CombinedValid",     "Pixels with valid Combined calibration");
    MHCamera disp16(geomcam, "Saturation",        "Pixels with saturated Hi Gain");
    MHCamera disp17(geomcam, "ConversionMeans",   "Conversion HiGain.vs.LoGain Means");
    MHCamera disp18(geomcam, "ConversionSigmas",  "Conversion HiGain.vs.LoGain Sigmas");
    MHCamera disp19(geomcam, "HiGainPickup",      "Number Pickup events Hi Gain");
    MHCamera disp20(geomcam, "LoGainPickup",      "Number Pickup events Lo Gain");
    MHCamera disp21(geomcam, "HiGainBlackout",    "Number Blackout events Hi Gain");
    MHCamera disp22(geomcam, "LoGainBlackout",    "Number Blackout events Lo Gain");
    MHCamera disp23(geomcam, "Excluded",          "Pixels previously excluded");
    MHCamera disp24(geomcam, "UnSuitable",        "Pixels not suited for further analysis");
    MHCamera disp25(geomcam, "UnReliable",        "Pixels suitable, but not reliable for further analysis");
    MHCamera disp26(geomcam, "HiGainOscillating", "Oscillating Pixels High Gain");
    MHCamera disp27(geomcam, "LoGainOscillating", "Oscillating Pixels Low Gain");
    MHCamera disp28(geomcam, "AbsTimeMean",       "Abs. Arrival Times");
    MHCamera disp29(geomcam, "AbsTimeRms",        "RMS of Arrival Times");
    MHCamera disp30(geomcam, "MeanTime",          "Mean Rel. Arrival Times");
    MHCamera disp31(geomcam, "SigmaTime",         "Sigma Rel. Arrival Times");
    MHCamera disp32(geomcam, "TimeProb",          "Probability of Time Fit");
    MHCamera disp33(geomcam, "TimeNotFitValid",   "Pixels with not valid fit results");
    MHCamera disp34(geomcam, "TimeOscillating",   "Oscillating Pixels");
    MHCamera disp35(geomcam, "TotalConv",         "Conversion Factor to photons");
    MHCamera disp36(geomcam, "RMSperMean",        "Charge histogram RMS per Mean");
    
    MCalibrationChargeCam  *cam    = NULL;
    MCalibrationQECam      *qecam  = NULL;
    MCalibrationRelTimeCam *relcam = NULL;
    MBadPixelsCam          *badcam = NULL;

    if (IsIntensity())
    {
        cam    = (MCalibrationChargeCam*) fIntensCalibCam.GetCam();
        qecam  = (MCalibrationQECam*)     fIntensQECam.GetCam();
        relcam = (MCalibrationRelTimeCam*)fIntensRelTimeCam.GetCam();
        badcam = (MBadPixelsCam*)         fIntensBadCam.GetCam();
    }
    else
    {
        cam    = &fCalibrationCam;
        qecam  = &fQECam;
        relcam = &fRelTimeCam;
        badcam = &fBadPixels;
    }
    
    // Fitted charge means and sigmas
    disp1.SetCamContent(*cam,  0);
    disp1.SetCamError(  *cam,  1);
    disp2.SetCamContent(*cam,  2);
    disp2.SetCamError(  *cam,  3);

    // Reduced Sigmas and reduced sigmas per charge
    disp3.SetCamContent(*cam,  5);
    disp3.SetCamError(  *cam,  6);
    disp4.SetCamContent(*cam,  7);
    disp4.SetCamError(  *cam,  8);

    // F-Factor Method
    disp5.SetCamContent(*cam,  9);
    disp5.SetCamError(  *cam, 10);
    disp6.SetCamContent(*cam, 11);
    disp6.SetCamError(  *cam, 12);
    disp7.SetCamContent(*cam, 13);
    disp7.SetCamError(  *cam, 14);

    // Quantum Efficiencies
    disp8.SetCamContent (*qecam, 0 );
    disp8.SetCamError   (*qecam, 1 );
    disp9.SetCamContent (*qecam, 2 );
    disp9.SetCamError   (*qecam, 3 );
    disp10.SetCamContent(*qecam, 4 );
    disp10.SetCamError  (*qecam, 5 );
    disp11.SetCamContent(*qecam, 6 );
    disp11.SetCamError  (*qecam, 7 );

    // Valid flags
    disp12.SetCamContent(*qecam, 8 );
    disp13.SetCamContent(*qecam, 9 );
    disp14.SetCamContent(*qecam, 10);
    disp15.SetCamContent(*qecam, 11);

    // Conversion Hi-Lo
    disp16.SetCamContent(*cam, 25);
    disp17.SetCamContent(*cam, 16);
    disp17.SetCamError  (*cam, 17);
    disp18.SetCamContent(*cam, 18);
    disp18.SetCamError  (*cam, 19);

    // Pickup and Blackout
    disp19.SetCamContent(*cam, 21);
    disp20.SetCamContent(*cam, 22);
    disp21.SetCamContent(*cam, 23);
    disp22.SetCamContent(*cam, 24);

    // Pixels with defects
    disp23.SetCamContent(*cam, 20);
    disp24.SetCamContent(*badcam, 6);
    disp25.SetCamContent(*badcam, 7);

    // Oscillations
    disp26.SetCamContent(*badcam, 10);
    disp27.SetCamContent(*badcam, 11);

    // Arrival Times
    disp28.SetCamContent(*cam, 26);
    disp28.SetCamError(  *cam, 27);
    disp29.SetCamContent(*cam, 27);

    // RMS and Mean
    disp36.SetCamContent(*cam,32);

    disp1.SetYTitle("Q [FADC counts]");
    disp2.SetYTitle("\\sigma_{Q} [FADC counts]");

    disp3.SetYTitle("\\sqrt{\\sigma^{2}_{Q} - RMS^{2}_{Ped}} [FADC Counts]");
    disp4.SetYTitle("Red.Sigma/<Q> [1]");

    disp5.SetYTitle("Photo-electons [1]");
    disp6.SetYTitle("Conv.Factor [PhE/FADC counts]");
    disp7.SetYTitle("Total F-Factor [1]");

    disp8.SetYTitle("QE [1]");
    disp9.SetYTitle("QE [1]");
    disp10.SetYTitle("QE [1]");
    disp11.SetYTitle("QE [1]");

    disp12.SetYTitle("[1]");
    disp13.SetYTitle("[1]");
    disp14.SetYTitle("[1]");
    disp15.SetYTitle("[1]");
    disp16.SetYTitle("[1]");

    disp17.SetYTitle("<Q>(High)/<Q>(Low) [1]");
    disp18.SetYTitle("\\sigma_{Q}(High)/\\sigma_{Q}(Low) [1]");

    disp19.SetYTitle("[1]");
    disp20.SetYTitle("[1]");
    disp21.SetYTitle("[1]");
    disp22.SetYTitle("[1]");
    //    disp23.SetYTitle("[1]");
    //    disp24.SetYTitle("[1]");
    //    disp25.SetYTitle("[1]");
    disp26.SetYTitle("[1]");
    disp27.SetYTitle("[1]");

    disp28.SetYTitle("Mean Abs. Time [FADC slice]");
    disp29.SetYTitle("RMS Abs. Time [FADC slices]");

    disp35.SetYTitle("Conv.Factor [Ph/FADC counts]");

    disp36.SetYTitle("Charge RMS/<Q> [1]");

    for (UInt_t i=0;i<geomcam.GetNumPixels();i++)
      {

        MCalibrationChargePix &pix = (MCalibrationChargePix&)(*cam)  [i];
        MCalibrationQEPix     &qe  = (MCalibrationQEPix&)    (*qecam)[i];

        if (!pix.IsFFactorMethodValid())
          continue;

        const Float_t convphe  = pix.GetMeanConvFADC2Phe();
        const Float_t quaeff   = qe.GetQECascadesFFactor(0.);
        
        disp35.Fill(i,convphe/quaeff);
        disp35.SetUsed(i);
      }
    

    if (IsRelTimes())
    {
        disp30.SetCamContent(*relcam,0);
        disp30.SetCamError(  *relcam,1);
        disp31.SetCamContent(*relcam,2);
        disp31.SetCamError(  *relcam,3);
        disp32.SetCamContent(*relcam,4);
        disp33.SetCamContent(fBadPixels,20);
        disp34.SetCamContent(fBadPixels,21);
    
        disp30.SetYTitle("Time Offset [FADC units]");
        disp31.SetYTitle("Timing resolution [FADC units]");
        disp32.SetYTitle("P_{Time} [1]");
        disp33.SetYTitle("[1]");
        disp34.SetYTitle("[1]");
    }

    if (fDisplayType == kDataCheckDisplay)
    {
      
      TCanvas &c1 = fDisplay->AddTab("FitCharge");
      c1.Divide(3, 3);
      
      //
      // MEAN CHARGES
      // 
      
      c1.cd(1);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj1=(MHCamera*)disp1.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      // obj1->SetMinimum(fgChargeMin);
      // obj1->SetMaximum(fgChargeMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj1);
      obj1->SetStats(kFALSE);
      //
      // set reference lines
      //
      //        DisplayReferenceLines(obj1,0);
      
      c1.cd(4);
      gPad->SetBorderMode(0);
      obj1->SetPrettyPalette();
      obj1->Draw();
      
      c1.cd(7);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      TH1D *obj2 = (TH1D*)obj1->Projection(obj1->GetName());
      obj2->Draw();
      obj2->SetBit(kCanDelete);
      obj2->Fit("gaus","Q");
      TF1 *fun2 = obj2->GetFunction("gaus");
      fun2->SetLineColor(kYellow);
      gPad->Modified();
      gPad->Update();      
      TPaveStats *st = (TPaveStats*)obj2->GetListOfFunctions()->FindObject("stats");
      st->SetY1NDC(0.55);
      st->SetY2NDC(0.89);
      st->SetX1NDC(0.65);
      st->SetX2NDC(0.99);
      gPad->Modified();
      gPad->Update();      
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist(obj2);
      obj2->SetStats(1);
      
      //
      // Display the outliers as dead and noisy pixels
      //
      DisplayOutliers(obj2,"low-ampl.","high-ampl.");
      TLatex flattex;
      flattex.SetTextSize(0.07);
      const Double_t minl = obj2->GetBinCenter(obj2->GetXaxis()->GetFirst());
      const Double_t maxl = obj2->GetBinCenter(obj2->GetXaxis()->GetLast());
      flattex.DrawLatex(minl+0.015*(maxl-minl),obj2->GetBinContent(obj2->GetMaximumBin())/1.35,
                        Form("Flatfield precision: %4.2f%%",
                             fun2->GetParameter(2)/fun2->GetParameter(1)*100.));
      
      //
      // RMS per Charge
      // 
      
      c1.cd(2);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj3=(MHCamera*)disp36.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      //      obj3->SetMinimum(0.);
      // obj3->SetMaximum(fgChargeMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj3);
      obj3->SetStats(kFALSE);
      //
      // set reference lines
      //
      //        DisplayReferenceLines(obj3,0);
      
      c1.cd(5);
      gPad->SetBorderMode(0);
      obj3->SetPrettyPalette();
      obj3->Draw();
      
      c1.cd(8);
      gPad->SetBorderMode(0);
      if (geomcam.InheritsFrom("MGeomCamMagic"))
        DisplayDoubleProject(&disp36, "dead", "noisy");
      
      //
      // PHOTO ELECTRONS
      // 
      
      c1.cd(3);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj4=(MHCamera*)disp5.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      // obj3->SetMinimum(fgChargeMin);
      // obj3->SetMaximum(fgChargeMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj4);
      obj4->SetStats(kFALSE);
      //
      // set reference lines
      //
      //        DisplayReferenceLines(obj3,0);
      
      c1.cd(6);
      gPad->SetBorderMode(0);
      obj4->SetPrettyPalette();
      obj4->Draw();
      
      c1.cd(9);
      gPad->SetBorderMode(0);
      if (geomcam.InheritsFrom("MGeomCamMagic"))
        DisplayDoubleProject(&disp5, "dead", "noisy");
      
      //
      // CONVERSION FACTORS
      // 
      TCanvas &c2 = fDisplay->AddTab("Conversion");
      c2.Divide(3,3);
      
      c2.cd(1);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj5=(MHCamera*)disp6.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      obj5->SetMinimum(fgConvFADC2PheMin);
      obj5->SetMaximum(fgConvFADC2PheMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj5);
      obj5->SetStats(kFALSE);
      //
      // set reference lines
      //
      DisplayReferenceLines(obj5,2);
      
      c2.cd(4);
      gPad->SetBorderMode(0);
      obj5->SetPrettyPalette();
      obj5->Draw();
      
      c2.cd(7);
      gPad->SetBorderMode(0);
      if (geomcam.InheritsFrom("MGeomCamMagic"))
        DisplayDoubleProject(&disp6, "noisy", "dead");
      
      //
      // QUANTUM EFFICIENCY
      //
      c2.cd(2);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj6=(MHCamera*)disp8.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      obj6->SetMinimum(fgQEMin);
      obj6->SetMaximum(fgQEMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj6);
      obj6->SetStats(kFALSE);
      //
      // set reference lines
      //
      DisplayReferenceLines(obj6,0);
      
      c2.cd(5);
      gPad->SetBorderMode(0);
      obj6->SetPrettyPalette();
      obj6->Draw();
      
      c2.cd(8);
      gPad->SetBorderMode(0);
      if (geomcam.InheritsFrom("MGeomCamMagic"))
        DisplayDoubleProject(&disp8, "noisy", "dead");
      
      //
      // CONVERSION FADC TO PHOTONS
      // 
      
      c2.cd(3);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj7=(MHCamera*)disp35.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      obj7->SetMinimum(fgConvFADC2PhotMin);
      obj7->SetMaximum(fgConvFADC2PhotMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj7);
      obj7->SetStats(kFALSE);
      //
      // set reference lines
      //
      DisplayReferenceLines(obj7,1);
      
      c2.cd(6);
      gPad->SetBorderMode(0);
      obj7->SetPrettyPalette();
      obj7->Draw();
      c2.cd(9);
      gPad->SetBorderMode(0);
      if (geomcam.InheritsFrom("MGeomCamMagic"))
        DisplayDoubleProject(&disp35, "noisy", "dead");

      //
      // ARRIVAL TIMES
      // 
      TCanvas &c3 = fDisplay->AddTab("AbsTimes");
      c3.Divide(2,3);
      
      c3.cd(1);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj10=(MHCamera*)disp28.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      obj10->SetMinimum(fgArrivalTimeMin);
      obj10->SetMaximum(fgArrivalTimeMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj10);
      obj10->SetStats(kFALSE);
      //
      // set reference lines
      //
      DisplayReferenceLines(obj10,3);
      
      c3.cd(3);
      gPad->SetBorderMode(0);
      obj10->SetPrettyPalette();
      obj10->Draw();
      
      c3.cd(5);
      gPad->SetBorderMode(0);
      if (geomcam.InheritsFrom("MGeomCamMagic"))
        DisplayDoubleProject(&disp28, "early", "late");

      //
      // ARRIVAL TIMES JITTER
      // 
      c3.cd(2);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj11=(MHCamera*)disp29.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      //      obj11->SetMinimum(fgArrivalTimeMin);
      //      obj11->SetMaximum(fgArrivalTimeMax);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj11);
      obj11->SetStats(kFALSE);
      //
      // set reference lines
      //
      DisplayReferenceLines(obj11,4);
      
      c3.cd(4);
      gPad->SetBorderMode(0);
      obj11->SetPrettyPalette();
      obj11->Draw();
      
      c3.cd(6);
      gPad->SetBorderMode(0);
      if (geomcam.InheritsFrom("MGeomCamMagic"))
        DisplayDoubleProject(&disp29, "", "jittering");

      //
      // UNSUITABLE PIXELS
      //
      TCanvas &c4 = fDisplay->AddTab("Defect");
      c4.Divide(2,2);
      
      c4.cd(1);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj8=(MHCamera*)disp24.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      const Double_t max = 11.;
      obj8->SetMinimum(0.);
      obj8->SetMaximum(11.);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj8);
      obj8->SetStats(kFALSE);

      gStyle->SetPalette(1);
      const Int_t numcol = gStyle->GetNumberOfColors()-3;
      
      TPaveText *pave = new TPaveText(0.0,0.0,0.99,0.99);
      pave->SetBit(kCanDelete);
      pave->ConvertNDCtoPad();
      pave->SetTextSize(0.05);
      pave->AddText("  ");
      TText *t1 = pave->AddText(Form("Signal smaller 3 Pedestal RMS:                 %3i pixels",
                                       CountBadPixels(&disp24,1)));
      t1->SetTextColor(gStyle->GetColorPalette(Int_t(1./max*numcol + 1.)));
      t1->SetTextAlign(12);
      TText *t2 = pave->AddText(Form("%s%3i%s","Signal Rel. error too large:                          ",
                                       CountBadPixels(&disp24,2)," pixels"));
      t2->SetTextColor(gStyle->GetColorPalette(Int_t(2./max*numcol + 1.)));
      t2->SetTextAlign(12);
      TText *t4 = pave->AddText(Form("Low Gain Saturation:                                   %3i pixels",
                                       CountBadPixels(&disp24,3)));
      t4->SetTextColor(gStyle->GetColorPalette(Int_t(3./max*numcol + 1.)));
      t4->SetTextAlign(12);
      TText *t5 = pave->AddText(Form("Mean Arr. Time In First Extraction Bin:      %3i pixels",
                                       CountBadPixels(&disp24,4)));
      t5->SetTextColor(gStyle->GetColorPalette(Int_t(4./max*numcol + 1.)));
      t5->SetTextAlign(12);
      TText *t6 = pave->AddText(Form("Mean Arr. Time In Last 2 Extraction Bins: %3i pixels",
                                       CountBadPixels(&disp24,5)));
      t6->SetTextColor(gStyle->GetColorPalette(Int_t(5./max*numcol + 1.)));
      t6->SetTextAlign(12);
      TText *t7 = pave->AddText(Form("Deviating Number of Photo-electrons:      %3i pixels",
                                       CountBadPixels(&disp24,6)));
      t7->SetTextColor(gStyle->GetColorPalette(Int_t(6./max*numcol + 1.)));
      t7->SetTextAlign(12);
      TText *t9 = pave->AddText(Form("Deviating Number of Photons:                    %3i pixels",
                                       CountBadPixels(&disp24,7)));
      t9->SetTextColor(gStyle->GetColorPalette(Int_t(7./max*numcol + 1.)));
      t9->SetTextAlign(12);
      TText *t10= pave->AddText(Form("High-Gain Histogram Overflow:                  %3i pixels",
                                       CountBadPixels(&disp24,8 )));
      t10->SetTextColor(gStyle->GetColorPalette(Int_t(8./max*numcol + 1.)));
      t10->SetTextAlign(12);
      TText *t11= pave->AddText(Form("Low-Gain Histogram Overflow:                   %3i pixels",
                                       CountBadPixels(&disp24,9 )));
      t11->SetTextColor(gStyle->GetColorPalette(Int_t(9./max*numcol + 1.)));
      t11->SetTextAlign(12);
      TText *t12= pave->AddText(Form("Previously Excluded:                                    %3i pixels",
                                       CountBadPixels(&disp24,10)));
      t12->SetTextColor(gStyle->GetColorPalette(Int_t(10./max*numcol + 1.)));
      t12->SetTextAlign(12);
      pave->Draw();

      c4.cd(3);
      gPad->SetBorderMode(0);
      obj8->Draw();
      obj8->SetPrettyPalette();

      //
      // UNRELIABLE PIXELS
      // 
      
      c4.cd(2);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      MHCamera *obj9=(MHCamera*)disp25.DrawCopy("hist");
      //
      // for the datacheck, fix the ranges!!
      //
      const Double_t max2 = 8.;
      obj9->SetMinimum(0.);
      obj9->SetMaximum(max2);
      //
      // Set the datacheck sizes:
      //
      FixDataCheckHist((TH1D*)obj9);
      obj9->SetStats(kFALSE);

      gStyle->SetPalette(1);
      
      TPaveText *pave2 = new TPaveText(0.0,0.0,0.99,0.99);
      pave2->SetBit(kCanDelete);
      pave2->ConvertNDCtoPad();
      pave2->SetTextSize(0.05);
      pave2->AddText("  ");
      TText *t3 = pave2->AddText(Form("Signal Sigma smaller Pedestal RMS:      %3i pixels",
                                       CountBadPixels(&disp25,1)));
      t3->SetTextColor(gStyle->GetColorPalette(Int_t(1./max*numcol + 1.)));
      t3->SetTextAlign(12);
      TText *tt1 = pave2->AddText(Form("High Gain Signals could not be fitted:   %3i pixels",
                                       CountBadPixels(&disp25,2)));
      tt1->SetTextColor(gStyle->GetColorPalette(Int_t(2./max2*numcol + 1.)));
      tt1->SetTextAlign(12);
      TText *tt2 = pave2->AddText(Form("Low  Gain Signals could not be fitted:   %3i pixels",
                                       CountBadPixels(&disp25,3)));
      tt2->SetTextColor(gStyle->GetColorPalette(Int_t(3./max2*numcol + 1.)));
      tt2->SetTextAlign(12);
      TText *tt3 = pave2->AddText(Form("Relative Arr. Times could not be fitted:  %3i pixels",
                                       CountBadPixels(&disp25,4)));
      tt3->SetTextColor(gStyle->GetColorPalette(Int_t(4./max2*numcol + 1.)));
      tt3->SetTextAlign(12);
      TText *tt4 = pave2->AddText(Form("High Gain Signals Oscillation:                %3i pixels",
                                       CountBadPixels(&disp25,5)));
      tt4->SetTextColor(gStyle->GetColorPalette(Int_t(5./max2*numcol + 1.)));
      tt4->SetTextAlign(12);
      TText *tt5 = pave2->AddText(Form("Low  Gain Signals Oscillation:                %3i pixels",
                                       CountBadPixels(&disp25,6)));
      tt5->SetTextColor(gStyle->GetColorPalette(Int_t(6./max2*numcol + 1.)));
      tt5->SetTextAlign(12); 
      TText *tt6 = pave2->AddText(Form("Relative Arr. Times Oscillation:               %3i pixels",
                                       CountBadPixels(&disp25,7)));
      tt6->SetTextColor(gStyle->GetColorPalette(Int_t(7./max2*numcol + 1.)));
      tt6->SetTextAlign(12);
      TText *tt8 = pave2->AddText(Form("Deviating global F-Factor:                        %3i pixels",
                                       CountBadPixels(&disp25,8)));
      tt8->SetTextColor(gStyle->GetColorPalette(Int_t(8./max2*numcol + 1.)));
      tt8->SetTextAlign(12);
      pave2->Draw();

      c4.cd(4);
      gPad->SetBorderMode(0);
      obj9->SetPrettyPalette();
      obj9->Draw();

      if (IsRelTimes())
        {
          TCanvas &c5 = fDisplay->AddTab("RelTimes");
          c5.Divide(2,3);
          
          //
          // MEAN REL. ARR. TIMES
          //
          c5.cd(1);
          gPad->SetBorderMode(0);
          gPad->SetTicks();
          MHCamera *obj10=(MHCamera*)disp30.DrawCopy("hist");
          //
          // for the datacheck, fix the ranges!!
          //
          //   obj10->SetMinimum(fgConvFADC2PheMin);
          //   obj10->SetMaximum(fgConvFADC2PheMax);
          //
          // Set the datacheck sizes:
          //
          FixDataCheckHist((TH1D*)obj10);
          obj10->SetStats(kFALSE);
          //
          // set reference lines
          //
          //  DisplayReferenceLines(obj10,2);
      
          c5.cd(3);
          gPad->SetBorderMode(0);
          obj10->SetPrettyPalette();
          obj10->Draw();
      
          c5.cd(5);
          gPad->SetBorderMode(0);
          if (geomcam.InheritsFrom("MGeomCamMagic"))
            DisplayDoubleProject(&disp30, "early", "late");
          
          //
          // JITTER Rel. Arr. Times
          //
          c5.cd(2);
          gPad->SetBorderMode(0);
          gPad->SetTicks();
          MHCamera *obj11=(MHCamera*)disp31.DrawCopy("hist");
          //
          // for the datacheck, fix the ranges!!
          //
          //          obj11->SetMinimum(fgQEMin);
          //          obj11->SetMaximum(fgQEMax);
          //
          // Set the datacheck sizes:
          //
          FixDataCheckHist((TH1D*)obj11);
          obj11->SetStats(kFALSE);
          //
          // set reference lines
          //
          //          DisplayReferenceLines(obj11,0);
          
          c5.cd(4);
          gPad->SetBorderMode(0);
          obj11->SetPrettyPalette();
          obj11->Draw();
          
          c5.cd(6);
          gPad->SetBorderMode(0);
          if (geomcam.InheritsFrom("MGeomCamMagic"))
            DisplayDoubleProject(&disp31, "too stable", "jittering");
          
        }
      return;
    }
    
    if (fDisplayType == kNormalDisplay)
      {
        
        // Charges
        TCanvas &c11 = fDisplay->AddTab("FitCharge");
        c11.Divide(2, 4);
        
        disp1.CamDraw(c11, 1, 2, 5, 1);
        disp2.CamDraw(c11, 2, 2, 5, 1);
        
        // Reduced Sigmas
        TCanvas &c12 = fDisplay->AddTab("RedSigma");
        c12.Divide(2,4);
        
        disp3.CamDraw(c12, 1, 2, 5, 1);
        disp4.CamDraw(c12, 2, 2, 5, 1);
        
        //  F-Factor
        TCanvas &c13 = fDisplay->AddTab("Phe's");
        c13.Divide(3,4);
        
        disp5.CamDraw(c13, 1, 3, 5, 1);
        disp6.CamDraw(c13, 2, 3, 5, 1);
        disp7.CamDraw(c13, 3, 3, 5, 1);
        
        // QE's
        TCanvas &c14 = fDisplay->AddTab("QE's");
        c14.Divide(4,4);
        
        disp8.CamDraw(c14, 1, 4, 5, 1);
        disp9.CamDraw(c14, 2, 4, 5, 1);
        disp10.CamDraw(c14, 3, 4, 5, 1);
        disp11.CamDraw(c14, 4, 4, 5, 1);
        
        // Defects
        TCanvas &c15 = fDisplay->AddTab("Defect");
        //      c15.Divide(5,2);
        c15.Divide(4,2);
        
        /*
        disp23.CamDraw(c15, 1, 5, 0);
        disp24.CamDraw(c15, 2, 5, 0);
        disp25.CamDraw(c15, 3, 5, 0);
        disp26.CamDraw(c15, 4, 5, 0);
        disp27.CamDraw(c15, 5, 5, 0);
        */
        disp24.CamDraw(c15, 1, 4, 0);
        disp25.CamDraw(c15, 2, 4, 0);
        disp26.CamDraw(c15, 3, 4, 0);
        disp27.CamDraw(c15, 4, 4, 0);
        
        // Abs. Times
        TCanvas &c16 = fDisplay->AddTab("AbsTimes");
        c16.Divide(2,3);
        
        disp28.CamDraw(c16, 1, 2, 5);
        disp29.CamDraw(c16, 2, 2, 5);

        if (IsRelTimes())
        {
            // Rel. Times
            TCanvas &c17 = fDisplay->AddTab("RelTimes");
            c17.Divide(2,4);

            disp30.CamDraw(c17, 1, 2, 5, 1);
            disp31.CamDraw(c17, 2, 2, 5, 1);
        }
        
        return;
    }
    
    if (fDisplayType == kFullDisplay)
    {
        MHCalibrationCam *cam = (MHCalibrationCam*)plist.FindObject("MHCalibrationChargeCam");

        for (Int_t sector=1;sector<cam->GetAverageSectors();sector++)
        {
            cam->GetAverageHiGainSector(sector).DrawClone("all");
            cam->GetAverageLoGainSector(sector).DrawClone("all");
        }

        // Charges
        TCanvas &c21 = fDisplay->AddTab("FitCharge");
        c21.Divide(2, 4);

        disp1.CamDraw(c21, 1, 2, 2, 1);
        disp2.CamDraw(c21, 2, 2, 2, 1);

        // Reduced Sigmas
        TCanvas &c23 = fDisplay->AddTab("RedSigma");
        c23.Divide(2,4);

        disp3.CamDraw(c23, 1, 2, 2, 1);
        disp4.CamDraw(c23, 2, 2, 2, 1);

        //  F-Factor
        TCanvas &c24 = fDisplay->AddTab("Phe's");
        c24.Divide(3,5);
        
        disp5.CamDraw(c24, 1, 3, 2, 1, 1);
        disp6.CamDraw(c24, 2, 3, 2, 1, 1);
        disp7.CamDraw(c24, 3, 3, 2, 1, 1);
        
        // QE's
        TCanvas &c25 = fDisplay->AddTab("QE's");
        c25.Divide(4,5);
        
        disp8.CamDraw(c25, 1, 4, 2, 1, 1);
        disp9.CamDraw(c25, 2, 4, 2, 1, 1);
        disp10.CamDraw(c25, 3, 4, 2, 1, 1);
        disp11.CamDraw(c25, 4, 4, 2, 1, 1);
        
        // Validity
        TCanvas &c26 = fDisplay->AddTab("Valid");
        c26.Divide(4,2);
        
        disp12.CamDraw(c26, 1, 4, 0);
        disp13.CamDraw(c26, 2, 4, 0);
        disp14.CamDraw(c26, 3, 4, 0);
        disp15.CamDraw(c26, 4, 4, 0);
        
        // Other info
        TCanvas &c27 = fDisplay->AddTab("HiLoGain");
        c27.Divide(3,3);
        
        disp16.CamDraw(c27, 1, 3, 0);
        disp17.CamDraw(c27, 2, 3, 1);
        disp18.CamDraw(c27, 3, 3, 1);
        
        // Pickup
        TCanvas &c28 = fDisplay->AddTab("Pickup");
        c28.Divide(4,2);
        
        disp19.CamDraw(c28, 1, 4, 0);
        disp20.CamDraw(c28, 2, 4, 0);
        disp21.CamDraw(c28, 3, 4, 0);
        disp22.CamDraw(c28, 4, 4, 0);
        
        // Defects
        TCanvas &c29 = fDisplay->AddTab("Defect");
        //      c29.Divide(5,2);
        c29.Divide(4,2);
        
        disp24.CamDraw(c29, 1, 4, 0);
        disp25.CamDraw(c29, 2, 4, 0);
        disp26.CamDraw(c29, 3, 4, 0);
        disp27.CamDraw(c29, 4, 4, 0);
        
        // Abs. Times
        TCanvas &c30 = fDisplay->AddTab("AbsTimes");
        c30.Divide(2,3);
        
        disp28.CamDraw(c30, 1, 2, 2);
        disp29.CamDraw(c30, 2, 2, 1);

        if (IsRelTimes())
        {
            // Rel. Times
            TCanvas &c31 = fDisplay->AddTab("RelTimes");
            c31.Divide(3,5);
            
            disp30.CamDraw(c31, 1, 3, 2, 1, 1);
            disp31.CamDraw(c31, 2, 3, 2, 1, 1);
            disp32.CamDraw(c31, 3, 3, 4, 1, 1);

            // Time Defects
            TCanvas &c32 = fDisplay->AddTab("DefTime");
            c32.Divide(2,2);
            
            disp33.CamDraw(c32, 1, 2, 0);
            disp34.CamDraw(c32, 2, 2, 0);

            MHCalibrationCam *cam = (MHCalibrationCam*)plist.FindObject("MHCalibrationRelTimeCam");
            
            for (Int_t sector=1;sector<cam->GetAverageSectors();sector++)
            {
                cam->GetAverageHiGainSector(sector).DrawClone("fourierevents");
                cam->GetAverageLoGainSector(sector).DrawClone("fourierevents");
            }

        }

        return;
    }
}
    

void  MJCalibration::DisplayReferenceLines(MHCamera *cam, const Int_t what) const
{

  const MGeomCam *geom = cam->GetGeometry();

  Double_t x = geom->InheritsFrom("MGeomCamMagic") ? 397 : cam->GetNbinsX() ;

  TLine line;
  line.SetLineStyle(kDashed);
  line.SetLineWidth(3);
  line.SetLineColor(kBlue);
  
  TLine *l1 = NULL;

  if (what == 0)
    l1 = line.DrawLine(0, fgRefQEInner, x, fgRefQEInner);      
  else if (what == 1)
    l1 = line.DrawLine(0, fgRefConvFADC2PhotInner, x, fgRefConvFADC2PhotInner);      
  else if (what == 2)
    l1 = line.DrawLine(0, fgRefConvFADC2PheInner, x, fgRefConvFADC2PheInner );
  else if (what == 3)
    l1 = line.DrawLine(0, fgRefArrivalTimeInner, x, fgRefArrivalTimeInner );
  else if (what == 4)
    l1 = line.DrawLine(0, fgRefArrivalTimeRmsInner, x, fgRefArrivalTimeRmsInner );
  
  if (geom->InheritsFrom("MGeomCamMagic"))
    {
      const Double_t x2 = cam->GetNbinsX();

      switch (what)
        {
        case 0:
          line.DrawLine(x2, fgRefQEOuter, 398, fgRefQEOuter);      
          break;
        case 1:
          line.DrawLine(x2, fgRefConvFADC2PhotOuter, 398, fgRefConvFADC2PhotOuter );      
          break;
        case 2:
          line.DrawLine(x2, fgRefConvFADC2PheOuter, 398, fgRefConvFADC2PheOuter);
          break;
        case 3:
          line.DrawLine(x2, fgRefArrivalTimeOuter, 398, fgRefArrivalTimeOuter);
          break;
        case 4:
          line.DrawLine(x2, fgRefArrivalTimeRmsOuter, 398, fgRefArrivalTimeRmsOuter);
          break;
        }
    }
  
  TLegend *leg = new TLegend(what ? 0.2 : 0.6,0.75,what ? 0.5 : 0.9 ,0.85);
  leg->SetBit(kCanDelete);
  leg->AddEntry(l1, "Reference","l");
  leg->Draw();
}

void  MJCalibration::DisplayOutliers(TH1D *hist, const char* whatsmall, const char* whatbig) const
{

  const Int_t kNotDraw = 1<<9;
  TF1 *f = hist->GetFunction("gaus");
  f->ResetBit(kNotDraw);

  const Float_t mean  = f->GetParameter(1);
  const Float_t lolim = mean - 4.0*f->GetParameter(2);
  const Float_t uplim = mean + 4.0*f->GetParameter(2);
  const Stat_t  dead  = hist->Integral(0,hist->FindBin(lolim)-1);
  const Stat_t  noisy = hist->Integral(hist->FindBin(uplim)+1,hist->GetNbinsX()+1);

  const Double_t max = hist->GetBinContent(hist->GetMaximumBin());                                           
  
  const Double_t minl = hist->GetBinCenter(hist->GetXaxis()->GetFirst());
  const Double_t maxl = hist->GetBinCenter(hist->GetXaxis()->GetLast());

  TLatex deadtex;
  deadtex.SetTextSize(0.07);
  deadtex.DrawLatex(minl+0.015*(maxl-minl),max/1.1,
                    Form("%3i %s pixels",(Int_t)dead,whatsmall));

  TLatex noisytex;
  noisytex.SetTextSize(0.07);
  noisytex.DrawLatex(minl+0.015*(maxl-minl),max/1.2,
                     Form("%3i %s pixels",(Int_t)noisy,whatbig));

}

void MJCalibration::FixDataCheckHist(TH1D *hist) const 
{

    hist->SetDirectory(NULL);

    //
    // set the labels bigger
    //
    TAxis *xaxe = hist->GetXaxis();
    TAxis *yaxe = hist->GetYaxis();
    
    xaxe->CenterTitle();
    yaxe->CenterTitle();    
    xaxe->SetTitleSize(0.06);
    yaxe->SetTitleSize(0.06);    
    xaxe->SetTitleOffset(0.8);
    yaxe->SetTitleOffset(0.85);    
    xaxe->SetLabelSize(0.05);
    yaxe->SetLabelSize(0.05);    

}

const Int_t MJCalibration::CountBadPixels ( MHCamera *cam , const Int_t what ) const
{

  Int_t cnt = 0;

  for (UInt_t pix=0; pix<cam->GetNumPixels();pix++)
    if ((Int_t)cam->GetPixContent(pix) == what)
      cnt++;

  return cnt;
}

// --------------------------------------------------------------------------
//
// Retrieve the output file written by WriteResult()
// 
const char* MJCalibration::GetOutputFile() const
{
    if (fSequence.IsValid())
        return Form("%s/calib%08d.root", (const char*)fPathOut, fSequence.GetSequence());
    if (!fRuns)
        return "";

    return Form("%s/%s-F1.root", (const char*)fPathOut, (const char*)fRuns->GetRunsAsFileName());
}

// --------------------------------------------------------------------------
//
// MJCalibration allows to setup several option by a resource file:
//   MJCalibration.Display: full, datacheck, normal
//   MJCalibration.RelTimeCalibration: yes,no
//   MJCalibration.DataCheck: yes,no
//   MJCalibration.Debug: yes,no
//   MJCalibration.Intensity: yes,no
//   MJCalibration.UseBlindPixel: yes,no
//   MJCalibration.UsePINDiode: yes,no
//   MJCalibration.Geometry: MGeomCamMagic, MGeomCamECO1000
//
// For more details see the class description and the corresponding Getters
// 
Bool_t MJCalibration::CheckEnvLocal()
{
    TString dis = GetEnv("Display", "");
    if (dis.BeginsWith("Full", TString::kIgnoreCase))
        SetFullDisplay();
    if (dis.BeginsWith("DataCheck", TString::kIgnoreCase))
        SetDataCheckDisplay();
    if (dis.BeginsWith("Normal", TString::kIgnoreCase))
        SetNormalDisplay();

    SetRelTimeCalibration(GetEnv("RelTimeCalibration", IsRelTimes()));
    SetIntensity(GetEnv("IntensityCalibration", IsIntensity()));
    SetDataCheck(GetEnv("DataCheck", IsDataCheck()));
    SetDebug(GetEnv("Debug", IsDebug()));

    SetUseBlindPixel(GetEnv("UseBlindPixel", IsUseBlindPixel()));
    SetUsePINDiode(GetEnv("UsePINDiode", IsUsePINDiode()));
    SetGeometry(GetEnv("Geometry", fGeometry));

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Call the ProcessFile(MPedestalCam)
// 
Bool_t MJCalibration::Process(MPedestalCam &pedcam)
{
    if (!ReadCalibrationCam())
        return ProcessFile(pedcam);

    return kTRUE;
}

void MJCalibration::InitBlindPixel(MExtractBlindPixel &blindext,
                                   MHCalibrationChargeBlindCam &blindcam)
{

  Int_t run = fSequence.IsValid() ? fSequence.GetLastRun() : fRuns->GetRuns()[fRuns->GetNumRuns()-1];
  
  //
  // Initialize the blind pixel. Unfortunately, there is a hardware difference
  // in the first blind pixel until run "gkSecondBlindPixelInstallation" and the
  // later setup. The first needs to use a filter because of the length of
  // spurious NSB photon signals. The latter get better along extracting the amplitude
  // from a small window.
  //
  if (run < gkSecondBlindPixelInstallation)
    {
      MCalibrationBlindCamOneOldStyle blindresults;
      if (IsIntensity())
        blindresults.Copy(*fIntensBlindCam.GetCam());
      else
        blindresults.Copy(fCalibrationBlindCam);

      blindext.SetExtractionType(MExtractBlindPixel::kIntegral);
      blindext.SetExtractionType(MExtractBlindPixel::kFilter);
      blindext.SetRange(10,19,0,6);
      blindext.SetNSBFilterLimit(70);
    }
  else if (run < gkThirdBlindPixelInstallation)
    {

      MCalibrationBlindCamTwoNewStyle blindresults;

      if (IsIntensity())
        blindresults.Copy(*fIntensBlindCam.GetCam());
      else
        blindresults.Copy(fCalibrationBlindCam);

      blindext.SetNumBlindPixels(blindresults.GetSize());
      for (Int_t i=0;i<blindresults.GetSize();i++)
        blindext.SetBlindPixelIdx(blindresults[i].GetPixId(),i);

      blindext.SetExtractionType(MExtractBlindPixel::kAmplitude);
      blindext.SetExtractionType(MExtractBlindPixel::kFilter);
      blindext.SetRange(5,8,0,2);
      blindext.SetNSBFilterLimit(38);
      
    }
  else
    {

      MCalibrationBlindCamThreeNewStyle blindresults;

      if (IsIntensity())
        blindresults.Copy(*fIntensBlindCam.GetCam());
      else
        blindresults.Copy(fCalibrationBlindCam);

      blindext.SetNumBlindPixels(blindresults.GetSize());

      for (Int_t i=0;i<blindresults.GetSize();i++)
        blindext.SetBlindPixelIdx(blindresults[i].GetPixId(),i);

      blindext.SetExtractionType(MExtractBlindPixel::kAmplitude);
      blindext.SetExtractionType(MExtractBlindPixel::kFilter);
      blindext.SetDataType(MExtractBlindPixel::kRawEvt2);
      blindext.SetRange(5,8,0,2);
      blindext.SetNSBFilterLimit(38);
      
    }
  
}

// --------------------------------------------------------------------------
//
// Execute the task list and the eventloop:
//
// - Check if there are fRuns, otherwise return
// - Check the colour of the files in fRuns (FindColor()), otherwise return
// - Check for consistency between run numbers and number of files
// - Add fRuns to MReadMarsFile
// - Put into MParList:
//   1) MPedestalCam          (pedcam)
//   2) MCalibrationQECam     (fQECam)
//   3) MCalibrationChargeCam (fCalibrationCam)
//   4) MCalibrationRelTimeCam (fRelTimeCam)   (only if flag IsRelTimes() is chosen)
//   5) MBadPixelsCam         (fBadPixels)
//   6) MCalibrationChargePINDiode
//   7) MCalibrationBlindPix
// - Put into the MTaskList:
//   1)  MReadMarsFile
//   2)  MBadPixelsMerge
//   3)  MGeomApply
//   4)  MExtractor
//   5)  MExtractPINDiode
//   6)  MExtractBlindPixel
//   7)  MExtractTime (only if flag IsRelTimes() is chosen)
//   8)  MContinue(MFCosmics)
//   9)  MFillH("MHCalibrationChargePINDiode", "MExtractedSignalPINDiode",   "FillPINDiode")
//   10) MFillH("MHCalibrationChargeBlindCam", "MExtractedSignalBlindPixel", "FillBlindCam")
//   11) MFillH("MHCalibrationChargeCam",      "MExtractedSignalCam",        "FillChargeCam")
//   12) MFillH("MHCalibrationChargeCam",      "MExtractedSignalCam",        "FillRelTime")
//   13) MCalibrationChargeCalc
//   14) MFillH("MHCalibrationRelTimeCam",     "MArrivalTimeCam") (only if flag IsRelTimes() is chosen)
//   15) MCalibrationRelTimeCalc
// - Execute MEvtLoop
// - DisplayResult()
// - WriteResult()
//
Bool_t MJCalibration::ProcessFile(MPedestalCam &pedcam)
{
    if (!fSequence.IsValid())
    {
        if (!fRuns)
        {
            *fLog << err << "No Runs choosen... abort." << endl;
            return kFALSE;
        }

        if (fRuns->GetNumRuns() != fRuns->GetNumEntries())
        {
            *fLog << err << "Number of files found doesn't match number of runs... abort."
                << fRuns->GetNumRuns() << " vs. " << fRuns->GetNumEntries() << endl;
            return kFALSE;
        }
    }

    //if (!CheckEnv())
    //    return kFALSE;

    CheckEnv();

    // --------------------------------------------------------------------------------

    *fLog << inf;
    fLog->Separator(GetDescriptor());

    *fLog << "Calculate MCalibrationCam from ";
    if (fSequence.IsValid())
        *fLog << "Sequence #" << fSequence.GetSequence() << endl;
    else
        *fLog << "Runs " << fRuns->GetRunsAsString() << endl;
    *fLog << endl;

    // --------------------------------------------------------------------------------
    
    // Setup Tasklist
    MParList plist;
    MTaskList tlist;
    plist.AddToList(&tlist);
    plist.AddToList(this); // take care of fDisplay!

    MDirIter iter;
    if (fSequence.IsValid())
    {
        const Int_t n0 = fSequence.SetupCalRuns(iter, fPathData, "C", IsDataCheck());
        const Int_t n1 = fSequence.GetNumCalRuns();
        if (n0==0)
        {
            *fLog << err << "ERROR - No input files of sequence found!" << endl;
            return kFALSE;
        }
        if (n0!=n1)
        {
            *fLog << err << "ERROR - Number of files found (" 
                  << n0 << ") doesn't match number of files in sequence (" 
                  << n1 << ")" << endl;
            return kFALSE;
        }
    }

    //
    // Input containers
    //
    pedcam.SetName("MPedestalCam");
    plist.AddToList(&pedcam);
    plist.AddToList(&fBadPixels);
        
    //
    // Calibration Results containers
    //
    if (IsIntensity())
    {
        plist.AddToList(&fIntensQECam);
        plist.AddToList(&fIntensCalibCam);
        plist.AddToList(&fIntensBlindCam);
        //        plist.AddToList(&fIntensCalibrationPINDiode);
        plist.AddToList(&fIntensRelTimeCam);
        plist.AddToList(&fIntensBadCam);
    }
    else
    {
        plist.AddToList(&fQECam);
        plist.AddToList(&fCalibrationCam);
        plist.AddToList(&fCalibrationBlindCam);
        plist.AddToList(&fCalibrationPINDiode);
        plist.AddToList(&fRelTimeCam);
    }

    //
    // Initialize two histogram containers which could be modified in this class
    //
    MHCalibrationRelTimeCam     reltimecam;
    MHCalibrationChargeCam      chargecam;
    MHCalibrationChargeBlindCam blindcam;
    plist.AddToList(&chargecam);
    plist.AddToList(&blindcam);
    if (IsRelTimes())
        plist.AddToList(&reltimecam);
    //
    // Data Reading tasks
    //
    MReadMarsFile read("Events");
    MRawFileRead rawread(NULL);

    if (IsDataCheck())
    {
        rawread.AddFiles(fSequence.IsValid() ? iter : *fRuns);
        tlist.AddToList(&rawread);
    }
    else
    {
        read.DisableAutoScheme();
        read.AddFiles(fSequence.IsValid() ? iter : *fRuns);
        tlist.AddToList(&read);
    }

    //
    // Other Tasks
    //
    MGeomApply               apply;
    apply.SetGeometry(fGeometry);
    MBadPixelsMerge          merge(&fBadPixels);
    MExtractPINDiode         pinext;
    MExtractBlindPixel       blindext;

    InitBlindPixel(blindext, blindcam);

    MExtractSlidingWindow    extract2;
    MExtractTimeFastSpline   timespline;
    MCalibrationChargeCalc   calcalc;
    MCalibrationRelTimeCalc  timecalc;
    calcalc.SetOutputFile("");
    timecalc.SetOutputFile("");

    if (!fSequence.IsValid())
    {
        calcalc.SetOutputPath(fPathOut);
        calcalc.SetOutputFile(Form("%s-ChargeCalibStat.txt",(const char*)fRuns->GetRunsAsFileName()));
        timecalc.SetOutputPath(fPathOut);
        timecalc.SetOutputFile(Form("%s-ChargeCalibStat.txt",(const char*)fRuns->GetRunsAsFileName()));
    }

    if (IsDebug())
    {
        chargecam.SetDebug();
        calcalc.SetDebug();
    }

    //
    // Calibration histogramming
    //
    MFillH fillpin("MHCalibrationChargePINDiode", "MExtractedSignalPINDiode",   "FillPINDiode");
    MFillH fillbnd("MHCalibrationChargeBlindCam", "MExtractedSignalBlindPixel", "FillBlindCam");
    MFillH fillcam("MHCalibrationChargeCam",      "MExtractedSignalCam",        "FillChargeCam");
    MFillH filltme("MHCalibrationRelTimeCam",     "MArrivalTimeCam",            "FillRelTime");
    fillpin.SetBit(MFillH::kDoNotDisplay);
    fillbnd.SetBit(MFillH::kDoNotDisplay);
    fillcam.SetBit(MFillH::kDoNotDisplay);
    filltme.SetBit(MFillH::kDoNotDisplay);

    //
    // Set default extractors in case, none has been set...
    //
    if (!fExtractor)
      fExtractor = &extract2;
    if (!fTimeExtractor)
      fTimeExtractor = &timespline;

    //
    // Look if the extractor is a pure charge or also a time extractor
    //
    if (fExtractor->InheritsFrom("MExtractTimeAndCharge"))
    {
        if (fExtractorCam.GetSize() == pedcam.GetSize())
            calcalc.SetPedestals(&fExtractorCam);
        else
        {
            *fLog << err << GetDescriptor() << "ERROR - ";
            *fLog << "Used Extractor derives from MExtractTimeAndCharge, " << endl;
            *fLog << "but MExtractorCam size " << fExtractorCam.GetSize() << " ";
            *fLog << "mismatch pedcam size " << pedcam.GetSize() << "! " << endl;
            return kFALSE;
        }
    }

    //
    // Setup more tasks and tasklist
    //
    MTaskEnv taskenv("ExtractSignal");
    taskenv.SetDefault(fExtractor);

    tlist.AddToList(&merge);
    tlist.AddToList(&apply);

    MPedCalcPedRun           pedcalc;
    pedcalc.SetExtractWindow(fExtractor->GetHiGainFirst(),TMath::Nint(fExtractor->GetNumHiGainSamples()));

    if (IsIntensity())
      tlist.AddToList(&pedcalc);

    MCalibColorSet colorset;
    tlist.AddToList(&colorset);

    tlist.AddToList(&taskenv);
    tlist.AddToList(&pinext);
    tlist.AddToList(&blindext);

    MTaskEnv taskenv2("ExtractTime");
    if (!fExtractor->InheritsFrom("MExtractTimeAndCharge"))
    {
        taskenv2.SetDefault(fTimeExtractor);
   
        if (IsRelTimes())
            tlist.AddToList(&taskenv2);
    }
    
    //
    // Apply a filter against cosmics
    // (will have to be needed in the future
    // when the calibration hardware-trigger is working)
    //
    MFCosmics cosmics;
    MContinue cont(&cosmics);

    if (fColor == MCalibrationCam::kCT1)
      tlist.AddToList(&cont);

    MCalibColorSteer steer;
    if (IsIntensity())
      tlist.AddToList(&steer);

    tlist.AddToList(&fillcam);

    if (IsRelTimes())
    {
        tlist.AddToList(&filltme);
        tlist.AddToList(&timecalc);
    }

    if (IsUseBlindPixel())
      tlist.AddToList(&fillbnd);
    if (IsUsePINDiode())
      tlist.AddToList(&fillpin);

    tlist.AddToList(&calcalc);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    if (!taskenv.GetTask() && !taskenv2.GetTask())
    {
        *fLog << err << "ERROR - Neither ExtractSignal nor ExtractTime initialized or both '<dummy>'." << endl;
        return kFALSE;
    }

    if (!WriteTasks(taskenv.GetTask(), IsRelTimes() ? taskenv2.GetTask() : 0))
        return kFALSE;

    // Execute first analysis
    if (!evtloop.Eventloop())
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    tlist.PrintStatistics();

    if (!fCalibrationPINDiode.IsValid())
        SetUsePINDiode(kFALSE);

    DisplayResult(plist);

    if (!IsNoStorage())
      if (!WriteResult(plist.FindObject("MGeomCam")))
        return kFALSE;

    if (IsHistsStorage())
      {
        if (!WriteHists(plist))
          return kFALSE;
      }
    
    *fLog << all << GetDescriptor() << ": Done." << endl;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Read the following containers from GetOutputFile()
// - MCalibrationChargeCam
// - MCalibrationQECam
// - MBadPixelsCam
//
Bool_t MJCalibration::ReadCalibrationCam()
{
  
  if (IsNoStorage())
    return kFALSE;

  const TString fname = GetOutputFile();
  
  if (gSystem->AccessPathName(fname, kFileExists))
  {
      *fLog << err << "Input file " << fname << " doesn't exist." << endl;
      return kFALSE;
  }

  *fLog << inf << "Reading from file: " << fname << endl;

  TFile file(fname, "READ");
  if (fCalibrationCam.Read()<=0)
  {
      *fLog << err << "Unable to read MCalibrationChargeCam from " << fname << endl;
      return kFALSE;
  }

  if (fQECam.Read()<=0)
  {
      *fLog << err << "Unable to read MCalibrationQECam from " << fname << endl;
      return kFALSE;
  }


  if (file.FindKey("MCalibrationRelTimeCam"))
      if (fRelTimeCam.Read()<=0)
      {
          *fLog << err << "Unable to read MCalibrationRelTimeCam from " << fname << endl;
          return kFALSE;
      }

  if (file.FindKey("MBadPixelsCam"))
  {
      MBadPixelsCam bad;
      if (bad.Read()<=0)
      {
          *fLog << err << "Unable to read MBadPixelsCam from " << fname << endl;
          return kFALSE;
      }
      fBadPixels.Merge(bad);
  }

  if (fDisplay /*&& !fDisplay->GetCanvas("Pedestals")*/) // FIXME!
      fDisplay->Read();

  return kTRUE;
}


// --------------------------------------------------------------------------
//
// Set the useage of the Blind Pixel device 
// 
void MJCalibration::SetUseBlindPixel(const Bool_t b)
{
  b ? SETBIT(fDevices,kUseBlindPixel) : CLRBIT(fDevices,kUseBlindPixel);
}

// --------------------------------------------------------------------------
//
// Set the useage of the PIN Diode device 
// 
void MJCalibration::SetUsePINDiode(const Bool_t b)
{
  b ? SETBIT(fDevices,kUsePINDiode) : CLRBIT(fDevices,kUsePINDiode);
}

Bool_t MJCalibration::WriteEventloop(MEvtLoop &evtloop) const
{

  if (IsNoStorage())
    return kTRUE;

  if (fPathOut.IsNull())
    return kTRUE;
  
  const TString oname(GetOutputFile());
  
  *fLog << inf << "Writing to file: " << oname << endl;
  
  TFile file(oname, fOverwrite?"RECREATE":"NEW", "File created by MJCalibration", 9);
  if (!file.IsOpen())
    {
      *fLog << err << "ERROR - Couldn't open file " << oname << " for writing..." << endl;
      return kFALSE;
    }
  
  if (evtloop.Write(fName)<=0)
    {
      *fLog << err << "Unable to write MEvtloop to " << oname << endl;
        return kFALSE;
    }
  
  return kTRUE;
}

Bool_t MJCalibration::WriteTasks(MTask *t1, MTask *t2) const
{
    if (IsNoStorage())
        return kTRUE;

    if (fPathOut.IsNull())
        return kTRUE;

    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, fOverwrite?"RECREATE":"NEW", "File created by MJCalibration", 9);
    if (!file.IsOpen())
    {
        *fLog << err << "ERROR - Couldn't open file " << oname << " for writing..." << endl;
        return kFALSE;
    }

    if (t1)
    {
        *fLog << inf << " - Writing Task " << t1->GetName() << " [" << t1->ClassName() << "]..." << flush;
        if (t1->Write()<=0)
        {
            *fLog << err << "Unable to write " << t1->GetName() << " to " << oname << endl;
            return kFALSE;
        }
        *fLog << "ok." << endl;
    }

    if (t2)
    {
        *fLog << inf << " - Writing Task " << t2->GetName() << " [" << t2->ClassName() << "]..." << flush;
        if (t2->Write()<=0)
        {
            *fLog << err << "Unable to write " << t2->GetName() << " to " << oname << endl;
            return kFALSE;
        }
        *fLog << "ok." << endl;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Write the result into the output file GetOutputFile(), if fOutputPath exists.
// 
// The following containers are written:
// - MStatusDisplay
// - MCalibrationChargeCam  or MCalibrationIntensityChargeCam 
// - MCalibrationBlindCam   or MCalibrationIntensityBlindCam
// - MCalibrationQECam      or MCalibrationIntensityQECam     
// - MCalibrationChargePINDiode
// - MBadPixelsCam
// If the flag kRelTimes is set, then also:
// - MCalibrationRelTimeCam or MCalibrationIntensityRelTimeCam     
//
Bool_t MJCalibration::WriteResult(TObject *geom)
{
    if (fPathOut.IsNull())
        return kTRUE;

    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "UPDATE", "File created by MJCalibration", 9);
    if (!file.IsOpen())
    {
        *fLog << err << "ERROR - Couldn't open file " << oname << " for writing..." << endl;
        return kFALSE;
    }

    TObjArray cont;
    if (fDisplay)
        cont.Add(fDisplay);

    if (IsIntensity())
    {
        cont.Add(&fIntensBadCam);
        cont.Add(&fIntensCalibCam);
        cont.Add(&fIntensQECam);
        cont.Add(&fIntensBlindCam);
    }
    else
    {
        cont.Add(&fBadPixels);
        cont.Add(&fCalibrationCam);
        cont.Add(&fQECam);
        cont.Add(&fCalibrationBlindCam);
    }
    cont.Add(&fCalibrationPINDiode);

    //if (IsRelTimes())
    cont.Add(IsIntensity() ? (TObject*)&fIntensRelTimeCam : (TObject*)&fRelTimeCam);

    if (!geom)
        *fLog << warn << " - WARNING - MGeomCam... not found!" << endl;
    else
        cont.Add(geom);

    return WriteContainer(cont);
}

// --------------------------------------------------------------------------
//
// Write the calibration histograms into the output file GetOutputFile(), 
// if fOutputPath exists.
// 
// The following containers are written:
// - MHCalibrationChargeCam
// - MHCalibrationChargeBlindCam
// - MHCalibrationChargePINDiode
// If the flag kRelTimes is set, then also:
// - MHCalibrationRelTimeBlindCam
//
Bool_t MJCalibration::WriteHists(MParList &plist)
{
    if (fPathOut.IsNull())
        return kTRUE;

    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;

    TFile file(oname, "UPDATE", "File created by MJCalibration", 9);
    if (!file.IsOpen())
    {
        *fLog << err << "ERROR - Couldn't open file " << oname << " for writing..." << endl;
        return kFALSE;
    }

    TObjArray cont;
    cont.Add(plist.FindObject("MHCalibrationChargeCam"));
    cont.Add(plist.FindObject("MHCalibrationChargeBlindCam"));
    cont.Add(plist.FindObject("MHCalibrationChargePINDiode"));
    if (IsRelTimes())
        cont.Add(plist.FindObject("MHCalibrationRelTimeCam"));

    return WriteContainer(cont);
}

void MJCalibration::DisplayDoubleProject(MHCamera *cam, const char* whatsmall, const char* whatbig) const
{
  
  TArrayI inner(1);
  inner[0] = 0;
  
  TArrayI outer(1);
  outer[0] = 1;
          
  TArrayI s1(3);
  s1[0] = 6;
  s1[1] = 1;
  s1[2] = 2;
  
  TArrayI s2(3);
  s2[0] = 3;
  s2[1] = 4;
  s2[2] = 5;
  
  TVirtualPad *pad = gPad;
  pad->Divide(2,1);
  
  TH1D *inout[2];

  for (int i=0; i<2; i++)
    {
      pad->cd(i+1);
      gPad->SetBorderMode(0);
      gPad->SetTicks();
      
      inout[i] = cam->ProjectionS(TArrayI(), TArrayI(1,&i), i==0 ? "Inner" : "Outer");
      FixDataCheckHist(inout[i]);
      inout[i]->SetTitle(Form("%s %s",cam->GetTitle(),i==0 ? "Inner" : "Outer"));
      inout[i]->SetDirectory(NULL);
      inout[i]->SetLineColor(kRed+i);
      inout[i]->SetBit(kCanDelete);
      inout[i]->Draw();
      //
      // Display the outliers as dead and noisy pixels
      //
      if (!inout[i]->Fit("gaus","0Q"))
        DisplayOutliers(inout[i],whatsmall,whatbig);

      gPad->Modified();
      gPad->Update();      
      TPaveStats *st = (TPaveStats*)inout[i]->GetListOfFunctions()->FindObject("stats");
      st->SetY1NDC(0.6);
      st->SetY2NDC(0.9);
      st->SetX1NDC(0.55);
      st->SetX2NDC(0.99);
      gPad->Modified();
      gPad->Update();      

      TLegend *leg2 = new TLegend(0.55,0.4,0.99,0.6);
      
      //
      // Display the two half of the camera separately
      //
      TH1D *half[2];
      half[0] = cam->ProjectionS(s1, TArrayI(1,&i), "Sector 6-1-2");
      half[1] = cam->ProjectionS(s2, TArrayI(1,&i), "Sector 3-4-5");
      
      for (int j=0; j<2; j++)
        {
          half[j]->SetLineColor(kRed+i+2*j+1);
          half[j]->SetDirectory(NULL);
          half[j]->SetBit(kCanDelete);
          half[j]->Draw("same");
          leg2->AddEntry(half[j], half[j]->GetName(), "l");
        }
      leg2->Draw();
    }
}
