/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Nicola Galante  12/2004 <mailto:nicola.galante@pi.infn.it>
!   Author(s): Thomas Bretz 12/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2004
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//   MCalibrationPatternDecode
//
//  Decodes the trigger pattern from MRawEvtData into MCalibrationPattern.
//
// For files before file version 5 the trigger pattern is set to 00000000.
// This can be changed using the information about the file-type.
//
// Input:
//   MRawEvtData
//
// Output:
//   MCalibrationPattern
//
/////////////////////////////////////////////////////////////////////////////
#include "MCalibrationPatternDecode.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MRawEvtHeader.h"
#include "MRawRunHeader.h"
#include "MCalibrationPattern.h"

ClassImp(MCalibrationPatternDecode);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor
//
MCalibrationPatternDecode::MCalibrationPatternDecode(const char *name, const char *title)
    : fRunHeader(0), fEvtHeader(0), fPattern(0)
{
    fName  = name  ? name  : "MCalibrationPatternDecode";
    fTitle = title ? title : "Task to decode Trigger Pattern";
}

// --------------------------------------------------------------------------
//
Int_t MCalibrationPatternDecode::PreProcess(MParList *pList)
{
    fRunHeader = (MRawRunHeader*)pList->FindCreateObj("MRawRunHeader");
    if (!fRunHeader)
    {
	*fLog << err << "MRawRunHeader not found... abort." << endl;
	return kFALSE;
    }

    fEvtHeader = (MRawEvtHeader *)pList->FindObject("MRawEvtHeader");
    if (!fEvtHeader)
    {
	*fLog << err << "MRawEvtHeader not found... abort." << endl;
	return kFALSE;
    }

    fPattern = (MCalibrationPattern*)pList->FindCreateObj("MCalibrationPattern");
    if (!fPattern)
    {
	*fLog << err << "MCalibratinPattern not found... abort." << endl;
	return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// For files before file version 5 the trigger pattern is set to 00000000.
// This can be changed using the information about the file-type.
//
Int_t MCalibrationPatternDecode::Process()
{
    // No setting necessary because MCalibrationPattern::reset() has
    // been called just before
    if (fRunHeader->GetFormatVersion()<5)
      return kTRUE;

    UInt_t pattern = fEvtHeader->GetCalibrationPattern();

    fPattern->fCLStrength = pattern & 0xff;
    fPattern->fCLColor    = (MCalibrationPattern::CLColor_t)((pattern>>8)&0xf);
    
    const UInt_t pulserpattern = (pattern >> 16) & 0x1ffff;

    fPattern->fPulserStrength =  MCalibrationCam::kNONE;    

    if ((pulserpattern & kGreenAndBlue) || (pulserpattern & kBlueAndUV) || (pulserpattern & kGreenAndUV))
      fPattern->fPulserColor =  MCalibrationCam::kNONE;
    if (pulserpattern & kCT1Pulser)
      fPattern->fPulserColor =  MCalibrationCam::kCT1;
    if (pulserpattern & kAnyUV)
      fPattern->fPulserColor =  MCalibrationCam::kUV;
    if (pulserpattern & kAnyGreen)
      fPattern->fPulserColor =  MCalibrationCam::kGREEN;
    if (pulserpattern & kAnyBlue)
      fPattern->fPulserColor =  MCalibrationCam::kBLUE;

    Float_t strength = 0.;

    switch (fPattern->fPulserColor)
      {
      case MCalibrationCam::kNONE:
	break;
      case MCalibrationCam::kGREEN:
        if (pattern & kSlot1Green)
          strength += 5.;
        if (pattern & kSlot2Green)
          strength += 2.;
        if (pattern & kSlot15Green)
          strength += 1.;
        if (pattern & kSlot16AttGreen)
          strength += 0.2;
	break;      
      case MCalibrationCam::kBLUE:
        if (pattern & kSlot3Blue)
          strength += 5.;
        if (pattern & kSlot6Blue)
          strength += 5.;
        if (pattern & kSlot7Blue)
          strength += 5.;
        if (pattern & kSlot8Blue)
          strength += 2.;
        if (pattern & kSlot9AttBlue)
          strength += 0.2;
        if (pattern & kSlot10Blue)
          strength += 0.;
        if (pattern & kSlot11Blue)
          strength += 1.;
        if (pattern & kSlot14Blue)
          strength += 5.;
	break;      
      case MCalibrationCam::kUV:
        if (pattern & kSlot4UV)
          strength += 1.;
        if (pattern & kSlot5UV)
          strength += 2.;
        if (pattern & kSlot12UV)
          strength += 5.;
        if (pattern & kSlot13UV)
          strength += 5.;
      break;      
      case MCalibrationCam::kCT1:
	strength = 20.;
	break;      
      }
  
    fPattern->fPulserStrength = strength;

    return kTRUE;
}
