/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Markus Gaug, 04/2004 <mailto:markus@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2005
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibTest
//
// If the flag SetDataCheckDisplay() is set, only the most important distributions
//  are displayed. 
// Otherwise, (default: SetNormalDisplay()), a good selection of plots is given
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibTest.h"

#include <TFile.h>
#include <TStyle.h>
#include <TCanvas.h>
#include <TSystem.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MTaskEnv.h"
#include "MEvtLoop.h"

#include "MHCamera.h"

#include "MPedestalCam.h"
#include "MBadPixelsCam.h"
#include "MBadPixelsTreat.h"
#include "MBadPixelsCalc.h"
#include "MBadPixelsMerge.h"
#include "MCerPhotEvt.h"
#include "MArrivalTime.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationQECam.h"
#include "MCalibrationTestCam.h"
#include "MCalibrationTestCalc.h"
#include "MHCamEvent.h"
#include "MHCalibrationTestCam.h"

#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MGeomApply.h"
#include "MGeomCam.h"
#include "MExtractTimeAndChargeSlidingWindow.h"
#include "MExtractor.h"
#include "MExtractTime.h"
#include "MExtractTimeFastSpline.h"
#include "MFCosmics.h"
#include "MContinue.h"
#include "MFillH.h"
#include "MCalibrateData.h"
#include "MCalibrateRelTimes.h"

#include "MTriggerPattern.h"
#include "MTriggerPatternDecode.h"
#include "MFTriggerPattern.h"

#include "MStatusDisplay.h"

ClassImp(MJCalibTest);

using namespace std;
// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets fUseCosmicsFilter to kTRUE, fRuns to 0, fExtractor to NULL, fTimeExtractor to NULL
// fDisplay to kNormalDisplay
//
MJCalibTest::MJCalibTest(const char *name, const char *title) 
    : fUseCosmicsFilter(kTRUE), fExtractor(NULL), fTimeExtractor(NULL),
      fDisplayType(kNormalDisplay), fGeometry("MGeomCamMagic")
{
    fName  = name  ? name  : "MJCalibTest";
    fTitle = title ? title : "Tool to extract, calibrate and test signals from a file";
}


void MJCalibTest::DisplayResult(MParList &plist)
{
  if (!fDisplay)
    return;
  
  //
  // Update display
  //
  TString title = fDisplay->GetTitle();
  title += "--  Extraction-Calibration-Test ";
  title += fRuns->GetRunsAsString();
  title += "  --";
  fDisplay->SetTitle(title);

  //
  // Get container from list
  //
  MGeomCam  &geomcam = *(MGeomCam*) plist.FindObject("MGeomCam");
  MHCalibrationTestCam &testcam = *(MHCalibrationTestCam*)plist.FindObject("MHCalibrationTestCam");
  
  // Create histograms to display
  MHCamera disp1 (geomcam, "Test;Photons",           "Mean of calibrated Photons");
  MHCamera disp2 (geomcam, "Test;SigmaPhotons",      "Sigma of calibrated photons");
  MHCamera disp3 (geomcam, "Test;PhotonsPerArea",    "Equiv. Cherenkov Photons per Area");
  MHCamera disp4 (geomcam, "Test;SigmaPhotPerArea",  "Sigma equiv. Cher. Photons per Area");
  MHCamera disp5 (geomcam, "Test;Phot",              "Calibrated Photons");
  MHCamera disp6 (geomcam, "Test;PhotPerArea",       "Calibrated Photons per Area");
  MHCamera disp7 (geomcam, "Test;NotInterpolate",    "Not interpolated pixels");
  MHCamera disp8 (geomcam, "Test;DeviatingPhots",    "Deviating Number Photons");

  // Fitted charge means and sigmas
  disp1.SetCamContent(testcam,  0);
  disp1.SetCamError(  testcam,  1);
  disp2.SetCamContent(testcam,  2);
  disp2.SetCamError(  testcam,  3);
  disp3.SetCamContent(testcam,  7);
  disp3.SetCamError(  testcam,  8);
  disp4.SetCamContent(testcam,  9);
  disp4.SetCamError(  testcam,  10);

  disp5.SetCamContent(fTestCam,  0);
  disp5.SetCamError(  fTestCam,  1);
  disp6.SetCamContent(fTestCam,  2);
  disp6.SetCamError(  fTestCam,  3);
  disp7.SetCamError(  fTestCam,  4);

  disp8.SetCamError(  fBadPixels, 22);


  disp1.SetYTitle("Photons");
  disp2.SetYTitle("\\sigma_{phot}");
  disp3.SetYTitle("Photons per Area [mm^{-2}]");
  disp4.SetYTitle("\\sigma_{phot} per Area [mm^{-2}]");

  disp5.SetYTitle("Photons");
  disp6.SetYTitle("Photons per Area [mm^{-2}]");
  disp7.SetYTitle("[1]");
  disp8.SetYTitle("[1]");
  
  gStyle->SetOptStat(1111);
  gStyle->SetOptFit();

  if (fDisplayType == kNormalDisplay)
    {

      TCanvas &c = fDisplay->AddTab("TestCharges");
      c.Divide(4,4);
      
      disp1.CamDraw(c, 1, 4, 2, 1);
      disp2.CamDraw(c, 2, 4, 2, 1);        
      disp3.CamDraw(c, 3, 4, 1, 1);        
      disp4.CamDraw(c, 4, 4, 2, 1);        
    }
  
  TCanvas &c2 = fDisplay->AddTab("TestResult");
  c2.Divide(2,4);

  disp5.CamDraw(c2, 1, 2, 2, 1);
  disp6.CamDraw(c2, 2, 2, 2, 1);        

  TCanvas &c3 = fDisplay->AddTab("TestDefects");
  c3.Divide(2,2);

  disp7.CamDraw(c3, 1, 2, 0);
  disp8.CamDraw(c3, 2, 2, 0);        

  return;

}


void MJCalibTest::DisplayResultT(MParList &plist)
{
  if (!fDisplay)
    return;
  
  //
  // Update display
  //
  TString title = fDisplay->GetTitle();
  title += "--  Extraction-Calibration-Test-Time";
  title += fRuns->GetRunsAsString();
  title += "  --";
  fDisplay->SetTitle(title);

  //
  // Get container from list
  //
  MGeomCam &geomcam = *(MGeomCam*)plist.FindObject("MGeomCam");
  
  // Create histograms to display
  MHCamera disp1 (geomcam, "Test;Arr.Times",           "Mean of calibrated Arr.Times");
  MHCamera disp2 (geomcam, "Test;SigmaArr.Times",      "Sigma of calibrated Arr.Times");

  // Fitted charge means and sigmas
  disp1.SetCamContent(fTestTimeCam,  0);
  disp1.SetCamError(  fTestTimeCam,  1);
  disp2.SetCamContent(fTestTimeCam,  2);
  disp2.SetCamError(  fTestTimeCam,  3);

  disp1.SetYTitle("Mean Arr.Times [FADC units]");
  disp2.SetYTitle("\\sigma_{t} [FADC units]");
  
  gStyle->SetOptStat(1111);
  gStyle->SetOptFit();

  TCanvas &c = fDisplay->AddTab("TestTimes");
  c.Divide(2,4);

  disp1.CamDraw(c, 1, 2,  5, 1);
  disp2.CamDraw(c, 2, 2,  5, 1);        

  return;

}


const char* MJCalibTest::GetInputFile() const
{

  if (fSequence.IsValid())
    return Form("%s/calib%08d.root", (const char*)fPathOut, fSequence.GetSequence());
  
  if (!fRuns)
    return "";
  
  return Form("%s/%s-F1.root", (const char*)fPathOut, (const char*)fRuns->GetRunsAsFileName());
  
}


const char* MJCalibTest::GetOutputFile() const
{

  if (fSequence.IsValid())
    return Form("%s/test%08d.root", (const char*)fPathOut, fSequence.GetSequence());
  
  if (!fRuns)
    return "";
  
  return Form("%s/%s-Test.root", (const char*)fPathOut, (const char*)fRuns->GetRunsAsFileName());
  
}

Bool_t MJCalibTest::ReadCalibration(TObjArray &l, MBadPixelsCam &cam, MExtractor* &ext1, MExtractor* &ext2, TString &geom) const
{

  const TString fname = GetInputFile();
  
  *fLog << inf << "Reading from file: " << fname << endl;

  TFile file(fname, "READ");
  if (!file.IsOpen())
    {
      *fLog << err << dbginf << "ERROR - Could not open file " << fname << endl;
      return kFALSE;
    }
  
  TObject *o = file.Get("ExtractSignal");
  if (o && !o->InheritsFrom(MExtractor::Class()))
    {
      *fLog << err << dbginf << "ERROR - ExtractSignal read from " << fname << " doesn't inherit from MExtractor!" << endl;
      return kFALSE;
    }
  ext1 = o ? (MExtractor*)o->Clone() : NULL;
  
  o = file.Get("ExtractTime");
  if (o && !o->InheritsFrom(MExtractor::Class()))
    {
      *fLog << err << dbginf << "ERROR - ExtractTime read from " << fname << " doesn't inherit from MExtractor!" << endl;
      return kFALSE;
    }
  ext2 = o ? (MExtractor*)o->Clone() : NULL;
  if (!ext1 && !ext2)
    {
      *fLog << err << dbginf << "ERROR - Neither ExtractSignal nor ExrtractTime found in " << fname << "!" << endl;
      return kFALSE;
    }
  
  o = file.Get("MGeomCam");
  if (o && !o->InheritsFrom(MGeomCam::Class()))
    {
      *fLog << err << dbginf << "ERROR - MGeomCam read from " << fname << " doesn't inherit from MGeomCam!" << endl;
      return kFALSE;
    }
  geom = o ? o->ClassName() : "";
  
  TObjArray cont(l);
  cont.Add(&cam);
  return ReadContainer(cont);
}

// --------------------------------------------------------------------------
//
// MJCalibration allows to setup several option by a resource file:
//   MJCalibrateSignal.RawData: yes,no
//
// For more details see the class description and the corresponding Getters
//
Bool_t MJCalibTest::CheckEnvLocal()
{

  SetUseRootData();
  
  if (HasEnv("DataType"))
    {
      TString dat = GetEnv("DataType", "");
      if (dat.BeginsWith("raw", TString::kIgnoreCase))
        {
          fDataFlag = 0;
          SetUseRawData();
        }
      if (dat.BeginsWith("root", TString::kIgnoreCase))
        {
          fDataFlag = 0;
          SetUseRootData();
      }
      if (dat.BeginsWith("mc", TString::kIgnoreCase))
        {
          fDataFlag = 0;
          SetUseMC();
        }
    }
  return kTRUE;
}

Bool_t MJCalibTest::Process(MPedestalCam &pedcam)
{
    // const TString fname = GetOutputFile();
  
//  if (gSystem->AccessPathName(fname, kFileExists))
    return ProcessFile(pedcam);
  
    // return kTRUE;
}

Bool_t MJCalibTest::ProcessFile(MPedestalCam &pedcam)
{
  

  if (!fSequence.IsValid())
    {
      if (!fRuns)
        {
            *fLog << err << "ERROR - Sequence invalid and no runs chosen!" << endl;
            return kFALSE;
	}
      
      if (fRuns->GetNumRuns() != fRuns->GetNumEntries())
        {
          *fLog << err << "Number of files found doesn't match number of runs... abort."
                << fRuns->GetNumRuns() << " vs. " << fRuns->GetNumEntries() << endl;
          return kFALSE;
        }
      *fLog << "Calibrate data from ";
      *fLog << "Runs " << fRuns->GetRunsAsString() << endl;
      *fLog << endl;
    }
  
  CheckEnv();

  *fLog << inf;
  fLog->Separator(GetDescriptor());
  *fLog << "Calculate MExtractedSignalCam from Runs " << fRuns->GetRunsAsString() << endl;
  *fLog << endl;

  MDirIter iter;

  if (fSequence.IsValid())
    {
      const Int_t n0 = fSequence.SetupDatRuns(iter, fPathData, "D", IsUseRawData());
      const Int_t n1 = fSequence.GetNumDatRuns();
      if (n0==0)
        {
          *fLog << err << "ERROR - No input files of sequence found!" << endl;
          return kFALSE;
        }
      if (n0!=n1)
        {
          *fLog << err << "ERROR - Number of files found (" << n0 << ") doesn't match number of files in sequence (" << n1 << ")" << endl;
          return kFALSE;
        }
    }
  
  MCalibrationChargeCam      calcam;
  MCalibrationQECam          qecam;
  MCalibrationRelTimeCam     tmcam;
  MBadPixelsCam              badpix;
  
  TObjArray calibcont;
  calibcont.Add(&calcam);
  calibcont.Add(&qecam);
  calibcont.Add(&tmcam);
    
  MExtractor *extractor1=0;
  MExtractor *extractor2=0;
  TString geom;

  if (!ReadCalibration(calibcont, badpix, extractor1, extractor2, geom))
    return kFALSE;
  
  *fLog << all;
  if (extractor1)
    {
      *fLog << underline << "Signal Extractor found in calibration file" << endl;
      extractor1->Print();
      *fLog << endl;
    }
  else
    *fLog << inf << "No Signal Extractor: ExtractSignal in file." << endl;
  
  if (extractor2)
    {
      *fLog << underline << "Time Extractor found in calibration file" << endl;
      extractor2->Print();
      *fLog << endl;
    }
  else
    *fLog << inf << "No Time Extractor: ExtractTime in file." << endl;
  
  if (!geom.IsNull())
    *fLog << inf << "Camera geometry found in file: " << geom << endl;
  else
    *fLog << inf << "No Camera geometry found using default <MGeomCamMagic>" << endl;

  if (fExtractor)
    extractor1 = fExtractor;
  if (fTimeExtractor)
    extractor2 = fTimeExtractor;

  // Setup Lists
  MParList plist;
  plist.AddToList(this); // take care of fDisplay!
  plist.AddToList(&fTestCam);
  plist.AddToList(&fTestTimeCam);
  plist.AddToList(&badpix);
  plist.AddToList(&pedcam);
  plist.AddToList(&calcam);
  plist.AddToList(&qecam);
  plist.AddToList(&tmcam);

  MCerPhotEvt          cerphot;
  MPedPhotCam          pedphot;
  MHCalibrationTestCam testcam;

  plist.AddToList(&cerphot);
  plist.AddToList(&pedphot);
  plist.AddToList(&testcam);

  pedcam.SetName("MPedestalFundamental");
  
  MTaskList tlist;
  plist.AddToList(&tlist);

  // Setup Task-lists
  MRawFileRead rawread(NULL);
  MReadMarsFile read("Events");
  read.DisableAutoScheme();

  if (IsUseRawData())
    rawread.AddFiles(fSequence.IsValid() ? iter : *fRuns);
  else
    static_cast<MRead&>(read).AddFiles(fSequence.IsValid() ? iter : *fRuns);

  // Check for interleaved events
  MTriggerPatternDecode decode;
  MFTriggerPattern      fcalib;
  fcalib.RequireCalibration();
  fcalib.SetInverted();
  MContinue conttp(&fcalib, "ContTrigPattern");

  MGeomApply            apply; // Only necessary to craete geometry
  if (!geom.IsNull())
    apply.SetGeometry(geom);  
  MBadPixelsMerge       merge(&badpix);

  MExtractTimeAndChargeSlidingWindow extrsw;
  MExtractTimeFastSpline             extime;
  extime.SetPedestals(&pedcam);

  MTaskEnv taskenv1("ExtractSignal");
  MTaskEnv taskenv2("ExtractTime");

  if (extractor1)
    { 
      extractor1->SetPedestals(&pedcam);
      taskenv1.SetDefault(extractor1);
    }

  if (extractor2)
    {
      fTimeExtractor->SetPedestals(&pedcam);
      taskenv2.SetDefault(fTimeExtractor);
    }
  else if (!(extractor1->InheritsFrom("MExtractTimeAndCharge")))
    {
      extrsw.SetPedestals(&pedcam);
      extrsw.SetWindowSize(8,8);
      taskenv2.SetDefault(&extrsw);
      *fLog << warn << GetDescriptor() 
            << ": No extractor has been chosen, take default MExtractTimeAndChargeSlidingWindow " << endl;
    }

  MCalibrateData        photcalc;
  MCalibrateRelTimes    caltimes;
  if (IsUseMC()) // MC file
    {
      photcalc.SetCalibrationMode(MCalibrateData::kFfactor);
      photcalc.SetPedestalFlag(MCalibrateData::kRun);
      photcalc.AddPedestal("MPedestalCam", "MPedPhotFundamental");
    }
  else
    {
      photcalc.SetCalibrationMode(MCalibrateData::kFfactor);
      photcalc.AddPedestal("Fundamental");
      photcalc.SetPedestalFlag(MCalibrateData::kEvent);
      photcalc.SetSignalType(MCalibrateData::kPhe);
    }

  MBadPixelsCalc        badcalc;
  MBadPixelsTreat       badtreat;
  badtreat.SetProcessTimes(kFALSE);

  badcalc.SetNamePedPhotCam("MPedPhotFundamental");
  //badtreat.SetUseInterpolation();
  badtreat.AddNamePedPhotCam("MPedPhotFundamental");

  MCalibrationTestCalc  testcalc;

  if (!fSequence.IsValid())
    {
      testcalc.SetOutputPath(fPathOut);
      testcalc.SetOutputFile(Form("%s-TestCalibStat.txt",(const char*)fRuns->GetRunsAsFileName()));
    }
  
  MHCamEvent evt0(0,"Signal", "Un-Calibrated Signal;;S [FADC cnts]" );
  MHCamEvent evt1(3,"CalSig", "Calibrated and Interpolated Signal;;S [\\gamma]");
  MHCamEvent evt2(0,"Times" , "Arrival Time;;T [slice]");

  MFillH fill0(&evt0, "MExtractedSignalCam", "FillUncalibrated");
  MFillH fill1(&evt1, "MCerPhotEvt", "FillCalibrated");
  MFillH fill2(&evt2, "MArrivalTime","FillTimes");
  
  MFillH fillcam("MHCalibrationTestCam", "MCerPhotEvt");
  fillcam.SetNameTab("Test");
  MFillH filltme("MHCalibrationTestTimeCam", "MArrivalTime");
  filltme.SetNameTab("TestTime");

  MFCosmics cosmics;
  cosmics.SetNamePedestalCam("MPedestalFundamental");
  MContinue contcos(&cosmics,"ContCosmics");
  
  tlist.AddToList(&read);
  tlist.AddToList(&decode);
  tlist.AddToList(&apply);
  tlist.AddToList(&merge);
  tlist.AddToList(&conttp);
  tlist.AddToList(&taskenv1);
  if (!extractor1->InheritsFrom("MExtractTimeAndCharge"))
    tlist.AddToList(&taskenv2);

  tlist.AddToList(&contcos);
  tlist.AddToList(&fill0);
  tlist.AddToList(&photcalc);
  tlist.AddToList(&caltimes);
  tlist.AddToList(&badcalc);
  tlist.AddToList(&badtreat);
  tlist.AddToList(&fill1);
  tlist.AddToList(&fill2);
  tlist.AddToList(&fillcam);
  tlist.AddToList(&filltme);
  tlist.AddToList(&testcalc);
  
  // Create and setup the eventloop
  MEvtLoop evtloop(fName);
  evtloop.SetParList(&plist);
  evtloop.SetDisplay(fDisplay);
  evtloop.SetLogStream(fLog);
  
  // Execute first analysis
  if (!evtloop.Eventloop())
    {
      *fLog << err << GetDescriptor() << ": Failed." << endl;
      return kFALSE;
    }
  
  tlist.PrintStatistics();
  
  DisplayResult(plist);

  if (!WriteResult())
    return kFALSE;

  *fLog << inf << GetDescriptor() << ": Done." << endl;
  
  return kTRUE;
}

Bool_t MJCalibTest::WriteResult()
{
    
    if (fPathOut.IsNull())
        return kTRUE;
    
    const TString oname(GetOutputFile());

    *fLog << inf << "Writing to file: " << oname << endl;
    
    TFile file(oname, "UPDATE");
    
    if (fDisplay && fDisplay->Write()<=0)
    {
        *fLog << err << "Unable to write MStatusDisplay to " << oname << endl;
        return kFALSE;
    }

    if (fPedPhotCam.Write()<=0)
    {
        *fLog << err << "Unable to write MPedPhotCam to " << oname << endl;
        return kFALSE;
    }

    if (fTestCam.Write()<=0)
    {
        *fLog << err << "Unable to write MCalibrationTestCam to " << oname << endl;
        return kFALSE;
    }

    if (fTestTimeCam.Write()<=0)
    {
        *fLog << err << "Unable to write MCalibrationTestCam to " << oname << endl;
        return kFALSE;
    }

    return kTRUE;

}

