/* ======================================================================== *\
   !
   ! *
   ! * This file is part of MARS, the MAGIC Analysis and Reconstruction
   ! * Software. It is distributed to you in the hope that it can be a useful
   ! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
   ! * It is distributed WITHOUT ANY WARRANTY.
   ! *
   ! * Permission to use, copy, modify and distribute this software and its
   ! * documentation for any purpose is hereby granted without fee,
   ! * provided that the above copyright notice appear in all copies and
   ! * that both that copyright notice and this permission notice appear
   ! * in supporting documentation. It is provided "as is" without express
   ! * or implied warranty.
   ! *
   !
   !
   !   Author(s): Abelardo Moralejo 1/2004 <mailto:moralejo@pd.infn.it>
   !              Thomas Bretz  5/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
   !
   !   Copyright: MAGIC Software Development, 2000-2004
   !
   !
   \* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MMCALIBRATE - Calibration of MC data
//
//  This macro converts raw MC data into calibrated data (photons per pixel) 
//
/////////////////////////////////////////////////////////////////////////////

#include "MImgCleanStd.h"

void mccalibrate()
{
  //
  // This is a demonstration program which reads in MC camera files
  // and produces and output with calibrated events (signal in photons
  // for all pixels, in MCerPhotEvt containers).
  //

  // ------------- user change -----------------
  TString* CalibrationFilename;

  CalibrationFilename = new TString("/users/emc/moralejo/mcdata/Period021_0.73_mirror/gammas_nonoise/Gamma_*root");  // File to be used for the calibration (must be a camera file without added noise)

  Char_t* AnalysisFilename = "Gamma_*.root";  // File to be analyzed

  Char_t* OutFilename      = "calibrated_gamma.root";  // Output file name


  // (other extraction methods can be used)
  //    MExtractFixedWindowPeakSearch sigextract;
  //    sigextract.SetWindows(6, 6, 4);

  MExtractTimeAndChargeDigitalFilter sigextract;
  sigextract.SetNameWeightsFile("/users/emc/moralejo/Mars/msignal/MC_weights.dat");
  sigextract.SetRange(1, 14, 3, 14);


  MMcCalibrationUpdate  mccalibupdate;
  ///// User change: calibrate in photons or phe- :
  mccalibupdate.SetSignalType(MCalibrateData::kPhe);
  //  mccalibupdate.SetSignalType(MCalibrateData::kPhot);

  // ---------------------------------------------------------------------
  //
  // Create a empty Parameter List and an empty Task List
  // The tasklist is identified in the eventloop by its name
  //
  MParList  plist;

  MTaskList tlist;

  plist.AddToList(&tlist);

  MBadPixelsCam badpix;
  plist.AddToList(&badpix);  // Not used for now.
 
  //
  // Now setup the tasks and tasklist:
  // ---------------------------------
  //
  MReadMarsFile read("Events");

  if (CalibrationFilename)
    read.AddFile(CalibrationFilename->Data());

  read.DisableAutoScheme();

  MGeomApply geom; 
  // Reads in geometry from MC file and sets the right sizes for
  // several parameter containers.

  MMcPedestalCopy   pcopy; 
  // Copies pedestal data from the MC file run fadc header to the 
  // MPedestalCam container.

  MPointingPosCalc pointcalc;
  // Creates MPointingPos object and fill it with the telescope orientation
  // information taken from MMcEvt.

  MCalibrateData calib; 
  // MCalibrateData transforms signals from ADC counts into photons. In the first
  // loop it applies a "dummy" calibration supplied by MMcCalibrationUpdate, just 
  // to equalize inner and outer pixels. At the end of the first loop, in the
  // PostProcess of MMcCalibrationCalc (see below) the true calibration constants
  // are calculated.

  calib.SetCalibrationMode(MCalibrateData::kFfactor);
  // Do not change the CalibrationMode above for MC...!

  // Now set also whether to calibrate in photons or phe-:
  calib.SetSignalType(mccalibupdate.GetSignalType());

  MImgCleanStd clean;
  //
  // Applies tail cuts to image. Since the calibration is performed on 
  // noiseless camera files, the precise values of the cleaning levels 
  // are unimportant (in any case, only pixels without any C-photon will
  // be rejected).
  //

  MHillasCalc hcalc; // Calculates Hillas parameters not dependent on source position.
  hcalc.Disable(MHillasCalc::kCalcHillasSrc);

  MMcCalibrationCalc mccalibcalc; 
  // Calculates calibration constants to convert from ADC counts to photons.
  

  tlist.AddToList(&read);
  tlist.AddToList(&geom);
  tlist.AddToList(&pcopy);
  tlist.AddToList(&pointcalc);
  tlist.AddToList(&sigextract);
  tlist.AddToList(&mccalibupdate);
  tlist.AddToList(&calib);
  tlist.AddToList(&clean);
  tlist.AddToList(&hcalc);

  tlist.AddToList(&mccalibcalc);

  //
  // Open output file:
  //
  MWriteRootFile write(OutFilename); // Writes output
  write.AddContainer("MGeomCam",            "RunHeaders");
  write.AddContainer("MMcConfigRunHeader",  "RunHeaders");
  write.AddContainer("MMcCorsikaRunHeader", "RunHeaders");
  write.AddContainer("MMcFadcHeader",       "RunHeaders");
  write.AddContainer("MMcRunHeader",        "RunHeaders");
  write.AddContainer("MMcTrigHeader",       "RunHeaders");
  write.AddContainer("MRawRunHeader",       "RunHeaders");


  write.AddContainer("MMcEvt",        "Events");
  write.AddContainer("MMcTrig",       "Events");
  write.AddContainer("MPointingPos",  "Events");
  write.AddContainer("MRawEvtHeader", "Events");
  write.AddContainer("MCerPhotEvt",   "Events");
  write.AddContainer("MPedPhotCam",   "Events");

  //
  // First loop: Calibration loop
  //

  MProgressBar bar;
  bar.SetWindowName("Calibrating...");

  MEvtLoop evtloop;
  evtloop.SetProgressBar(&bar);
  evtloop.SetParList(&plist);

  if (CalibrationFilename)
    {
      if (!evtloop.Eventloop())
	return;
      mccalibcalc->GetHistADC2PhotEl()->Write();
      mccalibcalc->GetHistPhot2PhotEl()->Write();
      // Writes out the histograms used for calibration.
    }

  //
  // Second loop: apply calibration factors to MC events in the 
  // file to be anlyzed:
  //

  //
  // Change the read task by another one which reads the file we want to analyze:
  //

  MReadMarsFile read2("Events");
  read2.AddFile(AnalysisFilename);
  read2.DisableAutoScheme();
  tlist.AddToListBefore(&read2, &read, "All");
  tlist.RemoveFromList(&read);

  bar.SetWindowName("Writing...");

  tlist.RemoveFromList(&clean);
  tlist.RemoveFromList(&hcalc);
  tlist.RemoveFromList(&mccalibcalc);

  tlist.AddToList(&write);            // Add task to write output.

  if (!evtloop.Eventloop())
    return;


  tlist.PrintStatistics();
}
