/********************************************************************\

  rootread.cc

  Reads rawfile created by the CTX DAQ.
  Run using '.x rootread.cc++(Filename)'.
  Generate shared library using make, then load library in root
  with gSystem->Load("RawDataCTX.so")
  
  Oliver Grimm

\********************************************************************/

#include <stdlib.h>

#include <TCanvas.h>
#include <TH1.h>
#include <TTimer.h>
#include <Getline.h>
#include <TLegend.h>

#include "../../drsdaq/RawDataCTX.h"

void rootread(char* Filename) {

  #ifdef __CINT__
  printf("Error: This script can only be executed via ACliC\n");
  return;
  #endif

  RawDataCTX *RD = new RawDataCTX();

  // Open data file and read run header
  if (RD->OpenDataFile(Filename, NULL) != CTX_OK) {
    delete RD;
    return;
  }

  // ...for convenience some abbrevations
  unsigned int Boards = RD->RHeader->NBoards;
  unsigned int Channels = RD->RHeader->NChips*RD->RHeader->NChannels;
  unsigned int Samples = RD->RHeader->Samples;
  
  // Allocate histograms and legends
  TH1F* h = new TH1F [Boards*Channels];
  TLegend* Legend = new TLegend [Boards];

  // Create a canvas
  TCanvas* CW = new TCanvas("CW","DRS Waveform",10,10,700,200*Boards);
  CW->Divide(1, Boards);
 
  // Initialise histograms and legends
  for (unsigned int i=0; i<Boards; i++) {
    for (unsigned int j=0; j<Channels; j++) {
      h[i*Channels+j].SetBins(Samples, 0, Samples);
      h[i*Channels+j].SetLineColor(1+j%8);
      h[i*Channels+j].SetLineWidth(3);
      if ((j+1)/8) h[i*Channels+j].SetLineStyle((j+2)%8);
      Legend[i].AddEntry(&h[i*Channels+j], Form("Ch. %d",j), "L");
    }
    h[i*Channels].GetXaxis()->SetTitle(Form("Time slice (%.1f ns/slice)", 1./RD->BStruct[i].NomFreq));
    h[i*Channels].GetYaxis()->SetTitle("Amplitude (a.u.)");

    Legend[i].SetX1NDC(0.91);
    Legend[i].SetY1NDC(0.40);
    Legend[i].SetX2NDC(0.98);
    Legend[i].SetY2NDC(0.90);
    Legend[i].SetFillColor(0);
    Legend[i].SetLineColor(0);
    Legend[i].SetBorderSize(0);
    Legend[i].SetTextSize(0.03);
    Legend[i].SetHeader(Form("Board %d",i));
  }

  // Read and display all events
  while (RD->ReadEvent(0, stdout) == CTX_OK) {
    int *TrigPnt = (int *) RD->Data;
    short *DataPnt = (short *) RD->Data + Boards*RD->RHeader->NChips*sizeof(int);
    
    printf("Trigger cells: ");
    for(unsigned int i=0; i<Boards*RD->RHeader->NChips; i++) printf("%d ", *(TrigPnt + i));
    printf("\n");
     
    for(unsigned int k=0; k<Boards; k++) {
      for(unsigned int j=0; j<Channels; j++) {
	for(unsigned int i=0; i<Samples ; i++)
          h[k*Channels+j].SetBinContent(i, DataPnt[k*Channels*Samples +
	    j*Samples + i] * RD->BStruct[k].ScaleFactor);
      }

      CW->cd(k+1);
      gPad->SetGrid();
      for (unsigned int j=0; j<Channels; j++) h[k*Channels+j].DrawCopy(j>0 ? "same":"");
      Legend[k].Draw();
      CW->Update();
    }
 
    // Process gui events asynchronously during input 
    TTimer timer("gSystem->ProcessEvents();", 50, kFALSE);
    timer.TurnOn();
    TString input = Getline("Type 'q' to exit, <return> to go on: ");
    timer.TurnOff();
    if (input=="q\n") break;
  }

  delete CW;   delete[] Legend;   delete[] h;   delete RD;
}
