// Function to read rawdata into Matlab
//
// Use 'mex matlabread.cc RawDataCTX.o' in Matlab to compile.
// Use 'make' on console to generate RawDataCTX.o.

#include <unistd.h>

#include "/ihp/local/Matlab08/extern/include/mex.h"
#include "../../drsdaq/RawDataCTX.h"

// Print text date in temporary file, than clear file
void PrintTempFile(FILE* File) {
  char Buffer[1000];
  rewind(File);
  while(fgets(Buffer, sizeof(Buffer), File) != NULL) printf("%s", Buffer);
  ftruncate(fileno(File),0);
}

// Interface function to Matlab
void mexFunction (int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[]) {
  
  RawDataCTX* RD = new RawDataCTX(true);	// Instantiate without console output
  FILE *Tmpfile = tmpfile();			// Temporary file for printing of headers

  // Check inputs and output arguments
  if (nrhs<1 || nrhs>2 || (nrhs==2 && !mxIsDouble(prhs[1]))) {
    mexErrMsgTxt("Parameters must be filename and (optional) event number.");
  }
  if (nlhs != 2) mexErrMsgTxt ("Need two return values.");
          
  switch (RD->OpenDataFile(mxArrayToString(prhs[0]), Tmpfile)) {
    case CTX_FOPEN:   mexErrMsgTxt("Could not open file.");
    case CTX_RHEADER: mexErrMsgTxt("Could not read run header.");
    case CTX_BSTRUCT: mexErrMsgTxt("Could not read board structures.");
  }
  if (RD->RHeader->MagicNum == MAGICNUM_OPEN) {
    mexWarnMsgTxt("Magic number in run header indicates that the file has not "
    		  "been closed properly.");
  }
  if (RD->RHeader->MagicNum == MAGICNUM_ERROR) {
    mexWarnMsgTxt("Magic number in run header indicates that an error occurred "
    		  "while writing the file.");
  }

  // ...some abbrevation for convenience
  unsigned int Boards = RD->RHeader->NBoards;
  unsigned int Channels = RD->RHeader->NChannels;
  unsigned int Chips = RD->RHeader->NChips;
  unsigned int Samples = RD->RHeader->Samples;
  
  // Print header data
  if(RD->ReadEvent(mxIsDouble(prhs[1]) ? (int) mxGetScalar(prhs[1]):0, Tmpfile) != CTX_OK) {
    mexErrMsgTxt("Could not read event.");
  }
  else PrintTempFile(Tmpfile);

  // Allocate memory
  plhs[0] = mxCreateDoubleMatrix(Boards*Chips*Channels*Samples, 1, mxREAL);
  plhs[1] = mxCreateDoubleMatrix(Chips, Boards, mxREAL);

  // Fill trigger cell array
  for(unsigned int i=0; i<Boards*Chips; i++) {
    *(mxGetPr(plhs[1])+i) = *((int *) (RD->Data+i*sizeof(int))); 
  }
  
  // Fill data array
  unsigned int Count = 0;
  for(unsigned int i=0; i<Boards; i++) {
    for(unsigned int j=0; j<Chips*Channels*Samples; j++) {
      *(mxGetPr(plhs[0])+Count) = (double) *((short *) (RD->Data+Boards*Chips*sizeof(int)) + Count++) * RD->BStruct[i].ScaleFactor; 
    }
  }

  // Resize data array into boards, channels and samples
  mwSize Dimensions[] = {Samples, Chips*Channels, Boards};
  mxSetDimensions(plhs[0], Dimensions, sizeof(Dimensions)/sizeof(mwSize));
  
  // Clean up
  fclose(Tmpfile);
  RD->CloseDataFile();      
}
