/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MCT1ReadAscii                                                           //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MCT1ReadAscii.h"

#include <fstream.h>

#include "MLog.h"
#include "MParList.h"
#include "MCerPhotEvt.h"
#include "MCT1Pedestals.h"

ClassImp(MCT1ReadAscii)

MCT1ReadAscii::MCT1ReadAscii(const char *fname,
			     const char *name, 
			     const char *title)
{
    *fName  = name  ? name  : "MCT1ReadAscii";
    *fTitle = title ? title : "Task to loop over events in CT1 ascii file";

    //
    // remember file name for opening the file in the preprocessor
    //
    fFileName = fname;
}

Bool_t MCT1ReadAscii::PreProcess(MParList *pList)
{
    //
    // Preprocessing
    //

    //
    // open the input stream and check if it is really open (file exists?)
    //
    fIn = new ifstream(fFileName);
    if (!(*fIn))
    {
        *fLog << "Error: MCT1ReadAscii::PreProcess: Cannot open file." << endl;
        return kFALSE;
    }

    //
    //  look for the MCerPhotEvt class in the plist
    //
    fNphot = (MCerPhotEvt*)pList->FindObject("MCerPhotEvt");
    if (!fNphot)
    {
        *fLog << "MRawCT1Ascii::PreProcess - WARNING: MCerPhotEvt not found... creating." << endl;
        fNphot = new MCerPhotEvt;
        pList->AddToList(fNphot);
    }

    //
    //  look for the pedestal class in the plist
    //
    fPedest = (MCT1Pedestals*)pList->FindObject("MCT1Pedestals");
    if (!fPedest)
    {
        *fLog << "MRawCT1Ascii::PreProcess - WARNING: MCT1Pedestals not found... creating." << endl;
        fPedest = new MCT1Pedestals;
        pList->AddToList(fPedest);
    }

    return kTRUE;
}

Bool_t MCT1ReadAscii::Process()
{
    //
    // read in a dummy number (event number)
    //
    Int_t dummyI;
    *fIn >> dummyI;

    if (fIn->eof())
    {
        *fLog << "MRawCT1Ascii::Process - Error: EOF reached." << endl;
        return kFALSE;
    }

    //
    // if the first number is negativ this is a pedestal line:
    // read in pedestals
    //
    if (dummyI < 0)
        fPedest->AsciiRead(*fIn);

    //
    // five unsused numbers
    //
    *fIn >> dummyI;
    *fIn >> dummyI;
    *fIn >> dummyI;
    *fIn >> dummyI;

    //
    // clear the list of cerphot-events
    //
    fNphot->Clear();

    //
    // read in the number of cerenkov photons and add the 'new' pixel
    // too the list with it's id, number of photons and error
    //
    for (Int_t i = 0; i<127; i++ )
    {
        Float_t dummyF;

        *fIn >> dummyF;

        if (dummyF > 0.0)
            fNphot->AddPixel(i, dummyF, (*fPedest)[i]);
    }

    return kTRUE;
}

Bool_t MCT1ReadAscii::PostProcess()
{
    //
    // close and delete the input stream
    //
    delete fIn;

    return kTRUE;
}

