#ifndef __MFadc__
#define __MFadc__
//
//     class MFadc
//
//     implemented by Harald Kornmayer
//
//     This is a class to simulate the FADC. 
//     It assumes a special response of the PMT for one single Photo-electron. 
//     
//
//
#include <sstream>
#include <math.h>

#include "TObject.h"
#include "TRandom.h"

#include "Mdefine.h" 

#include "MTriggerDefine.h"
#include "MFadcDefine.h"

class MMcEvt; 

//==========
//  MFadc 
//
//  The simulation of the Flash ADC system for the MAGIC teleskop is done with
//  this class. 
//  So all methods concerning the FADC System should be done inside this
//  class. 
//
//  The Idea is to (in)put the data of the photo electrons into the class and
//  generate the response (output) of the FADC to that input. Response means 
//  in this sense the ADC values of the different time slices for all pixels
//
//  The pixelisation is done by the camera program of Jose Carlos. 
//
//  This class is closly connected to the MTrigger classs. So some of the
//  values defined in MTriggerDefine.h are also used by this class. 
//
//  But a lot of other stuff is defined in MFadcDefine.h. 
//
//


class MFadc {
 private:
  Int_t numpix;
  Int_t fInnerPixelsNum; // number of inner (small) pixels.
  //
  //    then for all pixels the shape of all the analog signals 
  //
  Bool_t   used[CAMERA_PIXELS] ;  //  a boolean to indicated if the pixels is used in this event
  Float_t  pedestal[CAMERA_PIXELS] ;  //  Pedestal of FADCs

  Int_t    fSlices_mFadc;  // Number of simulated FADC slices. Larger than the actual number of slices
                           // that will be written, since we have to account for possible delays in the 
                           // trigger

  Int_t fResponseSlicesFadc;

  //
  //  Float_t sig[CAMERA_PIXELS][] ; 
  //  the analog signal for pixels, in bins of width
  //  equal to the FADC slice: (default 50/15 ns), but with a
  //  total of fSlices_mFadc bins (default 48).
  //
  Float_t *sig[CAMERA_PIXELS];
  Float_t *sig_LG[CAMERA_PIXELS];

  Float_t *noise;
  Float_t *noise_outer;
  Float_t *digital_noise;

  Float_t  *output[CAMERA_PIXELS];  //  the analog signal for pixels that is read after a trigger 
                                    //  occurs (high gain).

  Float_t  *output_lowgain[CAMERA_PIXELS]; //  analog signal, low gain.

  Float_t high2low_gain;

  //
  //    first the data for the response function
  //
  Int_t shape_resp;      // index shape of the phe_response function
                         // = 0 --> Gaussian
                         // = 1 --> Pulpo set-up
  Float_t fwhm_resp;     // fwhm of the phe_response function (in ns)
  Float_t integ_resp;    // area below curve of the phe_response function (in counts * ns)
  Float_t *sing_resp;    // the shape of the phe_response function 
  Float_t *sing_resp_lowgain; // phe response for the low gain

  Float_t fFadcSlicesPerNanosec; // Number of FADC slices per nanosecond, that is, the
                                 // sampling frequency of the FADC.

  Int_t fFadcSlices;     // Number of FADC slices that will be written on the output 
                         // file (same number for high and low gain)

  //
  //    We may end up with a different reponse for the outer pixels
  //
  Int_t shape_resp_outer ;  // index shape of the phe_response function
                            // = 0 --> Gaussian
                            // = 1 --> Pulpo set-up
  Float_t fwhm_resp_outer ; // fwhm of the phe_response function (in ns)
  Float_t integ_resp_outer; // area below curve of the phe_response function (in counts * ns)
  Float_t *sing_resp_outer; // the shape of the phe_response function 
  Float_t *sing_resp_outer_lowgain; // phe response for the low gain

  Int_t fGainSwitchAmp; // Height of the high gain signal (in ADC counts) at which we decide
                        // to fill the low gain with a scaled down version of the pulse in the
                        // high gain. Else we put in the continuation of the high gain. 
                        // By default it is now 120 ADC counts (see constructor).

  Int_t fShiftFromSwitch2LowGain; 
  // Distance in FADC slices from the slice in which the amplitude
  // fGainSwitchAmp is reached to were the switch to low gain will happen.
  // By default it is now 13 slices (see constructor)

  Int_t fHi2LoGainPeak;
  // Distance in FADC slices from the signal peak in the high gain to the signal peak in
  // the low gain. By default we set now 16 slices (see constructor).

  //
  //   RandomGenerator for the Electronic Noise
  //

  TRandom  *GenElec ; 

  Float_t fadc_time_offset; // Time offset to adjust the delay between trigger
                            // and the peak position in the FADC of the signal
                            // in the trigger pixels.

  Float_t fPulseParameters[7];    // Parameters for the parametrization of the real pulse shape
  Float_t fPulseParametersLG[7];  // The same for low gain

public:

  MFadc(Int_t pix = 577,
	Int_t shape = 0,
	Float_t ampl = MFADC_RESPONSE_INTEGRAL, 
	Float_t fwhm = MFADC_RESPONSE_FWHM,
	Int_t shapeout = 0,
	Float_t amplout = MFADC_RESPONSE_INTEGRAL, 
	Float_t fwhmout = MFADC_RESPONSE_FWHM,
	Float_t trig_delay = 0.,
	Float_t fadc_slices_per_ns = FADC_SLICES_PER_NSEC,
	Int_t   fadc_slices_written = FADC_SLICES,
	Int_t   gainswitchamp = 120,
	Int_t   shiftfromswitch2lowgain = 13,
	Int_t   hi2logainpeak = 16);

  void SetSeed(UInt_t seed)  {GenElec->SetSeed(seed);}

  void Reset() ; 

  void Fill( Int_t, Float_t, Float_t, Int_t  ) ;  

  void Fill( Int_t, Float_t, Float_t  ) ;  

  void FillOuter( Int_t, Float_t, Float_t  ) ;  

  void AddSignal( Int_t iPix, Float_t *res);

  void SetPedestals( Int_t ped);

  void SetPedestals( Float_t *ped);

  void SetShape( Int_t inner, Int_t outer){
    shape_resp=inner;
    shape_resp_outer=outer;
  }

  void SetFwhm( Float_t fwhm){
    fwhm_resp=fwhm;
  }

  void SetInteg( Float_t x){
    integ_resp=x;
  }

  void SetFwhmOuter( Float_t fwhm){
    fwhm_resp_outer=fwhm;
  }

  void SetIntegOuter( Float_t x){
    integ_resp_outer=x;
  }


  void Pedestals();

  void SetElecNoise(Float_t value1=2.0, Float_t value2=2.0, UInt_t ninpix=CAMERA_PIXELS);

  void ElecNoise(); 

  void SetDigitalNoise(Float_t value=2.0);

  void DigitalNoise() ; 

  void Scan() ; 

  void Scan(Float_t time) ;

  void GetResponse( Float_t *resp ) ; 

  void GetPedestals( Float_t *offset);

  Float_t GetPedestalNoise (Int_t pix, Int_t ishigh);

  Float_t AddNoiseInSlices( Int_t pix, Int_t ishigh, Int_t n_slices);

  void TriggeredFadc(Float_t time);

  void ShowSignal ( MMcEvt *McEvt , Float_t ) ; 

  UChar_t GetFadcSignal(Int_t pixel, Int_t slice);

  UChar_t GetFadcLowGainSignal(Int_t pixel, Int_t slice);

  void SetHigh2LowGain(Float_t h2l) {high2low_gain=h2l;}

  Int_t GetShape() {
    return shape_resp;
  }

  Float_t GetIntegral() {
    return integ_resp ; 
  }

  Float_t GetFwhm() {
    return fwhm_resp ; 
  }

  Float_t GetIntegralOuter() {
    return integ_resp_outer ; 
  }

  Float_t GetFwhmOuter() {
    return fwhm_resp_outer ; 
  }

  Float_t GetFadcSlicesPerNanosec() {
    return fFadcSlicesPerNanosec;
  }

  Int_t GetResponseSlicesFadc() {
    return fResponseSlicesFadc;
  }

  Bool_t IsPixelUsed(UInt_t p){
    return used[p];
  }
  
} ; 


#endif
