/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek, 03/2003 <mailto:wittek@mppmu.mpg.de>
!   Author(s): Thomas Bretz, 04/2003 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Markus Meyer, 02/2005 <mailto:meyer@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MHMuonPar
// This class is a histogram class for displaying muonparameters.
// The folowing histgrams will be plotted:
// - Radius (TH1F)
// - ArcWidth (TH1F)
// - ArcWidth/Radius vs Radius (TProfile) (it is the energy dependent
//   relative broadening of the muon ring)
// - Size Vs Radius
//
// Inputcontainer:
// MMuonSearchPar
// MMuonCalibPar
//
////////////////////////////////////////////////////////////////////////////
#include "MHMuonPar.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>
#include <TProfile.h>
#include "MLog.h"
#include "MLogManip.h"

#include "MGeomCam.h"
#include "MBinning.h"
#include "MParList.h"

#include "MHillas.h"
#include "MHMuonPar.h"
#include "MMuonSearchPar.h"
#include "MMuonCalibPar.h"

ClassImp(MHMuonPar);

using namespace std;

// --------------------------------------------------------------------------
//
// Setup histograms 
//
MHMuonPar::MHMuonPar(const char *name, const char *title):fGeomCam(NULL), fMuonSearchPar(NULL),
    fMuonCalibPar(NULL)
{
    fName  = name  ? name  : "MHMuonPar";
    fTitle = title ? title : "Histograms of muon parameters";

    fHistRadius.SetName("Radius");
    fHistRadius.SetTitle("Radius");
    fHistRadius.SetXTitle("Radius[deg]");
    fHistRadius.SetYTitle("Counts");
    fHistRadius.SetDirectory(NULL);
    fHistRadius.UseCurrentStyle();
    fHistRadius.SetFillStyle(4000);

    fHistArcWidth.SetName("ArcWidth");
    fHistArcWidth.SetTitle("ArcWidth");
    fHistArcWidth.SetXTitle("ArcWidth[deg]");
    fHistArcWidth.SetYTitle("Counts");
    fHistArcWidth.SetDirectory(NULL);
    fHistArcWidth.UseCurrentStyle();
    fHistArcWidth.SetFillStyle(4000);

    fHistBroad.SetName("Ringbroadening");
    fHistBroad.SetTitle("Ringbroadening");
    fHistBroad.SetXTitle("Radius[deg]");
    fHistBroad.SetYTitle("ArcWidth/Radius");
    fHistBroad.SetDirectory(NULL);
    fHistBroad.UseCurrentStyle();
    fHistBroad.SetFillStyle(4000);

    fHistSize.SetName("SizeVsRadius");
    fHistSize.SetXTitle("Radius");
    fHistSize.SetYTitle("MuonSize");
    fHistSize.SetDirectory(NULL);
    fHistSize.UseCurrentStyle();
    fHistSize.SetFillStyle(4000);

    MBinning bins;

    bins.SetEdges(20, 0.5, 1.5);
    bins.Apply(fHistRadius);

    bins.SetEdges(60, 0., 0.3);
    bins.Apply(fHistArcWidth);

    bins.SetEdges(20, 0.5, 1.5);
    bins.Apply(fHistBroad);

    bins.SetEdges(20, 0.5, 1.5);
    bins.Apply(fHistSize);

}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning
//
Bool_t MHMuonPar::SetupFill(const MParList *plist)
{
    fGeomCam = (MGeomCam*)plist->FindObject("MGeomCam");
    if (!fGeomCam)
    {
        *fLog << warn << "MGeomCam not found... abort." << endl;
        return kFALSE;
    }
    fMuonSearchPar = (MMuonSearchPar*)plist->FindObject("MMuonSearchPar");
    if (!fMuonSearchPar)
    {
        *fLog << warn << "MMuonSearchPar not found... abort." << endl;
        return kFALSE;
    }
    fMuonCalibPar = (MMuonCalibPar*)plist->FindObject("MMuonCalibPar");
    if (!fMuonCalibPar)
    {
        *fLog << warn << "MMuonCalibPar not found... abort." << endl;
        return kFALSE;
    }

    fMm2Deg = fGeomCam->GetConvMm2Deg();

    ApplyBinning(*plist, "Radius", &fHistRadius);

    ApplyBinning(*plist, "ArcWidth",  &fHistArcWidth);

    ApplyBinning(*plist, "Ringbroadening",  &fHistBroad);

    ApplyBinning(*plist, "SizeVsRadius",  &fHistSize);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the histograms with data from a MMuonCalibPar and
// MMuonSearchPar container.
//
Bool_t MHMuonPar::Fill(const MParContainer *par, const Stat_t w)
{
    fHistRadius.Fill(fMm2Deg*fMuonSearchPar->GetRadius(), w);

    fHistArcWidth.Fill(fMuonCalibPar->GetArcWidth(), w);

    fHistBroad.Fill(fMm2Deg*fMuonSearchPar->GetRadius(),
                    fMuonCalibPar->GetArcWidth()/(fMm2Deg*fMuonSearchPar->GetRadius()), w);

    fHistSize.Fill(fMm2Deg*fMuonSearchPar->GetRadius(),
                   fMuonCalibPar->GetMuonSize(), w);

    return kTRUE;
}
// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHMuonPar::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    AppendPad("");

    pad->Divide(2,2);

    pad->cd(1);
    gPad->SetBorderMode(0);
    fHistRadius.Draw();

    pad->cd(2);
    gPad->SetBorderMode(0);
    fHistArcWidth.Draw();

    pad->cd(3);
    gPad->SetBorderMode(0);
    fHistSize.Draw();

    pad->cd(4);
    gPad->SetBorderMode(0);
    fHistBroad.Draw();
}

TH1 *MHMuonPar::GetHistByName(const TString name)
{
    if (name.Contains("Radius", TString::kIgnoreCase))
        return &fHistRadius;
    if (name.Contains("ArcWidth", TString::kIgnoreCase))
        return &fHistArcWidth;
    return NULL;
}

