/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MHThreshold
//
//////////////////////////////////////////////////////////////////////////////
#include "MHThreshold.h" 

#include <TF1.h>
#include <TLatex.h>
#include <TCanvas.h>
#include <TPaveStats.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MBinning.h"

#include "MMcEvt.hxx"

#include "MParList.h"

ClassImp(MHThreshold);

using namespace std;

// --------------------------------------------------------------------------
//
//  Creates the three necessary histograms:
//   - selected showers (input)
//   - all showers (input)
//   - collection area (result)
//
MHThreshold::MHThreshold(const char *name, const char *title)
{ 
    //   initialize the histogram for the distribution r vs E
    //
    //   we set the energy range from 2 Gev to 20000 GeV (in log 4 orders
    //   of magnitude) and for each order we take 25 subdivision --> 100 xbins
    //
    //   we set the radius range from 0 m to 500 m with 10 m bin --> 50 ybins
    //
    fName  = name  ? name  : "MHThreshold";
    fTitle = title ? title : "Collection Area vs. Energy/Theta";

    fHEnergy.SetName("Threshold");
    fHEnergy.SetTitle("Energy Thrshold");
    fHEnergy.SetXTitle("E [GeV]");
    fHEnergy.SetYTitle("dN/dE [GeV^{-1}]");
    fHEnergy.SetDirectory(NULL);
    fHEnergy.UseCurrentStyle();

    MBinning binse(80, 10, 1000000, "", "log");
    binse.Apply(fHEnergy);
}

Bool_t MHThreshold::SetupFill(const MParList *pl)
{
    fMcEvt = (MMcEvt*)pl->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << "MMcEvt not found... abort." << endl;
        return kFALSE;
    }
/*
    MBinning binse;
    binse.SetEdges(fHEnergy, 'x');

    MBinning *bins = (MBinning*)pl->FindObject("BinningEnergyEst", "MBinning");
    if (bins)
        binse.SetEdges(*bins);

    binse.Apply(fHEnergy);
  */
    return kTRUE;
}

void MHThreshold::Paint(Option_t *option)
{
    const TAxis &axe = *fHEnergy.GetXaxis();

    const Int_t  bin  = fHEnergy.GetMaximumBin();

    // Assume that the energy binning is logarithmic
    const Axis_t maxe = TMath::Sqrt(axe.GetBinLowEdge(bin)*axe.GetBinUpEdge(bin));

    TLatex text(0.30, 0.95, Form("E_{max}=%dGeV", TMath::Nint(maxe)));
    text.SetBit(TLatex::kTextNDC);
    text.SetTextSize(0.04);
    text.Paint();
}

void MHThreshold::Draw(Option_t *option)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);

    // Do the projection before painting the histograms into
    // the individual pads
    pad->SetBorderMode(0);

    gPad->SetLogx();
    gPad->SetLogy();

    fHEnergy.Draw();

    AppendPad();
}

Bool_t MHThreshold::Fill(const MParContainer *par, const Stat_t weight)
{
    const Double_t energy = fMcEvt->GetEnergy();

    fHEnergy.Fill(energy, weight/energy);

    return kTRUE;
}
