/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Hengstebeck 3/2003 <mailto:hengsteb@alwa02.physik.uni-siegen.de>
!
!   Copyright: MAGIC Software Development, 2000-2003
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MRanForestGrow                                                         //
//                                                                         //
//  Grows a random forest.                                                 //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////
#include "MRanForestGrow.h"

#include "MHMatrix.h" // must be before MLogManip.h

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MRanTree.h"
#include "MRanForest.h"

ClassImp(MRanForestGrow);

using namespace std;

static const TString gsDefName  = "MRead";
static const TString gsDefTitle = "Tree Classification Loop 1/2";

// --------------------------------------------------------------------------
//
// Setup histograms and the number of distances which are used for
// avaraging in CalcDist
//
MRanForestGrow::MRanForestGrow(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : gsDefName.Data();
    fTitle = title ? title : gsDefTitle.Data();

    SetNumTrees();
    SetNumTry();
    SetNdSize();
}

// --------------------------------------------------------------------------
//
// Needs:
//  - MatrixGammas  [MHMatrix]
//  - MatrixHadrons {MHMatrix]
//  - MHadroness
//  - all data containers used to build the matrixes
//
// The matrix object can be filles using MFillH. And must be of the same
// number of columns (with the same meaning).
//
Int_t MRanForestGrow::PreProcess(MParList *plist)
{
    fMGammas = (MHMatrix*)plist->FindObject("MatrixGammas", "MHMatrix");
    if (!fMGammas)
    {
        *fLog << err << dbginf << "MatrixGammas [MHMatrix] not found... aborting." << endl;
        return kFALSE;
    }

    fMHadrons = (MHMatrix*)plist->FindObject("MatrixHadrons", "MHMatrix");
    if (!fMHadrons)
    {
        *fLog << err << dbginf << "MatrixHadrons [MHMatrix] not found... aborting." << endl;
        return kFALSE;
    }

    if (fMGammas->GetM().GetNcols() != fMHadrons->GetM().GetNcols())
    {
        *fLog << err << dbginf << "Error matrices have different numbers of columns... aborting." << endl;
        return kFALSE;
    }

    fRanTree = (MRanTree*)plist->FindCreateObj("MRanTree");
    if (!fRanTree)
    {
        *fLog << err << dbginf << "MRanTree not found... aborting." << endl;
        return kFALSE;
    }

    fRanForest = (MRanForest*)plist->FindCreateObj("MRanForest");
    if (!fRanForest)
    {
        *fLog << err << dbginf << "MRanForest not found... aborting." << endl;
        return kFALSE;
    }

    fRanTree->SetNumTry(fNumTry);
    fRanTree->SetNdSize(fNdSize);
    fRanForest->SetCurTree(fRanTree);
    fRanForest->SetNumTrees(fNumTrees);

    return fRanForest->SetupGrow(fMHadrons,fMGammas);
}

// --------------------------------------------------------------------------
//
//
Int_t MRanForestGrow::Process()
{
    const Bool_t not_last=fRanForest->GrowForest();

    fRanTree->SetReadyToSave();

    return not_last;
}

Int_t MRanForestGrow::PostProcess()
{
    fRanTree->SetReadyToSave();
    fRanForest->SetReadyToSave();

    return kTRUE;
}
