#ifndef MARS_MDataSet
#define MARS_MDataSet

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

#ifndef ROOT_TArrayI
#include <TArrayI.h>
#endif

class TChain;

class MRead;
class MDirIter;
class MPointingPos;

class MDataSet : public MParContainer
{
private:
    static const TString fgCatalog;       //! Default Catalog path
    static const TString fgPathDataFiles; //! Default path to data files
    static const TString fgPathSequences; //! Default path to sequences

    UInt_t  fNumAnalysis;     // Analysis number (artificial)

    TArrayI fNumSequencesOn;  // number of on-sequences
    TArrayI fNumSequencesOff; // number of off-sequences

    TList   fSequencesOn;     // list of names and paths of on-sequences
    TList   fSequencesOff;    // list of names and paths of off-sequences

    TString fNameSource;      // Name of source from catalog
    TString fCatalog;         // edb catalog (magic_favourites.edb)

    TString fComment;         // Comment from DS file

    Bool_t  fIsWobbleMode;    // Wobble Mode for this dataset?

    void Split(TString &runs, TArrayI &data) const;
    void ResolveSequences(TEnv &env, const TArrayI &num, TList &list, const TString &sequences, const TString &data) const;
    static void PrintFile(const TObject &obj);

    void ReplaceDir(TList &list, const TString &old, const TString &news) const;
    void ReplaceFile(TList &list, const TString &old, const TString &news) const;

    void SetupDefaultPath(TString &path, const TString &def) const
    {
        if (path.IsNull())
            path = def;
        if (!path.EndsWith("/"))
            path += "/";
    }

public:
    MDataSet() : fNumAnalysis((UInt_t)-1) { }
    MDataSet(const char *fname, TString sequences="", TString data="");

    // Getter
    Bool_t IsValid() const { return fNumAnalysis!=(UInt_t)-1; }

    UInt_t GetNumSequencesOn() const  { return fNumSequencesOn.GetSize(); }
    UInt_t GetNumSequencesOff() const { return fNumSequencesOff.GetSize(); }

    Bool_t HasOffSequences() const { return GetNumSequencesOff()>0; }

    UInt_t GetNumAnalysis() const { return fNumAnalysis; }
    void   SetNumAnalysis(UInt_t num) { fNumAnalysis=num; }

    Bool_t HasSource() const { return !fNameSource.IsNull(); }
    Bool_t GetSourcePos(MPointingPos &pos) const;

    Bool_t IsWobbleMode() const { return fIsWobbleMode; }

    // Setter
    static Bool_t AddSequencesFromList(const TList &list, MDirIter &files);
    static Int_t  AddFilesToChain(MDirIter &files, TChain &chain);

    const TList &GetSequencesOn() const  { return fSequencesOn; }
    const TList &GetSequencesOff() const { return fSequencesOff; }

    Bool_t AddFiles(MRead &read) const;
    Bool_t AddFilesOn(MRead &read) const;
    Bool_t AddFilesOff(MRead &read) const;

    Bool_t AddFiles(TChain &read) const;
    Bool_t AddFilesOn(TChain &read) const;
    Bool_t AddFilesOff(TChain &read) const;

    void ReplaceDir(const TString &old, const TString &news)
    {
        ReplaceDir(fSequencesOn,  old, news);
        ReplaceDir(fSequencesOff, old, news);
    }

    void ReplaceFile(const TString &old, const TString &news)
    {
        ReplaceFile(fSequencesOn,  old, news);
        ReplaceFile(fSequencesOff, old, news);
    }

    // TObject
    void Print(Option_t *o="") const;

    ClassDef(MDataSet, 0)
};

#endif
