/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 12/2005 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2005
!
!
\* ======================================================================== */


//////////////////////////////////////////////////////////////////////////////
//
// MZlib
//
// This is a C++ wrapper for zlib (taken from root)
//
// WARNING: - There might not be support for all features.
//          - seek calls might be rather slow
//
//////////////////////////////////////////////////////////////////////////////
#include "MZlib.h"

ClassImp(MZlib);

using namespace std;

// --------------------------------------------------------------------------
//
// Open a file by name. Test if it is open like for an ifstream
// It doesn't matter whether the file is gzip compressed or not.
//
void MZlib::open(const char* name)
{
    if (is_open())
    {
        clear(rdstate()|ios::badbit);
        return;
    }

    fFile = gzopen(name, "rb0");
    if (fFile == 0)
    {
        clear(rdstate()|ios::badbit);
        return;
    }
}

// --------------------------------------------------------------------------
//
// Close an open file.
//
void MZlib::close()
{
    if (!is_open())
        return;

    sync();

    if (gzclose(fFile) != Z_OK)
        clear(rdstate()|ios::badbit);

    fFile = 0;
}

// --------------------------------------------------------------------------
//
int MZlib::underflow()
{
    if (gptr() && gptr()<egptr())
        return * reinterpret_cast<unsigned char *>(gptr());

    if (!is_open())
        return EOF;

    // implementation of inbuf
    const int putback = gptr()-eback()>4 ? 4 : gptr()-eback();

    memcpy(fBuffer+(4-putback), gptr()-putback, putback);

    const int num = gzread(fFile, fBuffer+4, fgBufferSize-4);
    if (num <= 0) // ERROR or EOF
        return EOF;

    // reset buffer pointers
    setg(fBuffer+(4-putback), fBuffer+4, fBuffer+4+num);

    // return next character
    return *reinterpret_cast<unsigned char *>(gptr());
}

// --------------------------------------------------------------------------
//
int MZlib::flush_buffer()
{
    // Separate the writing of the buffer from overflow() and sync() operation.
    const int w = pptr() - pbase();

    if (gzwrite(fFile, pbase(), w) != w)
        return EOF;

    pbump(-w);

    return w;
}

/*
int MZlib::overflow( int c)
{
    if ( ! ( mode & ios::out) || ! opened)
        return EOF;

    if (c != EOF)
    {
        *pptr() = c;
        pbump(1);

    }
    if ( flush_buffer() == EOF)
        return EOF;

    return c;
}
*/

// --------------------------------------------------------------------------
//
int MZlib::sync()
{
    // Use flush_buffer() instead of overflow(EOF) to
    // cause proper behavior with std::endl and flush()
    if (pptr() && pptr()>pbase())
    {
        if (flush_buffer() == EOF)
            return -1;
    }
    return 0;
}

// --------------------------------------------------------------------------
//
streambuf::pos_type MZlib::seekoff(streambuf::off_type offset, ios_base::seekdir dir, ios_base::openmode)
{
    return gzseek(fFile, offset, dir);
}
