/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MReadTree                                                               //
//                                                                         //
// This tasks opens all branches in a specified tree and creates the       //
// corresponding parameter containers if not already existing in the       //
// parameter list.                                                         //
//                                                                         //
// The Process function reads one events from the tree. To go through the  //
// events of one tree make sure that the event number is increased from    //
// outside. It makes also possible to go back by decreasing the number.    //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MReadTree.h"

#include <fstream.h>

#include <TFile.h>
#include <TChain.h>
#include <TObjArray.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MTime.h"
#include "MParList.h"

ClassImp(MReadTree)

MReadTree::MReadTree(const char *tname, const char *fname,
                     const char *name, const char *title)
{
    *fName  = name  ? name  : "MReadTree";
    *fTitle = title ? title : "Task to loop over all events in one single tree";

    //
    // open the input stream
    //
    fChain = new TChain(tname);

    if (fname)
        fChain->Add(fname);
}

MReadTree::~MReadTree()
{
    delete fChain;
}

/*Int_t*/ void MReadTree::AddFile(const char *fname)
{
    //
    // FIXME! A check is missing whether the file already exists or not.
    //
    //
    // returns the number of file which were added
    //
    /*return  root >3.0*/ fChain->Add(fname);
}

Bool_t MReadTree::PreProcess (MParList *pList)
{
    //
    // open file and check if file is really open
    //

    //
    // get number of events in this tree
    //
    fNumEntries = (UInt_t)fChain->GetEntries();

    if (!fNumEntries)
    {
        *fLog << dbginf << "No Entries found in chain (file/s)." << endl;
        return kFALSE;
    }

    //
    // set pointer to first event
    //
    fNumEntry   = 0;

    //
    // output logging information
    //
    *fLog << fNumEntries << " Entries found in file(s)." << endl;

    //
    // Get all branches of this tree and
    // create the Iterator to loop over all branches
    //
    TIter Next(fChain->GetListOfBranches());
    TBranch *branch=NULL;
    
    //
    // loop over all tasks for processing
    //
    while ( (branch=(TBranch*)Next()) )
    {
        //
        // Get Name of Branch
        //
        const char *name = branch->GetName();

        //
        // check if object is existing in the list
        //
        MParContainer *pcont = pList->FindCreateObj(name);
        

        if (!pcont)
        {
            //
            // if class is not existing in the (root) environment
            // we cannot proceed reading this branch
            //
            *fLog << "MReadTree::PreProcess - Warning: Class '" << name << "' not existing in dictionary. Branch skipped." << endl;
            continue;
        }

        //
        // here pcont is a pointer the to container in which the data from
        // the actual branch should be stored - enable branch.
        //
        branch->SetAddress(&pcont);
    }

    return kTRUE;
}

Bool_t MReadTree::Process()
{
    //
    // check for end of file
    //
    if (fNumEntry==fNumEntries)
        return kFALSE;

    //
    // get entry
    //
    fChain->GetEntry(fNumEntry );

    fNumEntry ++ ; 

    return kTRUE;
}

Bool_t MReadTree::PostProcess()
{
    //
    // Close File
    //
    //fFile->Close();

    return kTRUE;
}

Bool_t MReadTree::GetEvent() 
{
    //
    // Get the Event with the current EventNumber fNumEntry
    //
    fChain->GetEntry(fNumEntry);

    return kTRUE;
}

Bool_t MReadTree::DecEventNum(UInt_t dec)
{
    //
    // Decrease the number of the event which is read by Process() next
    // by one or more
    //

    //!
    //! this function makes Process() read the event one (or more) before
    //! the actual position (event) in the tree
    //!
    if (fNumEntry < dec/*+1*/)
    {
        *fLog << "MReadTree::SetPrevEvent: WARNING: " << fNumEntry/*-1*/ << "-" << dec << " out of Range." << endl;
        return kFALSE;
    }

    fNumEntry -= dec/*+1*/;
    return kTRUE;
}

Bool_t MReadTree::IncEventNum(UInt_t inc)
{
    //
    // Increase the number of the event which is read by Process() next
    // by one or more
    //

    //!
    //! this function makes Process() read the next (or more) after
    //! the actual position (event) in the tree
    //! (Be careful: IncEventNum() or IncEventNum(1) does not chenge anything
    //!  in the standard behaviour of the task)
    //!
    if (fNumEntry+inc/*-1*/ >= fNumEntries)
    {
        *fLog << "MReadTree::SkipEvents: WARNING: " << fNumEntry/*-1*/ << "+" << inc << " out of Range." << endl;
        return kFALSE;
    }

    fNumEntry += inc/*-1*/;
    return kTRUE;
}

Bool_t MReadTree::SetEventNum(UInt_t nr)
{
    //
    // this function makes Process() read event number nr next
    //
    if (nr>=fNumEntries)
    {
        *fLog << "MReadTree::SetEventNum: WARNING: " << nr << " out of Range." << endl;
        return kFALSE;
    }

    fNumEntry = nr;
    return kTRUE;
}

