/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
// MParContainer                                                            //
//                                                                          //
// The MParContainer class is the base class for all MARS parameter         //
// containers. At the moment it is almost the same than ROOT's TNamed.      //
// A TNamed contains the essential elements (name, title)                   //
// to identify a derived object in lists like our MParList or MTaskList.    //
// The main difference is that the name and title isn't stored and read     //
// to and from root files ("//!")                                           //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////
#include "MParContainer.h"

#include <TClass.h>      // IsA
#include <TROOT.h>       // TROOT::Identlevel
#include <TVirtualPad.h> // gPad

#include "MLog.h"

ClassImp(MParContainer);

MParContainer::MParContainer(const MParContainer &named)
{
    //
    //  MParContainer copy ctor.
    //
    *fName = *(named.fName);
    *fTitle = *(named.fTitle);
}

MParContainer& MParContainer::operator=(const MParContainer& rhs)
{
    //
    //  MParContainer assignment operator.
    //
    if (this != &rhs) {
        TObject::operator=(rhs);
        *fName  = *(rhs.fName);
        *fTitle = *(rhs.fTitle);
    }
    return *this;
}

Int_t MParContainer::Compare(TObject *obj)
{
    //
    //  Compare two MParContainer objects. Returns 0 when equal, -1 when this is
    //  smaller and +1 when bigger (like strcmp).
    //
    if (this == obj) return 0;
    return fName->CompareTo(obj->GetName());
}

void MParContainer::Copy(TObject &obj)
{
    //
    //  Copy this to obj.
    //
    TObject::Copy(obj);
    *(((MParContainer&)obj).fName)  = *fName;
    *(((MParContainer&)obj).fTitle) = *fTitle;
}

void MParContainer::FillBuffer(char *&buffer)
{
    //
    //  Encode MParContainer into output buffer.
    //
    fName->FillBuffer(buffer);
    fTitle->FillBuffer(buffer);
}

void MParContainer::ls(Option_t *)
{
    //
    //  List MParContainer name and title.
    //
    TROOT::IndentLevel();
    *fLog <<"OBJ: " << IsA()->GetName() << "\t" << GetName() << "\t" << GetTitle() << " : "
        << Int_t(TestBit(kCanDelete)) << endl;
}

void MParContainer::Print(Option_t *)
{
    //
    //  Print MParContainer name and title.
    //
    *fLog <<"OBJ: " << IsA()->GetName() << "\t" << GetName() << "\t" << GetTitle() << endl;
}

void MParContainer::SetName(const char *name)
{
    //
    //  Change (i.e. set) the name of the MParContainer.
    //  WARNING !!
    //  If the object is a member of a THashTable, THashList container
    //  The HashTable must be Rehashed after SetName
    //  For example the list of objects in the current directory is a THashList
    //
    *fName = name;
    if (gPad && TestBit(kMustCleanup)) gPad->Modified();
}

void MParContainer::SetObject(const char *name, const char *title)
{
    //
    //  Change (i.e. set) all the MParContainer parameters (name and title).
    //  See also WARNING in SetName
    //
    *fName  = name;
    *fTitle = title;
    if (gPad && TestBit(kMustCleanup)) gPad->Modified();
}

void MParContainer::SetTitle(const char *title)
{
    //
    //  Change (i.e. set) the title of the MParContainer.
    //
    *fTitle = title;
    if (gPad && TestBit(kMustCleanup)) gPad->Modified();
}

Int_t MParContainer::Sizeof() const
{
    //
    //  Return size of the MParContainer part of the TObject.
    //
    Int_t nbytes = fName->Sizeof() + fTitle->Sizeof();
    return nbytes;
}
