
#include "GUI.h"

//---------------------------------------------------------------------
//************************ All functions ****************************
//-------------------------------------------------------------------

// +++ Open file dialog +++
void ddd::FileDialog(void) {
QString Filename = QFileDialog::getOpenFileName(this,
  "Open raw file", INITIAL_DIRECTORY, "Raw data files (*.raw);; All files (*)");
  if (Filename != NULL) {
    FilenameBox->setText(Filename);
    OpenDatafile();
  }
}

// +++ Open selected file and read run header +++
void ddd::OpenDatafile() {
  
  CloseDatafile();	// Close previous file if open

  // Write run header to temporary file  
  FILE *Tmpfile = tmpfile();
  if(Tmpfile==NULL) {
    QMessageBox::warning(this, "ddd Message","Could not open temporary file.",QMessageBox::Ok);
    CloseDatafile();
    return;
  }

  switch (RD->OpenDataFile(FilenameBox->text().toAscii().data(), Tmpfile)) {
    case CTX_FOPEN:   QMessageBox::warning(this, "ddd Message","Could not open file.",QMessageBox::Ok);
                      return;
    case CTX_RHEADER: QMessageBox::warning(this, "ddd Message","Could not read run header.",QMessageBox::Ok);
		      return;
    case CTX_BSTRUCT: QMessageBox::warning(this, "ddd Message","Could not read board structures.",QMessageBox::Ok);
		      return;
    default: break;
  }
  if (RD->RHeader->MagicNum == MAGICNUM_OPEN) {
    QMessageBox::warning(this, "ddd Message","Magic number in run header indicates that the file has not been closed properly.",QMessageBox::Ok);
  }
  if (RD->RHeader->MagicNum == MAGICNUM_ERROR) {
    QMessageBox::warning(this, "ddd Message","Magic number in run header indicates that an error occurred while writing the file.",QMessageBox::Ok);
  }
  
  rewind(Tmpfile);
  QTextStream in(Tmpfile);
  QString text = in.readAll();
  RunHeaderDisplay->setPlainText(text);
  fclose(Tmpfile);

  // Enable spin boxes, set ranges and display first event
  EventNo->setEnabled(true);
  ChannelNo->setEnabled(true);
  BoardNo->setEnabled(true);
  PixelID->setEnabled(true);
  M0Start->setEnabled(true);
  M0Stop->setEnabled(true);
  EventNo->setValue(1);
  EventNo->setRange(1, RD->RHeader->Events);
  M0Start->setValue(0);
  M0Stop->setValue((RD->RHeader->Samples)-1);
  M0Start->setRange(0,(RD->RHeader->Samples)-1);  
  M0Stop->setRange(0,(RD->RHeader->Samples)-1);
  ChannelNo->setValue(0);
  ChannelNo->setRange(0, RD->RHeader->NChannels*RD->RHeader->NChips-1);
  BoardNo->setValue(0);
  BoardNo->setRange(0, RD->RHeader->NBoards-1);
  DisplayEvent();
}

// +++ Close data file file if open, delete event header and disable spin boxes and displays  +++
void ddd::CloseDatafile() {
  if(RD->CloseDataFile()==CTX_OK) {
    EventNo->setEnabled(false);
    ChannelNo->setEnabled(false);
    BoardNo->setEnabled(false);
    PixelID->setEnabled(false);
    M0Start->setEnabled(false);
    M0Stop->setEnabled(false);
    RunHeaderDisplay->clear();
    EventHeaderDisplay->clear();
    Signal->hide();
  }
}
          
// +++ Read event header and display event (only called if Datafile is open) +++
void ddd::DisplayEvent(int) {
 
  PixelID->setText(PixMap->DRS_to_Pixel(BoardNo->value(),ChannelNo->value()/10,ChannelNo->value()%10).c_str());
  if(Socket->state() == QAbstractSocket::ConnectedState) return; // do not execute if socket is open
  
  // Read event
  FILE *Tmpfile = tmpfile();

   switch(RD->ReadEvent(EventNo->value(), Tmpfile)) {
    case !CTX_OK:  
      QMessageBox::warning(this, "ddd Warning","Could not read event.",QMessageBox::Ok);
      EventHeaderDisplay->clear();
      break;
    default:  // Print event header and trigger cell information from event data
      rewind(Tmpfile);
      QTextStream in(Tmpfile);
      QString text = in.readAll();
      text.append("\nTrigger cells: ");
      for (unsigned int i=0; i<RD->RHeader->NBoards*RD->RHeader->NChips; i++) {
        QString a;
        text.append(a.sprintf("%d ", *((int *)RD->Data + i)));
      }
      EventHeaderDisplay->setPlainText(text);
      
      // Case data in double format required by qwt library
      double* x = new double [RD->RHeader->Samples];
      double* y = new double [RD->RHeader->Samples];

      for (unsigned int i=0; i<RD->RHeader->Samples; i++) {
      	x[i] = (double) (i/RD->BStruct[BoardNo->value()].NomFreq);
        y[i] = (double) *((short *) (RD->Data + RD->RHeader->NBoards*RD->RHeader->NChips*sizeof(int)) + BoardNo->value()*RD->RHeader->NChips*RD->RHeader->NChannels *
		 RD->RHeader->Samples+ChannelNo->value()*RD->RHeader->Samples+i)*RD->BStruct[BoardNo->value()].ScaleFactor;
      }
      Signal->setData(x, y, RD->RHeader->Samples);
      Signal->show();
      Zoomer->setZoomBase(Signal->boundingRect());

      //Get data for M0 display (event based)

      double z[6][6];//36 pixels

      for(unsigned int i=0; i<RD->RHeader->NBoards; i++) {//board loop
	for(unsigned int j=0; j<RD->RHeader->NChips; j++) {//chip loop
	  for(unsigned int k=0; k<RD->RHeader->NChannels; k++) {//channel loop

	    //only interested in M0 data
	    if( ( (i==0 || i==1) && (j<=1) && (k<=7) ) || ( (i==2) && (j==0) && (k<=3) ) ) {

	      //get module, superpixel and pixel number from pixel name

	      std::string pixelname = PixMap->DRS_to_Pixel(i,j,k);
	      char pixelname_copy[256];
	      memset(pixelname_copy,'\0',256);
	      pixelname.copy(pixelname_copy, 256);
	      
	      char delim[] = "-";
	      char *buffer = NULL;
	      int module = -1;
	      int superpixel = -1;
	      int pixel = -1;
	      
	      buffer = strtok(pixelname_copy, delim);
	      module = atoi(buffer);
	      buffer = strtok(NULL, delim);
	      superpixel = atoi(buffer);
	      buffer = strtok(NULL, delim);
	      pixel = atoi(buffer);
	      	
	      //usual M0 mapping
	      //int binx = 5-(int((superpixel-1)/3)*2)-(int((pixel%4)/2));
	      //int biny = 5-(((superpixel-1)%3)*2)-(int((pixel-1)/2));
	      	      
	      //M0 upside down
	      int binx = 5-(5-(int((superpixel-1)/3)*2)-(int((pixel%4)/2)));
	      int biny = 5-(5-(((superpixel-1)%3)*2)-(int((pixel-1)/2)));

	      //search maximum sample amplitude within user specified window
	      //start bin is always smaller than stop bin (taken care of by updated ranges)
	      int StartBin = (int)(M0Start->value());
	      int StopBin = (int)(M0Stop->value());
	      
	      for(int l=StartBin; l<=StopBin; l++){
		  
		float sample = *((short *) (RD->Data + RD->RHeader->NBoards*RD->RHeader->NChips*sizeof(int)) + 
				 i*RD->RHeader->NChips*RD->RHeader->NChannels*RD->RHeader->Samples+
				 j*RD->RHeader->NChannels*RD->RHeader->Samples+
				 k*RD->RHeader->Samples+
				 l)*RD->BStruct[i].ScaleFactor;
		
		if (sample > z[binx][biny]) {
		  z[binx][biny]=sample;
		}
		
	      }//sample loop

	    }//only M0 data
	    
	  }//channel loop
	}//chip loop
      }//board loop
      
      //fill data to M0 display (event based)
      Signal2D->setData(SpectrogramDataM0(z));
      Graph2D->axisWidget(QwtPlot::yRight)->setColorMap(Signal2D->data().range(),Signal2D->colorMap());
      Graph2D->setAxisScale(QwtPlot::yRight,Signal2D->data().range().minValue(),Signal2D->data().range().maxValue() );
      Graph2D->replot();
      //Signal2D->show();

      //update ranges for start and stop bin to avoid startbin > stopbin
      M0Start->setRange(0, M0Stop->value());  
      M0Stop->setRange(M0Start->value(),(RD->RHeader->Samples)-1);

      delete[] x;	delete[] y;

  }
  if(Tmpfile!=NULL) fclose(Tmpfile);
}

// +++ Open sub window handling the socket interface +++
void ddd::OpenSocketWindow() {

  if(SocketWindow->isVisible()) SocketWindow->hide();
  else SocketWindow->show();
}

// +++ Open sub window for M0 Display +++
void ddd::OpenM0Window() {

  if(M0Window->isVisible()) M0Window->hide();
  else M0Window->show();
}


// +++ Acquire data through socket (acquire botton only available if socket exists) +++
void ddd::GetSignalFromSocket() {
  char Command[MAX_COM_SIZE];

  GetButton->setEnabled(false);
  WaitForData = true;
  sprintf(Command, "read %d %d %d restart", BoardNo->value(), ChannelNo->value()/10, ChannelNo->value()%10);
  Socket->write(Command);
}

// Quit application when clicking close button on window
void ddd::closeEvent(QCloseEvent *) {
  qApp->quit();
}

// +++ Connecting or disconnecting from client +++
void ddd::MakeConnection() {

  if(Socket->state() == QAbstractSocket::ConnectedState) {
    ManualDisconnect = true;
    Socket->disconnectFromHost();
  }
  else {
    if (RD->IsFileOpen() && QMessageBox::question(this, "ddd Request","Connecting will close current data file. Proceed?", 
        QMessageBox::No, QMessageBox::Yes) != QMessageBox::Yes) return;    
    Socket->connectToHost(IPAddress->text(),Port->value());
    Connect->setEnabled(false);  // While waiting for connection, button not available
    Socket->waitForConnected(SOCKET_TIMEOUT);
    Connect->setEnabled(true);
    if(Socket->state() != QAbstractSocket::ConnectedState)
      QMessageBox::warning(this, "ddd Message","Could not connect to host.",QMessageBox::Ok);
    else {
      Connect->setText("Disconnect");
      ConnectAction->setText("Disconnect");
      Port->setEnabled(false);
      IPAddress->setEnabled(false);
      Command->setEnabled(true);
      GetButton->setEnabled(true);
      ManualDisconnect = false;

      OpenAction->setEnabled(false);      
      FilenameBox->setEnabled(false);
      LoadButton->setEnabled(false);
      FilenameBox->clear();
      CloseDatafile();

      ChannelNo->setEnabled(true);
      BoardNo->setEnabled(true);
      PixelID->setEnabled(true);
      ChannelNo->setRange(0, 65535);
      BoardNo->setRange(0, 65535);
      
      TabWidget->setTabEnabled(1,false);
      TabWidget->setTabEnabled(2,false);

      RunHeaderDisplay->clear();
      EventHeaderDisplay->clear();
      Signal->hide();
    }
  }
}

// +++ Send command to socket (command button available only if socket existing) +++
void ddd::SendToSocket() {
  Socket->write(Command->text().toAscii());
  Command->clear();
}

// +++ Read data from socket and display +++
void ddd::ReadFromSocket() {
  // Check if channel data expected and error message arrived
  QByteArray Data = Socket->readAll();
  if (WaitForData && Data.contains("Error")) {
    WaitForData = false;
    GetButton->setEnabled(true);
    QMessageBox::warning(this, "ddd Message","Could not read waveform data from socket.",QMessageBox::Ok);
    return;
  }

  // Check if channel data were transmitted, if yes and waiting for data, extract and plot them
  SocketOutput->insertPlainText(Data);
  QString Text = SocketOutput->document()->toPlainText().trimmed();
  if (WaitForData && Text.endsWith(QLatin1String("==END=="))) {
    // Extract text between ==START== and ==END==
    QByteArray Data=Text.mid(Text.lastIndexOf("==START==")+9,Text.length()-Text.lastIndexOf("==START==")-16).toAscii();

    char *NextNumber = strtok(Data.data()," ");  // Number of entries that follow
    int Count=0, NumberOfEntries = atoi(NextNumber);
    double *x = new double [NumberOfEntries];
    double *y = new double [NumberOfEntries];

    // Convert all entries (separated by a whitespace) to numbers
    while((NextNumber=strtok(NULL, " "))!=NULL && Count<NumberOfEntries)
      *(y+Count++) = atof(NextNumber);
    if (Count==NumberOfEntries && NextNumber!=0)
      QMessageBox::warning(this, "ddd Message","Found too many numbers in data block, truncated.",QMessageBox::Ok);
    // Apply sampling frequency and scaling factor 
    for(int i=2; i<Count; i++) {
      x[i] = (i-2) / y[0];
      y[i] = y[i] * y[1];
    }
    if(NumberOfEntries>2) {
      Signal->setData(x+2, y+2, NumberOfEntries-2); // Copies data, arrays can be deleted afterwards
      Signal->show();
      Zoomer->setZoomBase(Signal->boundingRect());
    }  
    delete[] x;   delete[] y;

    if(ContinuousBox->isChecked()) {
      usleep(100000);	// Wait to limit maximum update rate
      GetSignalFromSocket();
    }
    else {
      WaitForData = false;
      GetButton->setEnabled(true);
    }
  }
}

// +++ Disconnect from socket +++
void ddd::GotDisconnected() {
  Connect->setText("Connect");
  ConnectAction->setText("Connect");
  Port->setEnabled(true);
  IPAddress->setEnabled(true);
  Command->setEnabled(false);

  GetButton->setEnabled(false);
  FilenameBox->setEnabled(true);
  OpenAction->setEnabled(true);    
  LoadButton->setEnabled(true);
  ChannelNo->setEnabled(false);
  BoardNo->setEnabled(false);
  PixelID->setEnabled(false);
  Signal->hide();
  TabWidget->setTabEnabled(1, true);
  TabWidget->setTabEnabled(2, true);
  
  SocketOutput->clear();
  if(!ManualDisconnect) QMessageBox::warning(this, "ddd Message","Socket disconnected, maybe host gone.",QMessageBox::Ok);
}

// +++ Translate pixel ID +++
void ddd::TranslatePixelID() {
  
  int Board = PixMap->Pixel_to_DRSboard(PixelID->text().toStdString());
  int Chip = PixMap->Pixel_to_DRSchip(PixelID->text().toStdString());
  int Channel = PixMap->Pixel_to_DRSchannel(PixelID->text().toStdString());

  if(Board>=BoardNo->minimum() && Board<=BoardNo->maximum() && 
     (Chip*10+Channel)>=ChannelNo->minimum() && (Chip*10+Channel)<=ChannelNo->maximum()) { 
    BoardNo->setValue(Board);
    ChannelNo->setValue(Chip*10+Channel);
  }
  else if(Board==999999999) QMessageBox::warning(this, "ddd Message","Pixel ID unknown.",QMessageBox::Ok);
  else QMessageBox::warning(this, "ddd Message","Pixel ID out of current range.",QMessageBox::Ok);
}


//------------------------------------------------------------------
//**************************** All menus ***************************
//------------------------------------------------------------------

void ddd::MenuSave() {
  QString Filename = QFileDialog::getSaveFileName(this,
     "Filename of image", "/home/ogrimm/ddd", "Image files (*.bmp *.jpg *.png *.ppm *.tiff *.xbm *.xpm);;All files (*)");
  if (Filename.length()>0) {
    QPixmap Pixmap = QPixmap::grabWidget(Graph);
    if(!Pixmap.save(Filename)) {
      QMessageBox::warning(this, "ddd Message","Could not write image file.",QMessageBox::Ok);
      remove(Filename.toAscii().data());
    }
  }
}

void ddd::MenuPrint() {
  QPrinter *Printer = new QPrinter;
  QPrintDialog *PrintDialog = new QPrintDialog(Printer, this);
  if (PrintDialog->exec() == QDialog::Accepted) {
    QPainter Painter(Printer);
    QPixmap Pixmap = QPixmap::grabWidget(Graph);
    Painter.drawPixmap(0, 0, Pixmap);
  }
  delete Printer;	delete PrintDialog;
}

void ddd::MenuHelp() {
  QMessageBox Message;
  Message.setText("The DRS Data Display program can be used for two purposes:\n\n"
     "1. Reading and displaying the content of a raw data file written by the drsdaq program\n"
     "2. Acquiring and displaying online data from a running drsdaq program via the socket interface\n\n"
     "With an established socket connection, displaying of raw data files is disabled.");
   Message.setWindowTitle("ddd Help"); 
   Message.exec();
}

void ddd::MenuAbout() {
  QMessageBox::about(this, "ddd About","DRS Data Display\n\n"
    "Written by Oliver Grimm, IPP, ETH Zurich\n"
    "This version compiled "__DATE__".\n\n"
    "Graphical user interface implemented with Qt.\n"
    "Bug reports to oliver.grimm@phys.ethz.ch.");
}
