/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2004 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2004
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJCalibrateSignal
//
// This class is reading the output written by callisto. It calibrates
// signal and time.
//
// The signal and time extractors are read from the callisto-output. In
// pricipal you could overwrite these default using the resource file,
// but this is NOT recommended!
//
/////////////////////////////////////////////////////////////////////////////
#include "MJCalibrateSignal.h"

#include <TEnv.h>
#include <TFile.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MDirIter.h"
#include "MRunIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MStatusDisplay.h"

#include "MGeomCam.h"
#include "MHCamEvent.h"
//#include "MHCamEventTH.h"
#include "MPedestalCam.h"
#include "MBadPixelsCam.h"
#include "MArrivalTimeCam.h"

#include "MCalibrationQECam.h"
#include "MCalibrationBlindCam.h"
#include "MCalibrationChargeCam.h"
#include "MCalibrationRelTimeCam.h"
#include "MCalibrationChargePINDiode.h"
#include "MCalibrationHiLoCam.h"
#include "MCalibrationPulseTimeCam.h"
#include "MCalibrationPatternDecode.h"

#include "MCalibrationChargeCalc.h"
#include "MCalibrationRelTimeCalc.h"

#include "MCalibrationIntensityChargeCam.h"
#include "MCalibrationIntensityBlindCam.h"
#include "MCalibrationIntensityRelTimeCam.h"
#include "MCalibrationIntensityQECam.h"
#include "MCalibrationIntensityConstCam.h"
#include "MBadPixelsIntensityCam.h"

#include "MHCalibrationChargeCam.h"
#include "MHCalibrationChargeBlindCam.h"
#include "MHCalibrationChargePINDiode.h"
#include "MHCalibrationRelTimeCam.h"
#include "MHCalibrationPulseTimeCam.h"
#include "MHCalibrationHiLoCam.h"
#include "MHCamera.h"

#include "MCalibCalcFromPast.h"

#include "MReadReports.h"
#include "MReadMarsFile.h"
#include "MRawFileRead.h"
#include "MContinue.h"
#include "MTriggerPatternDecode.h"
#include "MFTriggerPattern.h"
#include "MGeomApply.h"
//#include "MMcPedestalCopy.h"
#include "MPointingPosCalc.h"
#include "MPedCalcFromLoGain.h"
#include "MExtractor.h"
#include "MExtractTimeAndCharge.h"
#include "MExtractPINDiode.h"
#include "MExtractBlindPixel.h"
#include "MFCosmics.h"
#include "MTaskEnv.h"
#include "MCalibrateData.h"
#include "MCalibrateRelTimes.h"
#include "MBadPixelsMerge.h"
#include "MBadPixelsCalc.h"
#include "MBadPixelsTreat.h"
#include "MFillH.h"
#include "MWriteRootFile.h"

ClassImp(MJCalibrateSignal);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets:
// - fIsInterlaced to kTRUE
// - fIsRelTimesUpdate to kFALSE
// - fIsHiLoCalibration to kFALSE
// - fPulsePosCheck to kTRUE
//
MJCalibrateSignal::MJCalibrateSignal(const char *name, const char *title)
    : fExtractor(0), fIsInterlaced(kTRUE), fIsRelTimesUpdate(kFALSE)
{
    fName  = name  ? name  : "MJCalibrateSignal";
    fTitle = title ? title : "Tool to calibrate data";

    SetPulsePosCheck();
    //fCruns = NULL;
}

MJCalibrateSignal::~MJCalibrateSignal()
{
    if (fExtractor)
        delete fExtractor;
}

void MJCalibrateSignal::SetExtractor(const MExtractor *ext)
{
    if (fExtractor)
        delete fExtractor;

    fExtractor = ext ? (MExtractor*)ext->Clone() : NULL;
}

Bool_t MJCalibrateSignal::WriteResult(TObjArray &cont)
{
  if (IsNoStorage())
    return kTRUE;

  const TString name(Form("signal%08d.root", fSequence.GetSequence()));
  return WriteContainer(cont, name);
}

Bool_t MJCalibrateSignal::ReadCalibration(TObjArray &l, MBadPixelsCam &cam, MExtractor* &ext2, MExtractor* &ext3, TString &geom) const
{
    TString fname = Form("%s/calib%08d.root", fPathIn.Data(), fSequence.GetSequence());

    *fLog << inf << "Reading from file: " << fname << endl;

    TFile file(fname, "READ");
    if (!file.IsOpen())
    {
        *fLog << err << dbginf << "ERROR - Could not open file " << fname << endl;
        return kFALSE;
    }

    TObject *o = file.Get("ExtractSignal");
    if (o && !o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractSignal read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext3 = o ? (MExtractor*)o->Clone() : NULL;

    o = file.Get("ExtractTime");
    if (o && !o->InheritsFrom(MExtractor::Class()))
    {
        *fLog << err << dbginf << "ERROR - ExtractTime read from " << fname << " doesn't inherit from MExtractor!" << endl;
        return kFALSE;
    }
    ext2 = o ? (MExtractor*)o->Clone() : NULL;
    if (!ext3 && !ext2)
    {
        *fLog << err << dbginf << "ERROR - Neither ExtractSignal nor ExrtractTime found in " << fname << "!" << endl;
        return kFALSE;
    }

    o = file.Get("MGeomCam");
    if (o && !o->InheritsFrom(MGeomCam::Class()))
    {
        *fLog << err << dbginf << "ERROR - MGeomCam read from " << fname << " doesn't inherit from MGeomCam!" << endl;
        return kFALSE;
    }
    geom = o ? o->ClassName() : "";

    TObjArray cont(l);
    cont.Add(&cam);
    return ReadContainer(cont);
}

// --------------------------------------------------------------------------
//
// MJCalibration allows to setup several option by a resource file:
//   MJCalibrateSignal.RawData: yes,no
//
// For more details see the class description and the corresponding Getters
//
Bool_t MJCalibrateSignal::CheckEnvLocal()
{
    SetInterlaced(GetEnv("Interlaced", fIsInterlaced));
    SetRelTimesUpdate(GetEnv("RelTimesUpdate", fIsRelTimesUpdate));

    return MJCalib::CheckEnvLocal();
}

Bool_t MJCalibrateSignal::Process(MPedestalCam &pedcamab, MPedestalCam &pedcambias,
                                  MPedestalCam &pedcamextr)
{
    if (!fSequence.IsValid())
    {
	  *fLog << err << "ERROR - Sequence invalid..." << endl;
	  return kFALSE;
    }

    *fLog << inf;
    fLog->Separator(GetDescriptor());
    *fLog << "Calculate calibrated data from Sequence #";
    *fLog << fSequence.GetSequence() << endl << endl;


    //if (!CheckEnv())
    //    return kFALSE;

    CheckEnv();

    // --------------------------------------------------------------------------------

    MDirIter iter;
    if (fSequence.IsValid())
    {
        const Int_t n0 = fSequence.SetupDatRuns(iter, fPathData, IsUseRawData());
        const Int_t n1 = fSequence.GetNumDatRuns();
        if (n0==0)
        {
            *fLog << err << "ERROR - No input files of sequence found!" << endl;
            return kFALSE;
        }
        if (n0!=n1)
        {
            *fLog << err << "ERROR - Number of files found (" << n0 << ") doesn't match number of files in sequence (" << n1 << ")" << endl;
            if (fLog->GetDebugLevel()>4)
            {
                *fLog << dbg << "Files which are searched:" << endl;
                iter.Print();
            }
            return kFALSE;
        }
    }

    // Read File
    MCalibrationIntensityChargeCam      ichcam;
    MCalibrationIntensityQECam          iqecam;
    MCalibrationIntensityBlindCam       iblcam;
    MCalibrationIntensityRelTimeCam     itmcam;
    MCalibrationIntensityConstCam       icncam;
    MBadPixelsIntensityCam              ibdcam;

    MHCalibrationChargeCam      hchacam;
    MHCalibrationChargeBlindCam hbndcam;
    MHCalibrationChargePINDiode hpndiod;
    MHCalibrationRelTimeCam     hrelcam;
    MHCalibrationHiLoCam        hilocam;
    MHCalibrationPulseTimeCam   hpulcam;

    hchacam.SetOscillations(kFALSE);
    hbndcam.SetOscillations(kFALSE);
    hrelcam.SetOscillations(kFALSE);

    MCalibrationChargeCam      calcam;
    MCalibrationQECam          qecam;
    MCalibrationBlindCam       bndcam;
    MCalibrationChargePINDiode pind;
    MCalibrationRelTimeCam     tmcam;
    MCalibrationHiLoCam        hilcam;
    MCalibrationPulseTimeCam   pulcam;

    MBadPixelsCam              badpix;

    TObjArray interlacedcont;
    if (fIsInterlaced)
    {
        interlacedcont.Add(&ichcam);
        interlacedcont.Add(&iqecam);
        interlacedcont.Add(&iblcam);
        interlacedcont.Add(&itmcam);
        interlacedcont.Add(&ibdcam);
        interlacedcont.Add(&icncam);
        interlacedcont.Add(&hchacam);
	if (IsUseBlindPixel())
            interlacedcont.Add(&hbndcam);
        if (IsUsePINDiode())  
            interlacedcont.Add(&hpndiod);
        if (fIsRelTimesUpdate)
            interlacedcont.Add(&hrelcam);
    }

    MExtractor *extractor1=fExtractor;
    MExtractor *extractor2=0;
    MExtractor *extractor3=0;
    TString geom;

    TObjArray calibcont;
    calibcont.Add(&calcam);
    calibcont.Add(&qecam);
    calibcont.Add(&bndcam);
    calibcont.Add(&tmcam);
    if (IsUseBlindPixel())
        calibcont.Add(&bndcam);

    if (!ReadCalibration(calibcont, badpix, extractor2, extractor3, geom))
        return kFALSE;

    *fLog << all;
    if (!geom.IsNull())
        *fLog << inf << "Camera geometry found in file: " << geom << endl;
    else
        *fLog << inf << "No Camera geometry found using default <MGeomCamMagic>" << endl;

    if (extractor3)
    {
        *fLog << underline << "Signal Extractor found in calibration file" << endl;
        extractor3->Print();
        *fLog << endl;
    }
    else
        *fLog << inf << "No Signal Extractor: ExtractSignal in file." << endl;


    if (extractor1)
    {
      *fLog << underline << "Modified Signal Extractor set by user." << endl;
      extractor1->Print();
      *fLog << endl;
    }
    else
    {
        *fLog << inf << "No modified Signal Extractor set by user... using ExtractSignal." << endl;
        extractor1 = extractor3 ? (MExtractor*)extractor3->Clone() : 0;
    }

    if (extractor2)
    {
        *fLog << underline << "Time Extractor found in calibration file" << endl;
        extractor2->Print();
        *fLog << endl;
    }
    else
        *fLog << inf << "No Time Extractor: ExtractTime in file." << endl;

    // This is necessary for the case in which it is not in the files
    MBadPixelsCam badcam;
    icncam.SetBadPixels(&badpix);

    // Setup Parlist
    MParList plist;
    plist.AddToList(this); // take care of fDisplay!
    plist.AddToList(&badcam);
    plist.AddToList(&hilcam);
    plist.AddToList(&calibcont);       // Using AddToList(TObjArray *)
    plist.AddToList(&interlacedcont);  // Using AddToList(TObjArray *)
    plist.AddToList(&pulcam);

    // Setup Tasklist
    MTaskList tlist;
    plist.AddToList(&tlist);

    MReadReports readreal;
    readreal.AddTree("Events", "MTime.", MReadReports::kMaster);
    readreal.AddTree("Trigger");
    readreal.AddTree("Camera");
    readreal.AddTree("Drive");
    readreal.AddTree("CC");
    readreal.AddTree("Currents");

    MReadMarsFile readmc("Events");
    readmc.DisableAutoScheme();

    MRawFileRead rawread(NULL);

    MRead *read = 0;
    switch (GetDataFlag())
    {
    case kIsUseRawData:  read = &rawread;  break;
    case kIsUseMC:       read = &readmc;   break;
    case kIsUseRootData: read = &readreal; break;
    }
    read->AddFiles(iter);

    const TString fname(Form("%s{s/_D_/_Y_}{s/\\.raw$/.root}{s/\\.raw\\.gz$/.root}", fPathOut.Data()));

    
    // Skips MC which have no contents. This are precisely the
    // events which fullfilled the MC Lvl1 trigger and an
    // arbitrary cut (typically at 50phe) to speed up simulation
    MContinue contmc("MRawEvtData.GetNumPixels<0.5", "ContEmptyMC");

    //MPointingPosInterpolate pextr;
    //pextr.AddFiles(&iter);
    
    MGeomApply             apply; // Only necessary to create geometry
    if (!geom.IsNull())
        apply.SetGeometry(geom);
    MBadPixelsMerge        merge(&badpix);

    // Make sure that pedcamab has the correct name
    pedcamab.SetName("MPedestalFundamental");
    pedcamextr.SetName("MPedestalFromExtractorRndm");
    pedcambias.SetName("MPedestalFromExtractor");
    plist.AddToList(&pedcamextr);
    plist.AddToList(&pedcambias);
    plist.AddToList(&pedcamab);

    MArrivalTimeCam timecam;
    plist.AddToList(&timecam);

    // Check for interleaved events
    MCalibrationPatternDecode caldec;
    MTriggerPatternDecode     decode;

    // This will make that for data with version less than 5, where
    // trigger patterns were not yet correct, all the events in the real
    // data file will be processed. In any case there are no interleaved
    // calibration events in such data, so this is fine.
    MFTriggerPattern ftp;
    ftp.SetDefault(kTRUE);
    //    ftp.RequireCalibration();
    ftp.DenyCalibration();
    ftp.DenyPedestal();
    //    ftp.DenyPinDiode();
    ftp.SetInverted();

    // This will skip interleaved calibration events and pedestal events (if any)
    // --> tlist2
    MContinue conttp(&ftp, "ContTrigPattern");

    // Do signal and pedestal calculation
    MPedCalcFromLoGain     pedlo1("MPedCalcFundamental");
    pedlo1.SetPedestalUpdate(kTRUE);
    pedlo1.SetNamePedestalCamOut("MPedestalFundamental");

    MPedCalcFromLoGain     pedlo2("MPedCalcWithExtractorRndm");
    pedlo2.SetPedestalUpdate(kTRUE);
    pedlo2.SetRandomCalculation(kTRUE);
    pedlo2.SetNamePedestalCamIn("MPedestalFundamental");
    pedlo2.SetNamePedestalCamOut("MPedestalFromExtractorRndm");

    MPedCalcFromLoGain     pedlo3("MPedCalcWithExtractor");
    pedlo3.SetPedestalUpdate(kTRUE);
    pedlo3.SetRandomCalculation(kFALSE);
    pedlo3.SetNamePedestalCamIn("MPedestalFundamental");
    pedlo3.SetNamePedestalCamOut("MPedestalFromExtractor");

    if (extractor1)
    {
        extractor1->SetPedestals(&pedcamab);

        if (extractor1->InheritsFrom("MExtractTimeAndCharge"))
        {
            pedlo2.SetExtractor((MExtractTimeAndCharge*)extractor1);
            pedlo3.SetExtractor((MExtractTimeAndCharge*)extractor1);
            const Int_t win = ((MExtractTimeAndCharge*)extractor1)->GetWindowSizeHiGain();
            pedlo1.SetExtractWindow(15, win);
            pedlo2.SetExtractWindow(15, win/*obsolete*/);
            pedlo3.SetExtractWindow(15, win/*obsolete*/);
        }
        else
        {
            // FIXME: How to get the fixed value 15 automatically?
            const Int_t f = (Int_t)(15.5+extractor1->GetHiGainFirst());
            const Int_t n = (Int_t)(15.5+extractor1->GetNumHiGainSamples());
            pedlo1.SetExtractWindow(f, n);
            pedlo2.SetExtractWindow(f, n);
            pedlo3.SetExtractWindow(f, n);
        }
    }
    if (extractor2)
      extractor2->SetPedestals(&pedcamab);

    if (extractor3)
      extractor3->SetPedestals(&pedcamab);

    MFCosmics fcosmics;
    fcosmics.SetNamePedestalCam("MPedestalFundamental");
    MContinue contcos(&fcosmics, "ContTrigEvts");
    contcos.SetInverted();

    //MMcPedestalCopy pcopy;
    MTaskEnv taskenv1("ExtractSignal");
    MTaskEnv taskenv2("ExtractTime");
    MTaskEnv taskenv3("ExtractInterlaced");
    taskenv1.SetDefault(extractor1);
    taskenv2.SetDefault(extractor2);
    taskenv3.SetDefault(extractor3);

    // 
    // This is new calibration to photo-electrons, hard-coded
    // as decided at the Wuerzburg software meeting 26.01.05
    //
    MCalibrateData calib;
    calib.SetSignalType(MCalibrateData::kPhe);
    calib.AddPedestal("Fundamental");
    calib.AddPedestal("FromExtractor");
    calib.AddPedestal("FromExtractorRndm");
    calib.SetPedestalFlag(MCalibrateData::kEvent);
    //----------------------------------------------------------

    MExtractPINDiode        pinext;
    MExtractBlindPixel      bldext;

    MFTriggerPattern        fcalib("CalibFilter");
    fcalib.SetDefault(kFALSE);
    fcalib.RequireCalibration();

    MCalibrationChargeCalc  chcalc;
    chcalc.SetExtractor(extractor3);

    MCalibrationRelTimeCalc recalc;
    MCalibCalcFromPast      pacalc;
    chcalc.SetOutputFile("");
    recalc.SetOutputFile("");

    pinext.SetPedestals(&pedcamab);
    bldext.SetPedestals(&pedcamab);
    chcalc.SetPedestals(&pedcamextr);

    pacalc.SetChargeCalc(&chcalc);
    if (fIsRelTimesUpdate)
      pacalc.SetRelTimeCalc(&recalc);
    pacalc.SetCalibrate(&calib);

    //
    // Intensity Calibration histogramming
    //
    MFillH filpin(&hpndiod, "MExtractedSignalPINDiode",   "FillPINDiode");
    MFillH filbnd(&hbndcam, "MExtractedSignalBlindPixel", "FillBlindCam");
    MFillH filcam(&hchacam, "MExtractedSignalCam",        "FillChargeCam");
    MFillH filtme(&hrelcam, "MArrivalTimeCam",            "FillRelTime");
    MFillH filhil(&hilocam, "MExtractedSignalCam",        "FillHiLoRatio");
    MFillH filpul(&hpulcam, "MRawEvtData",                "FillPulseTime");
    filpin.SetBit(MFillH::kDoNotDisplay);
    filbnd.SetBit(MFillH::kDoNotDisplay);
    filcam.SetBit(MFillH::kDoNotDisplay);
    filtme.SetBit(MFillH::kDoNotDisplay);
    filhil.SetBit(MFillH::kDoNotDisplay);
    filpul.SetBit(MFillH::kDoNotDisplay);

    MCalibrateRelTimes caltm;
    MBadPixelsCalc     bpcal;
    MBadPixelsTreat    treat;

    bpcal.SetNamePedPhotCam("MPedPhotFromExtractor");
    treat.AddNamePedPhotCam("MPedPhotFundamental");
    treat.AddNamePedPhotCam("MPedPhotFromExtractor");
    treat.AddNamePedPhotCam("MPedPhotFromExtractorRndm");
    if (!extractor2 && !extractor1->InheritsFrom("MExtractTimeAndCharge"))
        treat.SetProcessTimes(kFALSE);

    MHCamEvent evt0(  0, "PedFLG",     "Fundamental Pedestal from Lo Gain;;P [cnts/sl]");
    MHCamEvent evt1(  2, "PedRmsFLG",  "RMS from Extractor applied to ped.;;\\sigma_{p} [cnts/sl]");
    MHCamEvent evt2(  0, "Extra'd",    "Extracted Signal;;S [cnts/sl]");
    // MHCamEvent evt3(4, "PedPhot",    "Calibrated Pedestal;;P [phe]");
    MHCamEvent evt4(  5, "PedRMS",     "Calibrated RMS from Extractor applied to ped.;;\\sigma_{p} [phe]");
    MHCamEvent evt5(  0, "Interp'd",   "Interpolated Signal scaled with A/A_{0};;S [phe]");
    MHCamEvent evt6(102, "Unsuitable", "Fraction of unsuitable events per Pixel;;[1]");
    MHCamEvent evt7(  6, "Times",      "Calibrated Arrival Time;;T [fadc sl]");
    MHCamEvent evt8(  0, "Conv",       "Calibration Conv. Factors;;[phe/fadc cnts]");
    MHCamEvent evt9(  7, "PulsePos",   "Pulse Position of cosmics (>15phe);;T");
    MHCamEvent evtH( 11, "PulseHi",    "Pulse Position of Hi-Gain;;T");
    MHCamEvent evtL( 10, "PulseLo",    "Pulse Position of Lo-Gain;;T");
    evt6.SetThreshold();

    MFillH fill0(&evt0, "MPedestalFundamental",          "FillPedFLG");
    MFillH fill1(&evt1, "MPedestalFromExtractorRndm",    "FillPedRmsFLG");
    MFillH fill2(&evt2, "MExtractedSignalCam",           "FillExtracted");
    // MFillH fill3(&evt3, "MPedPhotFundamental",  "FillPedPhot");
    MFillH fill4(&evt4, "MPedPhotFromExtractorRndm",     "FillPedRMS");
    MFillH fill5(&evt5, "MSignalCam",                    "FillInterpolated");
    MFillH fill6(&evt6, "MBadPixelsCam",                 "FillUnsuitable");
    MFillH fill7(&evt7, "MSignalCam",                    "FillTimes");
    MFillH fill8(&evt8, "MCalibrationIntensityConstCam", "FillConv");
    MFillH fill9(&evt9, "MSignalCam",                    "FillPulse");
    MFillH fillH(&evtH, "MSignalCam",                    "FillPulseHi");
    MFillH fillL(&evtL, "MSignalCam",                    "FillPulseLo");

    MTaskEnv fillflorian("FinalFantasy");
    fillflorian.SetDefault();

    // The second rule is for the case reading raw-files!
    MWriteRootFile write(2, fname, fOverwrite?"RECREATE":"NEW");
    // Run Header
    write.AddContainer("MRawRunHeader",             "RunHeaders");
    write.AddContainer("MBadPixelsCam",             "RunHeaders");
    write.AddContainer("MGeomCam",                  "RunHeaders");
    // Monte Carlo Headers
    write.AddContainer("MMcRunHeader",              "RunHeaders", kFALSE);
    write.AddContainer("MMcFadcHeader",             "RunHeaders", kFALSE);
    write.AddContainer("MMcTrigHeader",             "RunHeaders", kFALSE);
    write.AddContainer("MMcConfigRunHeader",        "RunHeaders", kFALSE);
    write.AddContainer("MMcCorsikaRunHeader",       "RunHeaders", kFALSE);
    // Monte Carlo
    write.AddContainer("MMcEvt",                    "Events",     kFALSE);
    write.AddContainer("MMcTrig",                   "Events",     kFALSE);
    // Data tree
    write.AddContainer("MSignalCam",                "Events");
    write.AddContainer("MPedPhotFundamental",       "Events");
    write.AddContainer("MPedPhotFromExtractor",     "Events");
    write.AddContainer("MPedPhotFromExtractorRndm", "Events");
    write.AddContainer("MTime",                     "Events",     kFALSE);
    write.AddContainer("MRawEvtHeader",             "Events");
    // Slow-Control: Current-tree
    write.AddContainer("MTimeCurrents",             "Currents",   kFALSE);
    write.AddContainer("MCameraDC",                 "Currents",   kFALSE);
    write.AddContainer("MReportCurrents",           "Currents",   kFALSE);
    // Slow-Control: Camera-tree
    write.AddContainer("MReportCamera",             "Camera",     kFALSE);
    write.AddContainer("MTimeCamera",               "Camera",     kFALSE);
    write.AddContainer("MCameraAUX",                "Camera",     kFALSE);
    write.AddContainer("MCameraCalibration",        "Camera",     kFALSE);
    write.AddContainer("MCameraCooling",            "Camera",     kFALSE);
    write.AddContainer("MCameraHV",                 "Camera",     kFALSE);
    write.AddContainer("MCameraLV",                 "Camera",     kFALSE);
    write.AddContainer("MCameraLids",               "Camera",     kFALSE);
    // Slow-Control: Trigger-tree
    write.AddContainer("MReportTrigger",            "Trigger",    kFALSE);
    write.AddContainer("MTimeTrigger",              "Trigger",    kFALSE);
    // Slow-Control: Drive-tree
    write.AddContainer("MReportDrive",              "Drive",      kFALSE);
    write.AddContainer("MTimeDrive",                "Drive",      kFALSE);
    // Slow-Control: Central Control-tree
    write.AddContainer("MReportCC",                 "CC",         kFALSE);
    write.AddContainer("MTimeCC",                   "CC",         kFALSE);

    // Write the special MC tree
    MWriteRootFile writemc(2, fname, fOverwrite?"RECREATE":"NEW");
    writemc.SetName("WriteMC");
    writemc.AddContainer("MMcEvtBasic", "OriginalMC");

    // Now setup tasklist for events
    MTaskList tlist2;

    tlist2.AddToList(&caldec);
    tlist2.AddToList(&decode);
    tlist2.AddToList(&apply);
    tlist2.AddToList(&merge);
    tlist2.AddToList(&pedlo1);
    tlist2.AddToList(&pedlo2);
    tlist2.AddToList(&pedlo3);
    //-----------------------------------------------------------

    MTaskList tlist3;
    tlist3.SetFilter(&fcalib);

    if (fIsInterlaced)
    {
        tlist2.AddToList(&fcalib);
        tlist2.AddToList(&tlist3);

        if (IsUsePINDiode())
            tlist3.AddToList(&pinext);
        if (IsUseBlindPixel())
            tlist3.AddToList(&bldext);
	tlist3.AddToList(&taskenv3);
        tlist3.AddToList(&pacalc);
        tlist3.AddToList(&filcam);
	if (fIsRelTimesUpdate)
	  tlist3.AddToList(&filtme);
        if (IsUseBlindPixel())
            tlist3.AddToList(&filbnd);
        if (IsUsePINDiode())
            tlist3.AddToList(&filpin);
        tlist3.AddToList(&chcalc);
	if (fIsRelTimesUpdate)
            tlist3.AddToList(&recalc);
    }  

    // Continue for all non-cosmic events
    tlist2.AddToList(&conttp);
    if (extractor1)
      tlist2.AddToList(&taskenv1);
    if (extractor2)
      tlist2.AddToList(&taskenv2);
    tlist2.AddToList(&fill0);
    tlist2.AddToList(&fill1);
    tlist2.AddToList(&contcos);
    if (fIsHiLoCalibration)
    {
        plist.AddToList(&hilocam);
        tlist2.AddToList(&filhil);
    }
    if (fIsPulsePosCheck)
    {
        plist.AddToList(&hpulcam);
        tlist2.AddToList(&filpul);
    }

    tlist2.AddToList(&fill2);
    tlist2.AddToList(&calib);
    if (extractor2 || extractor1->InheritsFrom("MExtractTimeAndCharge"))
        tlist2.AddToList(&caltm);
    if (fIsInterlaced)
        tlist2.AddToList(&fill8);
    tlist2.AddToList(&bpcal);
    tlist2.AddToList(&treat);
    tlist2.AddToList(&fill6);
    //    tlist2.AddToList(&fill3);
    tlist2.AddToList(&fill4);
    tlist2.AddToList(&fill5);
    if (extractor2 || extractor1->InheritsFrom("MExtractTimeAndCharge"))
        tlist2.AddToList(&fill7);
    tlist2.AddToList(&fill9);
    tlist2.AddToList(&fillH);
    tlist2.AddToList(&fillL);

    tlist2.AddToList(&fillflorian);

    // Setup List for Drive-tree
    //MPointingPosCalc pcalc;

    // Now setup main tasklist
    tlist.AddToList(read);

    if (IsUseMC())
    {
        tlist.AddToList(&writemc);
        tlist.AddToList(&contmc);
    }

    //if (IsUseRootData())
    //  tlist2.AddToList(&pextr);
    tlist.AddToList(&tlist2, IsUseRootData() ? "Events" : "All");

    //if (IsUseMC())
    //  tlist.AddToList(&pcalc, "Drive");

    tlist.AddToList(&write);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(fLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    // Execute first analysis
    const Bool_t rc = evtloop.Eventloop(fMaxEvents);

    // make sure owned object are deleted
    if (extractor1 && extractor1!=fExtractor)
        delete extractor1;
    if (extractor2)
        delete extractor2;
    if (extractor3)
        delete extractor3;

    // return if job failed
    if (!rc)
    {
        *fLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    // if everything went ok write and display result
    DisplayResult(plist);

    if (fIsPixelCheck)
      {
        if (fIsPulsePosCheck)
            hpulcam[fCheckedPixId].DrawClone("");
        
	if (fIsHiLoCalibration)
	    hilocam[fCheckedPixId].DrawClone("");
      }

    interlacedcont.Add(&pulcam);

    if (fIsHiLoCalibration)
      interlacedcont.Add(&hilcam);

    if (fIsPulsePosCheck)
        interlacedcont.Add(plist.FindObject("MHCalibrationPulseTimeCam"));

    if (fIsHiLoCalibration)
        interlacedcont.Add(plist.FindObject("MHCalibrationHiLoCam"));

    if (!WriteResult(interlacedcont))
      return kFALSE;

    // return if job went ok
    *fLog << all << GetDescriptor() << ": Done." << endl;
    *fLog << endl << endl;

    return kTRUE;
}


void MJCalibrateSignal::DisplayResult(MParList &plist)
{
    if (!fDisplay || !fIsHiLoCalibration)
        return;

    MCalibrationHiLoCam *hcam = (MCalibrationHiLoCam*)plist.FindObject("MCalibrationHiLoCam");
    MGeomCam            *geom = (MGeomCam*)plist.FindObject("MGeomCam");
    if (!hcam || !geom)
        return;

    // Create histograms to display
    MHCamera disp1(*geom, "HiLoConv", "Ratio Amplification HiGain vs. LoGain (Charges)");
    MHCamera disp2(*geom, "HiLoDiff", "Arrival Time Diff. HiGain vs. LoGain (Times)");

    disp1.SetCamContent(*hcam, 0);
    disp1.SetCamError(  *hcam, 1);
    disp2.SetCamContent(*hcam, 5);
    disp2.SetCamError(  *hcam, 6);

    disp1.SetYTitle("R [1]");
    disp2.SetYTitle("\\Delta T [FADC sl.]");

    TCanvas &c1 = fDisplay->AddTab("HiLoConv");
    c1.Divide(2,3);
    
    disp1.CamDraw(c1, 1, 2, 1);
    disp2.CamDraw(c1, 2, 2, 1);
}

