/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 04/2005 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Daniela Dorner, 04/2005 <mailto:dorner@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2006
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// fillsignal.C
// ============
//
// This macro is used to read the calibration-/callisto-output files
// signal00000.root.
//
// From this file the mean pedrms, the mean signal and the pulse position
// for the inner and outer camera is extracted and inserted into the database
// in the table Calibration, where the results of callisto are stored.
// The sequence number is extracted from the filename.
//
// Usage:
//   .x fillsignal.C("/magic/data/callisto/0004/00047421/signal00047421.root", kTRUE)
//
// The second argument is the 'dummy-mode'. If it is kTRUE dummy-mode is
// switched on and nothing will be written into the database. This is usefull
// for tests.
//
// The macro can also be run without ACLiC but this is a lot slower...
//
// Remark: Running it from the commandline looks like this:
//   root -q -l -b fillsignal.C+\(\"filename\"\,kFALSE\) 2>&1 | tee fillsignal.log
//
// Make sure, that database and password are corretly set in a resource
// file called sql.rc and the resource file is found.
//
// Returns 2 in case of failure, 1 in case of success and 0 if the connection
// to the database is not working.
//
/////////////////////////////////////////////////////////////////////////////
#include <iostream>

#include <TEnv.h>
#include <TRegexp.h>

#include <TFile.h>
#include <TSQLResult.h>

#include "MSQLServer.h"

#include "MStatusArray.h"
#include "MHCamera.h"

#include "MCalibrationPulseTimeCam.h"
#include "MCalibrationPix.h"

using namespace std;

int Process(MSQLServer &serv, TString fname, Bool_t dummy)
{
    TFile file(fname, "READ");
    if (!file.IsOpen())
    {
        cout << "ERROR - Could not find file " << fname << endl;
        return 2;
    }

    Float_t meanextpul = -1;
    Float_t rmsextpul  = -1;

    MCalibrationPulseTimeCam *pt;
    file.GetObject("MCalibrationPulseTimeCam", pt);
    if (pt)
    {
        meanextpul = pt->GetAverageArea(0).GetHiGainMean();
        rmsextpul  = pt->GetAverageArea(0).GetHiGainRms();

        meanextpul = TMath::Nint(meanextpul*100)/100.;
        rmsextpul  = TMath::Nint(rmsextpul*100)/100.;
    }

    MStatusArray arr;
    if (arr.Read()<=0)
    {
        cout << "ERROR - Reading of MStatusDisplay failed." << endl;
        return 2;
    }

    MHCamera *cam = (MHCamera*)arr.FindObjectInCanvas("PedRMS;avg", "MHCamera", "PedRMS");
    if (!cam)
    {
        cout << "WARNING - Reading of PedRMS;avg failed." << endl;
        return 2;
    }

    MHCamera *pul = (MHCamera*)arr.FindObjectInCanvas("PulsePos;avg", "MHCamera", "PulsePos");
    if (!pul)
    {
        cout << "WARNING - Reading of PulsePos;avg failed." << endl;
        return 2;
    }

    MHCamera *difflo = (MHCamera*)arr.FindObjectInCanvas("DiffLo;avg", "MHCamera", "DiffLo");
    if (!difflo)
    {
        cout << "WARNING - Reading of DiffLo;avg failed." << endl;
        return 2;
    }
    MHCamera *diffhi = (MHCamera*)arr.FindObjectInCanvas("DiffHi;avg", "MHCamera", "DiffHi");
    if (!diffhi)
    {
        cout << "WARNING - Reading of DiffHi;avg failed." << endl;
        return 2;
    }

    MHCamera *hilooff = (MHCamera*)arr.FindObjectInCanvas("HiLoOff;avg", "MHCamera", "HiLoOff");
    if (!hilooff)
    {
        cout << "WARNING - Reading of HiLoOff failed." << endl;
        return 2;
    }

    MHCamera *hilocal = (MHCamera*)arr.FindObjectInCanvas("HiLoCal;avg", "MHCamera", "HiLoCal");
    if (!hilocal)
    {
        cout << "WARNING - Reading of HiLoCal failed." << endl;
        return 2;
    }

    TArrayI inner(1);
    inner[0] = 0;

    TArrayI outer(1);
    outer[0] = 1;

    Int_t s0[] = { 1, 2, 3, 4, 5, 6 };

    Stat_t meanrmsi = cam->GetMeanSectors(TArrayI(6, s0), inner);
    Stat_t meanrmso = cam->GetMeanSectors(TArrayI(6, s0), outer);

    if (meanrmsi<0 || meanrmso<0)
    {
        cout << "WARNING - MeanPedRMS inner or outer < 0 " << endl;
        cout << "MeanPedRMS inner " << meanrmsi << endl;
        cout << "MeanPedRMS outer " << meanrmso << endl;
        return 2;
    }

    meanrmsi = TMath::Nint(meanrmsi*100)/100.;
    meanrmso = TMath::Nint(meanrmso*100)/100.;

    cam = (MHCamera*)arr.FindObjectInCanvas("Interp'd;avg", "MHCamera", "Interp'd");
    if (!cam)
    {
        cout << "WARNING - Reading of Interp'd;avg failed." << endl;
        return 2;
    }

    Stat_t meansigi = cam->GetMeanSectors(TArrayI(6, s0), inner);
    Stat_t meansigo = cam->GetMeanSectors(TArrayI(6, s0), outer);

    if (meansigi<0 || meansigo<0)
    {
        cout << "WARNING - MeanInterp'd inner or outer < 0 " << endl;
        cout << "MeanInterp'd inner " << meansigi << endl;
        cout << "MeanInterp'd outer " << meansigo << endl;
        return 2;
    }

    meansigi = TMath::Nint(meansigi*100)/100.;
    meansigo = TMath::Nint(meansigo*100)/100.;

    Stat_t meanpul = pul->GetMean();
    Stat_t rmspul  = pul->GetRMS();

    if (meanpul<0 || rmspul<0)
    {
        cout << "WARNING - PulsePos'd mean or rms < 0 " << endl;
        cout << "PulsePos'd mean " << meanpul << endl;
        cout << "PulsePos'd rms  " << rmspul << endl;
        return 2;
    }

    meanpul = TMath::Nint(meanpul*100)/100.;
    rmspul  = TMath::Nint(rmspul *100)/100.;

/*
    Double_t meanhi  = TMath::Nint(pulhi->GetMean()*100.)/100.;
    Double_t rmshi   = TMath::Nint(pulhi->GetRMS() *100.)/100.;

    Double_t meanlo  = TMath::Nint(pullo->GetMean()*100.)/100.;
    Double_t rmslo   = TMath::Nint(pullo->GetRMS() *100.)/100.;
    pullo->Add(pullo, pulhi, 1, -1);
    pullo->ResetBit(MHCamera::kProfile);

    Double_t meanoff = TMath::Nint(pullo->GetMean()*100.)/100.;
    Double_t rmsoff  = TMath::Nint(pullo->GetRMS() *100.)/100.;
 */
    Double_t meanoff = TMath::Nint(hilooff->GetMean()*100.)/100.;
    Double_t rmsoff  = TMath::Nint(hilooff->GetRMS() *100.)/100.;

    Double_t meancal = TMath::Nint(hilocal->GetMean()*100.)/100.;
    Double_t rmscal  = TMath::Nint(hilocal->GetRMS() *100.)/100.;

    //get sequence number from the filename
    TString sequence = fname(TRegexp("signal[0-9]+[.]root$"));
    if (sequence.IsNull())
    {
        cout << "WARNING - Sequ# empty" << endl;
        cout << "Sequ# " << sequence << endl;
        return 2;
    }

    Int_t seq = atoi(sequence.Data()+6);
    TString meanrmsinner =Form("%6.2f", meanrmsi);
    TString meanrmsouter =Form("%6.2f", meanrmso);
    TString meansiginner =Form("%6.2f", meansigi);
    TString meansigouter =Form("%6.2f", meansigo);
    TString meanpulpos   =Form("%6.2f", meanpul);
    TString rmspulpos    =Form("%6.2f", rmspul);
    TString meanextpulpos=Form("%6.2f", meanextpul);
    TString rmsextpulpos =Form("%6.2f", rmsextpul);
   /*
    TString meanpulhi    =Form("%6.2f", meanhi);
    TString rmspulhi     =Form("%6.2f", rmshi);

    TString meanpullo    =Form("%6.2f", meanlo);
    TString rmspullo     =Form("%6.2f", rmslo);
    */
    TString meanpuloff   =Form("%6.2f", meanoff);
    TString rmspuloff    =Form("%6.2f", rmsoff);
    TString meanhilocal  =Form("%6.2f", meancal);
    TString rmshilocal   =Form("%6.2f", rmscal);

    if (meanextpul<0 && rmsextpul<0)
    {
        meanextpulpos = "NULL";
        rmsextpulpos  = "NULL";
    }

    // *****************************************************

    // *****************************************************


    cout << "Sequence #" << seq << endl;
    cout << "  Mean Ped RMS inner [phe] " << meanrmsinner  << endl;
    cout << "  Mean Ped RMS outer [phe] " << meanrmsouter  << endl;
    cout << "  Mean Signal  inner [phe] " << meansiginner  << endl;
    cout << "  Mean Signal  outer [phe] " << meansigouter  << endl;
    cout << "  Mean calibrated PulsePos " << meanpulpos    << " +- " << rmspulpos    << endl;
    cout << "  Mean extracted  PulsePos " << meanextpulpos << " +- " << rmsextpulpos << endl;
//    cout << "  Mean ext.HiGain PulsePos " << meanpulhi     << " +- " << rmspulhi     << endl;
//    cout << "  Mean ext.LoGain PulsePos " << meanpullo     << " +- " << rmspullo     << endl;
    cout << "  Lo-Hi gain offset:       " << meanpuloff    << " +- " << rmspuloff    << endl;
    cout << "  Hi/Lo gain ratio:        " << meanhilocal   << " +- " << rmshilocal   << endl;
    cout << endl;

    //build query and insert information into the database
    // here only a update query is built, as this macro is exexuted after
    // the macro fillcalib.C in the script fillcallisto
    // and so the table Calibration is always updated
    TString query = Form("UPDATE Calibration SET "
                         " fMeanPedRmsInner=%s,   fMeanPedRmsOuter=%s,  "
                         " fMeanSignalInner=%s,   fMeanSignalOuter=%s,  "
                         " fPulsePosMean=%s,      fPulsePosRms=%s,      "
                         " fPulsePosCheckMean=%s, fPulsePosCheckRms=%s  "
                         //" fPulsePosHiMean=%s,    fPulsePosHiRms=%s,    "
                         //" fPulsePosLoMean=%s,    fPulsePosLoRms=%s,    "
                         " fPulsePosOffMean=%s,   fPulsePosOffRms=%s,   "
                         " fHiLoGainRatioMean=%s, fHiLoGainRatioRms=%s, "
                         " WHERE fSequenceFirst='%d' ",
                         meanrmsinner.Data(),  meanrmsouter.Data(),
                         meansiginner.Data(),  meansigouter.Data(),
                         meanpulpos.Data(),    rmspulpos.Data(),
                         meanextpulpos.Data(), rmsextpulpos.Data(),
                         //meanpulhi.Data(),     rmspulhi.Data(),
                         //meanpullo.Data(),     rmspullo.Data(),
                         meanpuloff.Data(),    rmspuloff.Data(),
                         meanhilocal.Data(),   rmshilocal.Data(),
                         seq);

    if (dummy)
        return 1;

    TSQLResult *res = serv.Query(query);
    if (!res)
    {
        cout << "ERROR - Query failed: " << query << endl;
        return 2;
    }
    delete res;
    return 1;
}

int fillsignal(TString fname, Bool_t dummy=kTRUE)
{
    TEnv env("sql.rc");

    MSQLServer serv(env);
    if (!serv.IsConnected())
    {
        cout << "ERROR - Connection to database failed." << endl;
        return 0;
    }

    cout << "fillsignal" << endl;
    cout << "----------" << endl;
    cout << endl;
    cout << "Connected to " << serv.GetName() << endl;
    cout << "File: " << fname << endl;
    cout << endl;

    //process file
    return Process(serv, fname, dummy);
}
