/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Josep Flix 04/2001 (jflix@ifae.es)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//   MPedCalcPedRun                                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MPedCalcPedRun.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MRawEvtPixelIter.h"
#include "MRawEvtData.h"

#include "MPedestalPix.h"
#include "MPedestalCam.h"

ClassImp(MPedCalcPedRun)

MPedCalcPedRun::MPedCalcPedRun(const char *name, const char *title)
{
    *fName  = name  ? name  : "MPedCalcPedRun";
    *fTitle = title ? title : "Task to calculate pedestals from pedestal runs raw data";
}

Bool_t MPedCalcPedRun::PreProcess( MParList *pList )
{
    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fPedestals = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!fPedestals)
    {
        *fLog << dbginf << "MPedestalCam not found... aborting." << endl;
        return kFALSE;
    }

    return kTRUE;
}


Bool_t MPedCalcPedRun::Process()
{

    MRawEvtPixelIter pixel(fRawEvt);

    while (pixel.Next())
    {
        const UInt_t pixid = pixel.GetPixelId();

        Byte_t *ptr = pixel.GetHiGainFadcSamples();
        const Byte_t *end = ptr + fRawEvt->GetNumHiGainSamples();

        Float_t HiGainPed = PedMeanCalcHiGain(ptr,end);
        Float_t HiGainRms = PedRmsCalcHiGain(ptr,end,HiGainPed);

        Float_t HiGainPedErr = PedMeanErrCalcHiGain(HiGainRms);
        Float_t HiGainRmsErr = PedRmsErrCalcHiGain(HiGainRms);

        (*fPedestals)[pixid].SetPedestal(HiGainPed,HiGainRms);
        (*fPedestals)[pixid].SetPedestalRms(HiGainPedErr,HiGainRmsErr);

    }

    return kTRUE;

}



Float_t MPedCalcPedRun::PedMeanCalcHiGain(Byte_t *ptr, const Byte_t *end)
{
    Float_t sum=0;

    do sum += *ptr++;
    while (ptr != end);

    sum = sum/(Int_t)fRawEvt->GetNumHiGainSamples();

    return sum;
}


Float_t MPedCalcPedRun::PedRmsCalcHiGain(Byte_t *ptr, const Byte_t *end, Float_t HiGainPed)
{

  Float_t rms = 0;

  do{
    rms = rms + pow((*ptr - HiGainPed),2);
    *ptr++;
    }while (ptr != end);

    rms = sqrt((Float_t)rms/((Int_t)fRawEvt->GetNumHiGainSamples()-1));

  return rms;
}


Float_t MPedCalcPedRun::PedMeanErrCalcHiGain(Float_t HiGainRms){
  return HiGainRms/sqrt((Int_t)fRawEvt->GetNumHiGainSamples());
}


Float_t MPedCalcPedRun::PedRmsErrCalcHiGain(Float_t HiGainRms){
  return HiGainRms/sqrt(2*(Int_t)fRawEvt->GetNumHiGainSamples());
}

