#ifndef MARS_MRawEvtPixelIter
#define MARS_MRawEvtPixelIter
///////////////////////////////////////////////////////////////////////////////
//
//  MRawEvtPixelIter
//
//  class to iterate over all pixels of one event.
//
///////////////////////////////////////////////////////////////////////////////
#ifndef MARS_MAGIC
#include "MAGIC.h"
#endif

class MRawEvtData;

class MRawEvtPixelIter : public TObject
{
private:
    UShort_t  fNumHiGainEntry;  //! actual number of entry in fHiGainPixId
    UShort_t  fNumLoGainEntry;  //! actual number of entry in fLoGainPixId

    UShort_t *fHiGainId;        //! actual entry of fHiGainPixId
    UShort_t *fLoGainId;        //! actual entry of fLoGainPixId

    Byte_t   *fHiGainPos;       //! pointer to hi-gain samples of actual pixel
    Byte_t   *fLoGainPos;       //! pointer to lo-gain samples of actual pixel

    Byte_t   *fABFlags;         //! pointer to AB flags

    UShort_t  fNumEntry;

    Byte_t fNumHiGainSamples;   //!
    Byte_t fNumLoGainSamples;   //!

    MRawEvtData *fData;         //! pointer to object which we are iterating

public:
    MRawEvtPixelIter(MRawEvtData *dat);

    MRawEvtData *Next();

    Bool_t Jump(UShort_t id)
    {
        //
        // Jump to the pixel with the pixel-id ID
        // If it doesn't exist return FALSE
        //
        Reset();
        while (Next())
            if (GetPixelId() == id)
                return kTRUE;
        return kFALSE;
    }

    UShort_t GetPixelId() const
    {
        //
        // return Id of actual pixel
        //
        return *fHiGainId;
    }

    UShort_t GetNumEntry() const
    {
        return fNumHiGainEntry;
    }

    Byte_t *GetHiGainSamples() const
    {
        //
        // return a pointer to the fadc samples of the hi gains
        // WARNING: Don't forget to get the number of valid entries
        //          (GetNumSamples) to know how many entries of the array
        //          belong to the actual pixel
        //
        return fHiGainPos;
    }

    /*
     This function is dangerous as long as MC with split hi-/lo-gain are used
    Byte_t *GetSamples() const
    {
        //
        // return a pointer to the fadc samples of the hi gains
        // WARNING: Don't forget to get the number of valid entries
        //          (GetNumSamples) to know how many entries of the array
        //          belong to the actual pixel
        //
        return fHiGainPos;
    }
    */

    Byte_t  GetNumHiGainSamples() const { return fNumHiGainSamples ; }// Use is deprecated!
    Byte_t  GetNumLoGainSamples() const { return fNumLoGainSamples ; }// Use is deprecated!
    Byte_t  GetNumSamples() const { return fNumHiGainSamples+fNumLoGainSamples; }

    Byte_t  GetIdxMaxHiGainSample(const Byte_t hifirst=0, const Byte_t hilast=0xff) const;
    Short_t GetIdxMaxLoGainSample(const Byte_t lofirst=0, const Byte_t lolast=0xff) const;

    Byte_t  GetMaxHiGainSample(const Byte_t hifirst=0, const Byte_t hilast=0xff) const;

    ULong_t GetSumHiGainSamples() const;
    ULong_t GetSumSqrHiGainSamples() const;
    Float_t GetVarHiGainSamples() const;

    Byte_t  GetMaxLoGainSample() const;
    Short_t GetIdxMaxHiLoGainSample() const;

    ULong_t GetSumLoGainSamples() const;
    ULong_t GetSumSqrLoGainSamples() const;

    Bool_t HasLoGain() const { return fNumLoGainSamples>0; }
    Bool_t IsABFlagValid() const { return fABFlags ? kTRUE : kFALSE; }
    Bool_t HasABFlag() const
    {
        //
        // return kTRUE  the lo gains exist for the actual pixel, else return kFALSE
        //
        return TESTBIT(GetABFlag(), GetPixelId()%8);
    }
    Byte_t GetABFlag() const
    {
        //
        // return kTRUE  the lo gains exist for the actual pixel, else return kFALSE
        //
        return fABFlags ? fABFlags[GetPixelId()/8] : 0;
    }

    Byte_t *GetLoGainSamples() const
    {
        //
        // return a pointer to the fadc samples of the lo gains if they exist
        // for the actual pixel, else return zero
        //
        return HasLoGain() ? fLoGainPos : NULL;
    }

    void Reset();

    void Draw(Option_t *t="GRAPH");

    ClassDef(MRawEvtPixelIter, 0) // iterates over all pixels of one MRawEvtData object
};

#endif
