/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Josep Flix 04/2001 (jflix@ifae.es)
!              Thomas Bretz 05/2001 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//   MPedCalcPedRun                                                        //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MPedCalcPedRun.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MRawEvtPixelIter.h"
#include "MRawEvtData.h"

#include "MPedestalPix.h"
#include "MPedestalCam.h"

ClassImp(MPedCalcPedRun)

MPedCalcPedRun::MPedCalcPedRun(const char *name, const char *title)
{
    *fName  = name  ? name  : "MPedCalcPedRun";
    *fTitle = title ? title : "Task to calculate pedestals from pedestal runs raw data";
}

Bool_t MPedCalcPedRun::PreProcess( MParList *pList )
{
    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fPedestals = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!fPedestals)
        return kFALSE;

    fNumHiGainSamples = fRawEvt->GetNumHiGainSamples();

    return kTRUE;
}


Bool_t MPedCalcPedRun::Process()
{
    MRawEvtPixelIter pixel(fRawEvt);

    while (pixel.Next())
    {
              Byte_t *ptr = pixel.GetHiGainFadcSamples();
        const Byte_t *end = ptr + fRawEvt->GetNumHiGainSamples();

        const Float_t higainped = CalcHiGainMean(ptr, end);
        const Float_t higainrms = CalcHiGainRms(ptr, end, higainped);

        const Float_t higainpederr = CalcHiGainMeanErr(higainrms);
        const Float_t higainrmserr = CalcHiGainRmsErr(higainrms);

        const UInt_t pixid = pixel.GetPixelId();
        MPedestalPix &pix = (*fPedestals)[pixid];

        pix.SetPedestal(higainped, higainrms);
        pix.SetPedestalRms(higainpederr, higainrmserr);
    }

    fPedestals->SetHasChanged();

    return kTRUE;
}

Float_t MPedCalcPedRun::CalcHiGainMean(Byte_t *ptr, const Byte_t *end) const
{
    Int_t sum=0;

    do sum += *ptr;
    while (++ptr != end);

    return (Float_t)sum/fNumHiGainSamples;
}


Float_t MPedCalcPedRun::CalcHiGainRms(Byte_t *ptr, const Byte_t *end, Float_t higainped) const
{
    Float_t rms = 0;

    do
    {
        const Float_t diff = (Float_t)(*ptr)-higainped;

        rms += diff*diff;
    } while (++ptr != end);

    return sqrt(rms/(fNumHiGainSamples-1));
}

Float_t MPedCalcPedRun::CalcHiGainMeanErr(Float_t higainrms) const
{
    return higainrms/sqrt(fNumHiGainSamples);
}

Float_t MPedCalcPedRun::CalcHiGainRmsErr(Float_t higainrms) const
{
    return higainrms/sqrt(2.*fNumHiGainSamples);
}

