/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  06/2001 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MWriteAsciiFile                                                         //
//                                                                         //
// If you want to store a single container into an Ascii file you have     //
// to use this class. You must know the name of the file you wanne write   //
// (you should know it) and the name of the container you want to write.   //
// This can be the name of the class or a given name, which identifies     //
// the container in a parameter container list (MParList).                 //
// The container is writte to the ascii file if its HasChanged flag is     //
// set (MParContainer)                                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MWriteAsciiFile.h"

#include <fstream.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

ClassImp(MWriteAsciiFile)

// --------------------------------------------------------------------------
//
// Specify the name of the ascii output file 'filename' and the name
// of the container you want to write. (Normally this is the name
// of the class (eg. MHillas) but it can also be a different name which
// identifies the container in the parameter list.
//
MWriteAsciiFile::MWriteAsciiFile(const char *filename, const char *contname,
                                 const char *name, const char *title)
    : fOut(NULL), fContainer(NULL)
{
    *fName  = name  ? name  : "MWriteAsciiFile";
    *fTitle = title ? title : "Task to write one container to an ascii file";

    fNameFile      = filename;
    fNameContainer = contname;
}

// --------------------------------------------------------------------------
//
// Specify a the name of the ascii output file 'filename' and a pointer to
// the container you want to write. 
//
MWriteAsciiFile::MWriteAsciiFile(const char *filename, MParContainer *cont,
                                 const char *name, const char *title)
    : fOut(NULL), fContainer(cont)
{
    *fName  = name  ? name  : "MWriteAsciiFile";
    *fTitle = title ? title : "Task to write one container to an ascii file";

    fNameFile      = filename;
    fNameContainer = cont->GetName();
}

// --------------------------------------------------------------------------
//
// Destructor. Delete the output file if necessary (it is closed
// automatically by its destructor.
//
MWriteAsciiFile::~MWriteAsciiFile()
{
    if (fOut)
        delete fOut;
}

// --------------------------------------------------------------------------
//
// Tries to open the given output file. And tries to get a pointer to the
// instance of the container which should be written.
// If the container has already the HasChanged flag it is immediatly written
// to the output file.
//
Bool_t MWriteAsciiFile::PreProcess (MParList *pList)
{
    //
    // Try to find the container which should be stored.
    //
    if (!fContainer)
    {
        fContainer = (MParContainer*)pList->FindObject(fNameContainer);
        if (!fContainer)
        {
            *fLog << dbginf << "Cannot find parameter container '" << fContainer << "'." << endl;
            return kFALSE;
        }
    }

    //
    // Try to open the output file
    //
    fOut = new ofstream(fNameFile);

    if (!(*fOut))
    {
        *fLog << dbginf << "Cannot open Ascii file '" << fNameFile << "' for writing." << endl;
        return kFALSE;
    }

    *fLog << "Ascii file '" << fNameFile << "' opened for writing." << endl;

    //
    // write the container if it is already in changed state
    //
    if (fContainer->HasChanged())
        fContainer->AsciiWrite(*fOut);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Checks if the HasChanged flag of the output container is set. If it is set
// the container is instructed to write itself to the ascii file.
// (By calling its memberfunction AsciiWrite. You find the Prototype in
// MParContainer)
//
Bool_t MWriteAsciiFile::Process()
{
    if (fContainer->HasChanged())
        fContainer->AsciiWrite(*fOut);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// If the output container has the HasChanged flag set, it is written to the
// output file (eg. this could be some calculated result)
// If the output file object exists, delete it. (The file is closed
// automatically in the corresponding destructor)
//
Bool_t MWriteAsciiFile::PostProcess()
{
    //
    // check if the container changed state is set
    //
    if (fContainer->HasChanged())
        fContainer->AsciiWrite(*fOut);

    //
    // delete (close) file
    //
    delete fOut;
    fOut = NULL;

    return kTRUE;
}

