/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//   MCerPhotCalc                                                           //
//                                                                          //
//   This is a task which calculates the number of photons from the FADC    //
//   time slices. At the moment it integrates simply the FADC values.       //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MCerPhotCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MRawEvtPixelIter.h"
#include "MCerPhotEvt.h"
#include "MPedestalPix.h"
#include "MPedestalCam.h"

ClassImp(MCerPhotCalc)

// --------------------------------------------------------------------------
//
// Default constructor.
//
MCerPhotCalc::MCerPhotCalc(const char *name, const char *title)
{
    *fName  = name  ? name  : "MCerPhotCalc";
    *fTitle = title ? title : "Task to calculate Cerenkov photons from raw data";
}

// --------------------------------------------------------------------------
//
// The PreProcess searches for the following input containers:
//  - MRawEvtData
//  - MPedestalCam
//
// The following output containers are also searched and created if
// they were not found:
//  - MCerPhotEvt
//
Bool_t MCerPhotCalc::PreProcess( MParList *pList )
{
    fRawEvt = (MRawEvtData*)pList->FindObject("MRawEvtData");
    if (!fRawEvt)
    {
        *fLog << dbginf << "MRawEvtData not found... aborting." << endl;
        return kFALSE;
    }

    fPedestals = (MPedestalCam*)pList->FindObject("MPedestalCam");
    if (!fPedestals)
    {
        *fLog << dbginf << "MPedestalCam not found... aborting." << endl;
        return kFALSE;
    }

    fCerPhotEvt = (MCerPhotEvt*)pList->FindCreateObj("MCerPhotEvt");
    if (!fCerPhotEvt)
        return kFALSE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Calculate the integral of the FADC time slaices and store them as a new
// pixel in the MCerPhotEvt container.
//
Bool_t MCerPhotCalc::Process()
{
    fCerPhotEvt->Clear();

    MRawEvtPixelIter pixel(fRawEvt);

    while (pixel.Next())
    {
        const UInt_t pixid = pixel.GetPixelId();

        const MPedestalPix &ped = (*fPedestals)[pixid];

        const Float_t nphot = (Float_t)pixel.GetSumHiGainFadcSamples() - ped.GetMean();

        fCerPhotEvt->AddPixel(pixid, nphot, ped.GetMeanRms());

        // FIXME! Handling of Lo Gains is missing!
    }

    fCerPhotEvt->SetReadyToSave();

    return kTRUE;
}

