/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 2/2006 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2007
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
// MHSrcPosCam
//
//
// FIXME: Use ReInit...
//
//////////////////////////////////////////////////////////////////////////////
#include "MHSrcPosCam.h"

#include <TCanvas.h>
#include <TEllipse.h>

#include "MGeomCam.h"
#include "MSrcPosCam.h"
#include "MParameters.h"
#include "MPointingPos.h"

#include "MString.h"
#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHSrcPosCam);

using namespace std;

// --------------------------------------------------------------------------
//
// Default Constructor
//
MHSrcPosCam::MHSrcPosCam(Bool_t wobble, const char *name, const char *title)
    : fTimeEffOn(NULL), fEffOnTime(NULL), fSourcePos(NULL)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHSrcPosCam";
    fTitle = title ? title : "Histogram for source position distribution";

    fHist.SetName("SourcePos");
    fHist.SetTitle("Source position distribution in camera coordinates");
    fHist.SetXTitle("dx [\\circ]");
    fHist.SetYTitle("dy [\\circ]");
    fHist.SetZTitle("T_{eff} [s]");
    fHist.SetDirectory(NULL);
    fHist.UseCurrentStyle();
    fHist.GetXaxis()->CenterTitle();
    fHist.GetYaxis()->CenterTitle();
    fHist.SetContour(99);

    const Float_t x = wobble ? 0.5499 : 0.2499;
    const Int_t   n = wobble ? 101    : 51;

    MBinning bins;
    bins.SetEdges(n, -x, x); // bin=0.01  ~0.5SE

    MH::SetBinning(&fHist, &bins, &bins);
}

Bool_t MHSrcPosCam::SetupFill(const MParList *pl)
{
    fTimeEffOn = (MTime*)pl->FindObject("MTimeEffectiveOnTime");
    if (!fTimeEffOn)
    {
        *fLog << err << "ERROR - MTimeEffOnTime not found... aborting." << endl;
        return kFALSE;
    }

    MGeomCam *geom = (MGeomCam*)pl->FindObject("MGeomCam");
    if (!geom)
    {
        *fLog << err << "ERROR - MGeomCam not found... aborting." << endl;
        return kFALSE;
    }

    fEffOnTime = (MParameterD*)pl->FindObject("MEffectiveOnTime");
    if (!fEffOnTime)
    {
        *fLog << err << "ERROR - MEffectiveOnTime not found... aborting." << endl;
        return kFALSE;
    }

    MPointingPos *pos = (MPointingPos*)pl->FindObject("MSourcePos", "MPointingPos");
    if (!pos)
    {
        *fLog << warn << "ERROR - MSourcePos not found... aborting." << endl;
        return kFALSE;
    }

    const TString src = pos->GetString("radec");
    fHist.SetTitle(MString::Format("SrcPos distribution in camera: %s", src.Data()));

    fHist.Reset();
    fXY            = TVector2();
    fNum           = 0;
    fTimeLastEffOn = MTime();
    fConvMm2Deg    = geom->GetConvMm2Deg();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// 
// 
Bool_t MHSrcPosCam::Fill(const MParContainer *par, const Stat_t w)
{
    const MSrcPosCam *cam = dynamic_cast<const MSrcPosCam*>(par);
    if (!cam)
    {
        *fLog << err << dbginf << "Got no MSrcPosCam as argument of Fill()..." << endl;
        return kFALSE;
    }

//    if (fName=="MHSrcPosCam")
//    {
    fXY += cam->GetXY();
    fNum++;

    if (fTimeLastEffOn==MTime())
        fTimeLastEffOn=*fTimeEffOn;

    if (fTimeLastEffOn == *fTimeEffOn)
        return kTRUE;

    fXY *= fConvMm2Deg/fNum;

    fHist.Fill(fXY.X(), fXY.Y(), fEffOnTime->GetVal());
//    }
//    else
//        fHist.Fill(cam->GetX()*fConvMm2Deg, cam->GetY()*fConvMm2Deg);

    fXY            = TVector2();
    fNum           = 0;
    fTimeLastEffOn = *fTimeEffOn;

    return kTRUE;
}

void MHSrcPosCam::Paint(Option_t *)
{
    MH::SetPalette("pretty", 99);
}

// --------------------------------------------------------------------------
//
// 
// 
void MHSrcPosCam::Draw(Option_t *)
{
    TVirtualPad *pad = gPad ? gPad : MakeDefCanvas(this);
    pad->SetBorderMode(0);

    //pad->Divide(2,2);

    gPad->SetLeftMargin(0.25);
    gPad->SetRightMargin(0.25);

    gPad->SetGridx();
    gPad->SetGridy();

    AppendPad();

    fHist.Draw("colz");

    if (fHist.GetXaxis()->GetXmax()>0.5)
    {
        TEllipse el;
        el.SetFillStyle(4000);
        el.SetLineColor(kMagenta);
        el.DrawEllipse(0, 0, 0.4, 0, 0, 360, 0);
    }
}

