/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 05/2005 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Daniela Dorner, 05/2005 <mailto:dorner@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2006
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// fillstar.C
// ==========
//
// This macro is used to read the star-output files.
// These files are automatically called star00000000.root.
// From these files the muon parameters (psf, muon number, ratio, muon rate),
// the  rate, the number of islands, the effective ontime, the maximum
// humidity and a parameter describing the inhomogeneity are extracted from
// the status display.
// The sequence number is extracted from the filename.
//
// Usage:
//   .x fillstar.C("/magic/data/star/0004/00047421/star00047421.root", kTRUE)
//
// The second argument is the 'dummy-mode'. If it is kTRUE dummy-mode is
// switched on and nothing will be written into the database. This is usefull
// for tests.
//
// The macro can also be run without ACLiC but this is a lot slower...
//
// Remark: Running it from the commandline looks like this:
//   root -q -l -b fillstar.C+\(\"filename\"\,kFALSE\) 2>&1 | tee fillstar.log
//
// Make sure, that database and password are corretly set in a resource
// file called sql.rc and the resource file is found.
//
// Returns 2 in case of failure, 1 in case of success and 0 if the connection
// to the database is not working.
//
/////////////////////////////////////////////////////////////////////////////
#include <iostream>
#include <iomanip>

#include <TEnv.h>
#include <TRegexp.h>

#include <TH1.h>
#include <TH2.h>
#include <TGraph.h>
#include <TProfile.h>
#include <TFile.h>
#include <TSQLResult.h>
#include <TSQLRow.h>

#include "MSQLServer.h"

#include "MHCamera.h"
#include "MHMuonPar.h"
#include "MStatusArray.h"
#include "MGeomCamMagic.h"
#include "MBadPixelsCam.h"

using namespace std;

// --------------------------------------------------------------------------
//
// Checks whether an entry is already existing
//
Bool_t ExistStr(MSQLServer &serv, const char *column, const char *table, Int_t test)
{
    TString query(Form("SELECT %s FROM %s WHERE %s='%d'", column, table, column, test));
    TSQLResult *res = serv.Query(query);
    if (!res)
        return kFALSE;

    TSQLRow *row;

    Bool_t rc = kFALSE;
    while ((row=res->Next()))
    {
        if ((*row)[0])
        {
            rc = kTRUE;
            break;
        }
    }

    delete res;

    return rc;
}


int Process(MSQLServer &serv, TString fname, Bool_t dummy)
{
    TFile file(fname, "READ");
    if (!file.IsOpen())
    {
        cout << "ERROR - Could not find file " << fname << endl;
        return 2;
    }


    MStatusArray arr;
    if (arr.Read()<=0)
    {
        cout << "ERROR - Reading of MStatusDisplay failed." << endl;
        return 2;
    }

    MHCamera *hsparks = (MHCamera*)arr.FindObjectInCanvas("Sparks;avg", "MHCamera", "Sparks");
    if (!hsparks)
    {
        cout << "WARNING - Reading of Sparks failed." << endl;
        return 2;
    }

    TH2F *hcog = (TH2F*)arr.FindObjectInCanvas("Center", "TH2F", "MHHillas");
    if (!hcog)
    {
        cout << "WARNING - Reading of MHHillas failed." << endl;
        return 2;
    }

    MHMuonPar *hmuon = (MHMuonPar*)arr.FindObjectInCanvas("MHMuonPar", "MHMuonPar", "MHMuonPar");
    if (!hmuon)
    {
        cout << "WARNING - Reading of MHMuon failed." << endl;
        return 2;
    }

    Double_t val[6];
    for (int x=1; x<hcog->GetNbinsX(); x++)
        for (int y=1; y<hcog->GetNbinsY(); y++)
        {
            Stat_t px = hcog->GetXaxis()->GetBinCenter(x);
            Stat_t py = hcog->GetYaxis()->GetBinCenter(y);
            Int_t  i  = (TMath::Nint(3*TMath::ATan2(px,py)/TMath::Pi())+6)%6;
            val[i] += hcog->GetBinContent(x, y);
        }

    Double_t inhom = TMath::RMS(6, val)*6/hcog->GetEntries()*100;
    inhom = TMath::Nint(inhom*10)/10.;
    TString inhomogen = Form("%5.1f", inhom);

    Float_t mw    = hmuon->GetMeanWidth();
    Float_t psf   = 71.685*mw - 28.323;
    psf = TMath::Nint(psf*10)/10.;
    if (psf<0)
        psf=0;
    TString PSF = Form("%5.1f", psf);
    Int_t   num = (int)hmuon->GetEntries();

    Float_t ratiodatamc = (hmuon->GetMeanSize()/7464)*100;
    TString ratio = Form("%5.1f", ratiodatamc);

    TH1 *h = (TH1*)arr.FindObjectInCanvas("Islands", "TH1F", "MHImagePar");
    if (!h)
    {
        cout << "WARNING - Reading of Islands failed." << endl;
        return 2;
    }

    Float_t quality = h->GetMean();
    quality = TMath::Nint(quality*100)/100.;
    TString islands = Form("%6.2f", quality);

    h = (TH1*)arr.FindObjectInCanvas("EffOnTheta", "TH1D", "EffOnTime");
    if (!h)
    {
        cout << "WARNING - Reading of EffOnTheta failed." << endl;
        return 2;
    }

    Float_t effon = h->Integral();
    Float_t mrate  = num/effon;
    mrate = TMath::Nint(mrate*100)/100.;
    if (mrate<0)
        mrate=0;
    TString muonrate = Form("%6.2f", mrate);
    Int_t effontime = TMath::Nint(effon);

    h = (TH1*)arr.FindObjectInCanvas("ProjDeltaT", "TH1D", "EffOnTime");
    if (!h)
    {
        cout << "WARNING - Reading of ProjDeltaT failed." << endl;
        return 2;
    }

    Int_t numsparks = (int)hsparks->GetEntries();
    Int_t numevents = (int)h->GetEntries() - numsparks;
    Int_t datarate  = (int)(numevents/effon);

    TString sparkrate = Form("%5.2f", numsparks/effon);

    if (sparkrate.Contains("inf"))
        sparkrate="0.00";

    TGraph *g = (TGraph*)arr.FindObjectInCanvas("Humidity", "TGraph", "MHWeather");
    if (!g)
    {
        cout << "WARNING - Reading of Humidity failed." << endl;
        return 2;
    }

    Double_t max = TMath::MaxElement(g->GetN(), g->GetY());
    TString maxhum = Form("%6.1f", max);


    g = (TGraph*)arr.FindObjectInCanvas("NumStars", "TGraph", "MHPointing");
    if (!g)
    {
        cout << "WARNING - Reading of NumStars failed." << endl;
//        return 2;
    }

    Double_t numstarmed = g ? TMath::Median(g->GetN(), g->GetY()) : -1;
    TString numstarsmed = Form("%5.1f", numstarmed);
    Double_t numstarrms = g ? g->GetRMS(2) : -1;
    TString numstarsrms = Form("%5.1f", numstarrms);

    g = (TGraph*)arr.FindObjectInCanvas("NumStarsCor", "TGraph", "MHPointing");
    if (!g)
    {
        cout << "WARNING - Reading of NumStarsCor failed." << endl;
//        return 2;
    }

    Double_t numcormed = g ? TMath::Median(g->GetN(), g->GetY()) : -1;
    TString numcorsmed = Form("%5.1f", numcormed);
    Double_t numcorrms = g ? g->GetRMS(2) : -1;
    TString numcorsrms = Form("%5.1f", numcorrms);

    g = (TGraph*)arr.FindObjectInCanvas("Brightness", "TGraph", "MHPointing");
    if (!g)
    {
        cout << "WARNING - Reading of SkyBrightness failed." << endl;
//        return 2;
    }

    Double_t brightnessmed = g ? TMath::Median(g->GetN(), g->GetY()) : -1;
    TString skybrightnessmed = Form("%5.1f", brightnessmed);
    Double_t brightnessrms = g ? g->GetRMS(2) : -1;
    TString skybrightnessrms = Form("%5.1f", brightnessrms);


    TString sequence = fname(TRegexp("star[0-9]+[.]root$"));
    if (sequence.IsNull())
    {
        cout << "WARNING - Could not determine sequence# from filename: " << fname << endl;
        return 2;
    }

    Int_t seq = atoi(sequence.Data()+5);

    cout << "Sequence #" << seq << endl;
    cout << "  Inhomogeneity            " << inhomogen << endl;
    cout << "  PSF [mm]                 " << PSF       << endl;
    cout << "  Island Quality           " << islands   << endl;
    cout << "  Ratio [%]                " << ratio     << endl;
    cout << "  Muon Number              " << num       << endl;
    cout << "  EffOnTime [s]            " << effontime << endl;
    cout << "  Muon Rate [Hz]           " << muonrate  << endl;
    cout << "  # of Events (w/o sparks) " << numevents << endl;
    cout << "  # of Sparks              " << numsparks << endl;
    cout << "  Rate after ImgCl [Hz]    " << datarate  << endl;
    cout << "  Rate of sparks [Hz]      " << sparkrate << endl;
    cout << "  Maximum Humidity [%]     " << maxhum    << endl;
    cout << "  Number of Stars          " << numstarsmed      << " +/- " << numstarsrms       << endl;
    cout << "  Number of cor. Stars     " << numcormed        << " +/- " << numcorrms         << endl;
    cout << "  Skybrightness            " << skybrightnessmed << " +/- " << skybrightnessrms  << endl;

    TString query;
    if (!ExistStr(serv, "fSequenceFirst", "Star", seq))
    {
        query = Form("INSERT Star SET"
                     " fSequenceFirst=%d,"
                     " fMeanNumberIslands=%s, "
                     " fRatio=%s, "
                     " fMuonNumber=%d, "
                     " fEffOnTime=%d, "
                     " fMuonRate=%s, "
                     " fPSF=%s, "
                     " fDataRate=%d, "
                     " fSparkRate=%s, "
                     " fMaxHumidity=%s ,"
                     " fNumStarsMed=%s ,"
                     " fNumStarsRMS=%s ,"
                     " fNumStarsCorMed=%s ,"
                     " fNumStarsCorRMS=%s ,"
                     " fBrightnessMed=%s ,"
                     " fBrightnessRMS=%s ,"
                     " fInhomogeneity=%s ",
                     seq, islands.Data(), ratio.Data(),
                     num, effontime,
                     muonrate.Data(), PSF.Data(),
                     datarate, sparkrate.Data(), maxhum.Data(),
                     numstarsmed.Data(), numstarsrms.Data(),
                     numcorsmed.Data(), numcorsrms.Data(),
                     skybrightnessmed.Data(), skybrightnessrms.Data(),
                     inhomogen.Data());
    }
    else
    {
        query = Form("UPDATE Star SET"
                     " fMeanNumberIslands=%s, "
                     " fRatio=%s, "
                     " fMuonNumber=%d, "
                     " fEffOnTime=%d, "
                     " fMuonRate=%s, "
                     " fPSF=%s, "
                     " fDataRate=%d, "
                     " fSparkRate=%s, "
                     " fMaxHumidity=%s, "
                     " fNumStarsMed=%s ,"
                     " fNumStarsRMS=%s ,"
                     " fNumStarsCorMed=%s ,"
                     " fNumStarsCorRMS=%s ,"
                     " fBrightnessMed=%s ,"
                     " fBrightnessRMS=%s ,"
                     " fInhomogeneity=%s "
                     " WHERE fSequenceFirst=%d ",
                     islands.Data(), ratio.Data(),
                     num, effontime,
                     muonrate.Data(), PSF.Data(),
                     datarate, sparkrate.Data(), maxhum.Data(),
                     numstarsmed.Data(), numstarsrms.Data(),
                     numcorsmed.Data(), numcorsrms.Data(),
                     skybrightnessmed.Data(), skybrightnessrms.Data(),
                     inhomogen.Data(), seq);
    }

//    cout << "Q: " << query << endl;

    if (dummy)
        return 1;

    TSQLResult *res = serv.Query(query);
    if (!res)
    {
        cout << "ERROR - Query failed: " << query << endl;
        return 2;
    }
    delete res;

    return 1;
}

int fillstar(TString fname, Bool_t dummy=kTRUE)
{
    TEnv env("sql.rc");

    MSQLServer serv(env);
    if (!serv.IsConnected())
    {
        cout << "ERROR - Connection to database failed." << endl;
        return 0;
    }

    cout << "fillstar" << endl;
    cout << "---------" << endl;
    cout << endl;
    cout << "Connected to " << serv.GetName() << endl;
    cout << "File: " << fname << endl;
    cout << endl;

    return Process(serv, fname, dummy);
}
