/********************************************************************\

  DAQReadout.cc

  Main DAQ routines.
   
  Sebastian Commichau, Oliver Grimm
  
\********************************************************************/

#include "DAQReadout.h"
#include "SlowData.h"

#define TMPNAME "/tmp/__tmp__drsdaq__"

static const char* daq_state_str[] = {"active", "stopped"};
static const char* daq_runtype_str[] = {"data", "pedestal", "reserved", "test"};

static const struct CL_Struct { const char *Name;    
                	  void (DAQReadout::*CommandPointer)();
			  bool NeedNotBusy;
			  const char *Parameters;
			  const char *Help;
  } CommandList[] = 
  {{"board", &DAQReadout::cmd_board, true, "<i> [j] | <all>" ,"Address board i, boards i-j, all boards"},
   {"status", &DAQReadout::cmd_status, false, "[daq|drs]", "Show DAQ/DRS status information"},
   {"freq", &DAQReadout::cmd_freq, true, "<GHz> [reg]", "Set DRS sampling frequency (regulated)"},
   {"calib", &DAQReadout::cmd_calib, true, "<t_f> <c_f> [dir]", "Response calibration"},
   {"trigger", &DAQReadout::cmd_trigger, true, "<on|off>", "Hardware trigger on or off"},
   {"delayed", &DAQReadout::cmd_delayed, true, "<on|off>", "Switch delayed start on or off"},
   {"wmode", &DAQReadout::cmd_wmode, true, "<0|1>", "Set DRS wave mode"},
   {"rmode", &DAQReadout::cmd_rmode, true, "<0|1>", "Set DRS readout mode"},
   {"mode", &DAQReadout::cmd_mode, true, "<single|continuous>", "Set DRS single shot or continuous mode"},
   {"read", &DAQReadout::cmd_read, false, "<brd chip chan> [res]", "Read current data (and restart if DAQ not active)"},
   {"take", &DAQReadout::cmd_take, false, "<d|p|t> [n] [source]", "Start run (data, pedestal or test) with n events"},
   {"events", &DAQReadout::cmd_events, false, "", "Number of events in current run"},
   {"start", &DAQReadout::cmd_start, true, "", "Start domino wave"},
   {"stop", &DAQReadout::cmd_stop, false, "", "Issue soft trigger and stop DAQ"},
   {"test", &DAQReadout::cmd_test, true, "[2e]<blt32|blt64> [n]", "Test read access of VMEbus (n blocks)"},
   {"regtest", &DAQReadout::cmd_regtest, true, "", "DRS register test"},
   {"ramtest", &DAQReadout::cmd_ramtest, true, "", "DRS RAM integrity and speed test"},
   {"led", &DAQReadout::cmd_led, true, "<on|off>", "Turn LED on or off"},
   {"config", &DAQReadout::cmd_config, false, "", "Print drsdaq configuration"},
   {"serial", &DAQReadout::cmd_serial, true, "<i> <n>", "Set serial# of board <i> to <n> (experts only)"},
   {"disk", &DAQReadout::cmd_disk, false, "" ,"Remaining disk space"},
   {"uptime", &DAQReadout::cmd_uptime, false, "", "Get DAQ uptime"},		  
   {"exit", &DAQReadout::cmd_exit, false, "", "Exit program"},
   {"fmode", &DAQReadout::cmd_fmode, false, "[off|active|targ]", "Set or get feedback mode"},
   {"faverage", &DAQReadout::cmd_faverage, false, "[n]", "Set ot get number of averages for feedback"},
   {"fgain", &DAQReadout::cmd_fgain, false, "[gain]", "Set ot get feedback gain"},
   {"ftarget", &DAQReadout::cmd_ftarget, false, "[brd chip chan]", "Set or get target value (also 'all' supported)"},
   {"fresponse", &DAQReadout::cmd_fresponse, false, "[voltage]", "Start response measurement with given voltage difference"},
   {"fconfig", &DAQReadout::cmd_fconfig, false, "", "Print feedback configuration"},
   {"help", &DAQReadout::cmd_help, false, "", "Print help"}};


// -----------------------------------------------
// *****  Constructor: Class initialisation  *****
// -----------------------------------------------
//
 
DAQReadout::DAQReadout(const char *Configfile) {
   
  time(&StartTime);  // Start time of DAQ

  // Initialize status structure
  daq_state	     = stopped;
  daq_runtype	     = data;
  Socket	     = -1;
  Exit		     = false;
  CalibrationRead    = false;
  NumEvents	     = 0;
  NumEventsRequested = 100;
  NumBoards	     = 0;
  FirstBoard	     = 0;
  LastBoard	     = -1;
  CmdFromSocket      = false;
      
  // Read configuration file
  FILE *File;
  if ((File = fopen(Configfile,"r")) == NULL) {
    printf("Error: Could not open drsdaq configuration file '%s'\n", Configfile);
  }
  else {
    printf("Reading drsdaq configuration file %s\n", Configfile);
    ReadCard("LogFile",             fLogFile,           's', File);
    ReadCard("MaxLogLines",        &fMaxLogLines,       'U', File);
    ReadCard("RawDataPath",         fRawDataPath,       's', File);
    ReadCard("FirstSample",        &fFirstSample,       'I', File);
    ReadCard("Samples",     	   &fSamples,        	'U', File);
    ReadCard("MinDiskSpaceMB",     &fMinDiskSpaceMB,    'U', File);
    ReadCard("MaxFileSizeMB",      &fMaxFileSizeMB,     'I', File);
    ReadCard("CCPort",             &fCCPort,            'I', File);
    ReadCard("FirstVMESlot",       &fFirstVMESlot,      'I', File);
    ReadCard("LastVMESlot",        &fLastVMESlot,       'I', File);
    ReadCard("HVFeedbackConfig",    fHVFeedbackConfig,  's', File);
    fclose(File);
  }
  if (fFirstSample < 0 || fFirstSample > kNumberOfBins || fSamples > kNumberOfBins) {
    PrintMessage("Warning: Sample range in configuration beyond limits, setting to full range\n");
    fFirstSample = kNumberOfBins;
    fSamples = kNumberOfBins;
  }
  
  // Truncate log file to given number of lines
  char ShellCmd[MAX_COM_SIZE]; 
  snprintf(ShellCmd, sizeof(ShellCmd), "tail --lines=%u %s >%s; cp %s %s; rm %s", fMaxLogLines, fLogFile, TMPNAME, TMPNAME, fLogFile, TMPNAME);
  if (system(ShellCmd) != 0) printf("Warning: Could not truncate log file '%s' to %u lines\n", fLogFile, fMaxLogLines);

  // Open log file
  if ((Logfile = fopen(fLogFile, "a")) == NULL) printf("Warning: Could not open log file '%s'\n", fLogFile);
  PrintMessage(MsgToLog,"********** Logging started **********\n");
  
  // Print configuration
  cmd_config();

  // Create DRS instance and perform initial scan
  drs = new DRS();
  drs->SetFirstVMESlot(fFirstVMESlot);
  drs->SetLastVMESlot(fLastVMESlot);
  drs->InitialScan();

  // Allocate headers and initialise to zero
  RHeader = new RunHeader;
  memset(RHeader, 0, sizeof(RunHeader));
  EHeader = new EventHeader;
  memset(EHeader, 0, sizeof(EventHeader));
  
  // Scan for DRS boards
  DRSFreq = new float [drs->GetNumberOfBoards()];
  if (drs->GetNumberOfBoards()==0) PrintMessage("No DRS boards found - check VME crate and configuration file!\n");

  for (int i=0; i<drs->GetNumberOfBoards(); i++) {
    PrintMessage("Init. mezz. board %2d on VME slot %2d %s, serial #%d, firmware revision %d\n", 
      i, (drs->GetBoard(i)->GetSlotNumber() >> 1)+2, ((drs->GetBoard(i)->GetSlotNumber() & 1) == 0) ? "upper" : "lower", 
      drs->GetBoard(i)->GetCMCSerialNumber(), drs->GetBoard(i)->GetFirmwareVersion());
    NumBoards++;
    LastBoard++;
    drs->GetBoard(i)->Init();
    drs->GetBoard(i)->SetRotation(false);
    DRSFreq[i] = 0;
  }
  BStruct  = new BoardStructure [NumBoards == 0 ? 1:drs->GetNumberOfBoards()];
  memset(BStruct, 0, sizeof(BoardStructure)*(NumBoards == 0 ? 1:drs->GetNumberOfBoards()));

  WaveForm = new short [NumBoards == 0 ? 1:NumBoards][kNumberOfChips][kNumberOfChannels][kNumberOfBins];
  TriggerCell = new int [NumBoards == 0 ? 1:NumBoards][kNumberOfChips] ();  // Zero initialised
  
  // Create instance of HV feedback (must be called after CMC board detection)
  HVFB	  = new HVFeedback(this, fHVFeedbackConfig);
  
  // Create instance of slow data class for DAQ
  char Filename[MAX_PATH];
  snprintf(Filename,sizeof(Filename),"%s/SlowData/", fRawDataPath);
  SlowDataClass = new SlowData("DAQ", Filename);
  if(SlowDataClass->ErrorCode != 0) {
   PrintMessage("Warning: Could not open DAQ slowdata file (%s)\n", strerror(SlowDataClass->ErrorCode));
  }
}

// ------------------------
// *****  Destructor  *****
// ------------------------

DAQReadout::~DAQReadout() {
  delete SlowDataClass;
  delete RHeader;     delete EHeader;
  delete drs;	      delete HVFB;
  delete[] DRSFreq;   delete[] BStruct;
  delete[] WaveForm;  delete[] TriggerCell;
  
  PrintMessage(MsgToLog,"********** Logging ended **********\n\n");
  if(Logfile) {
    if(!fclose(Logfile)) printf("Closing logfile\n");
    else perror("Error closing logfile");
  }
}

// --------------------------------
// *****  Command evaluation  *****
// --------------------------------

void DAQReadout::CommandControl(char *Command) {

  if (strlen(Command)==0) return;  // Ignore empty commands

  if(Command[0]=='.') {   // Shell command
    system(&(Command[1]));
    return;
  }

  for(int i=0; i<MAX_NUM_TOKEN; i++) Param[i] = "";  // All pointers point initially to empty string
  NParam = ParseInput(Command, Param);

  for(CmdNumber=0; CmdNumber<sizeof(CommandList)/sizeof(CL_Struct); CmdNumber++)
    if (Match(Param[0], CommandList[CmdNumber].Name)) {
      if(CommandList[CmdNumber].NeedNotBusy && daq_state==active) PrintMessage("DAQ is busy\n");
      else if(CommandList[CmdNumber].NeedNotBusy && NumBoards==0) PrintMessage("No mezzanine boards available\n");
      else (this->*CommandList[CmdNumber].CommandPointer)();
      return;  
    }
  PrintMessage("Unknown command '%s'\n",Param[0]);
  return;
}
  	  
// Get uptime
void DAQReadout::cmd_uptime() {
  time_t ActualT;
  time (&ActualT);
  PrintMessage("%d:%02d:%02d\n", (int) difftime(ActualT, StartTime)/3600, ((int) difftime(ActualT, StartTime)/60)%60, (int) difftime(ActualT, StartTime)%60);
} 

// Print disk space
void DAQReadout::cmd_disk() {
  PrintMessage("Free disk space (%s) [MB]: %lu\n", fRawDataPath, CheckDisk(fRawDataPath));
} 

// Print current number of events
void DAQReadout::cmd_events() {
  if(daq_state != active) PrintMessage("DAQ not active.\n");
  else PrintMessage("Current number of events: %d (of %d requested)\n", NumEvents, NumEventsRequested);
} 

// Print DAQ configuration
void DAQReadout::cmd_config() {
  PrintMessage("LogFile: %s\tMaxLogLines: %u\tRawDataPath: %s\n"
      	       "FirstSample: %d\tSamples: %u\n"
     	       "MinDiskSpaceMB: %u\tMaxFileSizeMB: %d\tCCPort: %d\n"
      	       "FirstVMESlot: %d\t\tLastVMESlot: %d\n"
	       "HVFeedbackConfig: %s\n",
    fLogFile,fMaxLogLines,fRawDataPath,fFirstSample,fSamples,fMinDiskSpaceMB,
    fMaxFileSizeMB,fCCPort,fFirstVMESlot,fLastVMESlot,fHVFeedbackConfig);
}

// Start DAQ 
void DAQReadout::cmd_take() {
  
  if(!Match(Param[1],"test")) {
    if (daq_state==active || NumBoards==0) {
      PrintMessage("DAQ is busy or no boards available.\n");
      return;
    }
    if (!IsDRSFreqSet()) return;
    if (!CalibrationRead && !ReadCalibration()) {
      PrintMessage("Cannot start run if response calibration not read.\n");
      return;
    }
  }
    
  if (Match(Param[1],"data")) {
    HWTrigger(1);
    daq_runtype = data;
  } 
  else if (Match(Param[1],"pedestal")) {
    HWTrigger(0);
    daq_runtype = pedestal;	
  } 
  else if (Match(Param[1],"test")) {
    daq_runtype = test;	
  }
  else {
    PrintUsage();
    return;
  }

  if (NParam==3) NumEventsRequested = atoi(Param[2]);
  if (NParam==4) {
    NumEventsRequested = atoi(Param[2]);
    strncpy(RHeader->Description, Param[3], sizeof(RHeader->Description));
  }
  else snprintf(RHeader->Description,sizeof(RHeader->Description),"DUMMY");

  // Determine new run number using the file RUN_NUM_FILE
  FILE *RunNumFile = fopen(RUN_NUM_FILE,"r+");
  if(RunNumFile == NULL) {
    PrintMessage("Error: Could not open file '%s' that contains the last run number (%s)\n",RUN_NUM_FILE,strerror(errno));
    return;
  }
  if(fscanf(RunNumFile,"%u", &RunNumber) != 1 ) {
    PrintMessage("Error: Could not read run number from file '%s'\n",RUN_NUM_FILE);
    fclose(RunNumFile);
    return;
  }
  RunNumber++;
  rewind(RunNumFile);
  if((fprintf(RunNumFile,"%.8u   ",RunNumber) < 0) || (fclose(RunNumFile)!=0)) {
    PrintMessage("Error: Could not write to or close run number file '%s'\n",RUN_NUM_FILE);
    PrintMessage("*** This is a serious error because run numbers will get mixed. Fix it. DAQ will terminate.\n");
    throw;
  }

  // Create DAQ thread
  if ((pthread_create(&thread_DAQ, NULL, (void * (*)(void *)) DAQ,(void *) this)) != 0)
    PrintMessage("pthread_create failed with DAQ thread (%s)\n",strerror(errno));
  else {
    daq_state = active;
    Stop = false;
    pthread_detach(thread_DAQ);
  }  
}
  
// Start DRS
void DAQReadout::cmd_start() {
  if (IsDRSFreqSet()) {
    StartDRS();
    PrintMessage("Domino wave started\n");
  }
}

// RAM test
void DAQReadout::cmd_ramtest() {
  for (int i=FirstBoard; i<=LastBoard; i++) {
    PrintMessage("RAM integrity and speed test (board #%d):\n",i);
    (drs->GetBoard(i))->RAMTest(3);
  }
} 

// Register test
void DAQReadout::cmd_regtest() {
  for (int i=FirstBoard; i<=LastBoard; i++) {
    PrintMessage("Register test (board #%d):\n",i);
    (drs->GetBoard(i))->RegisterTest();
  }
}

// Test VME transfer
void DAQReadout::cmd_test() {
  int Type=-1, i;
  
  if (Match(Param[1], "2eblt64")) Type = 2;
  else if (Match(Param[1], "blt32")) Type = 0;
  else if (Match(Param[1], "blt64")) Type = 1;
  else {
    PrintMessage("Unknown type for testing.\n");
    return;
  }
  
  if (NumBoards) for (i=FirstBoard; i<=LastBoard; i++) {
        PrintMessage("BLT test started (board #%d)\n",i);
        (drs->GetBoard(i))->TestRead(Param[2][0] && atoi(Param[2])<=10000 && atoi(Param[2])>0 ? atoi(Param[2]):1, Type);
      }
  else PrintMessage("No DRS boards available\n");
} 

// Stop DAQ 
void DAQReadout::cmd_stop() {
  if(!daq_state==active && !IsDRSBusy()) PrintMessage("Nothing to stop\n");
  if (daq_state==active) StopRun();
  if (IsDRSBusy()) {
    StopDRS();
    PrintMessage("Domino wave stopped\n");
  }
} 
  
// Read current data
// For socket transmission: all numbers must be separated by exactly one
// whitespace; the first number is the number of numbers that follow, the
// second number the sampling frequency in GHz, the third the conversion factor 

void DAQReadout::cmd_read() {
  if(NumBoards==0) {
    PrintMessage("No mezzanine boards available\n");
    return;
  }
  if (NParam<4) {
    PrintUsage();
    return;
  }      
  if (atoi(Param[1])>LastBoard || atoi(Param[1])<FirstBoard) {
    PrintMessage("Error: Board number out of range\n");
    return;
  }
  if (atoi(Param[3])<0 || atoi(Param[3])>=kNumberOfChannels) {
    PrintMessage("Error: Channel number out of range\n");
    return;
  }
  if (atoi(Param[2])<0 || atoi(Param[2])>=kNumberOfChips) {
    PrintMessage("Error: Chip number out of range\n");
    return;
  }
  
  if (daq_state!=active) {
    if (!CalibrationRead) ReadCalibration();
    if(NParam==5) StopDRS();
    ReadCalibratedDRSData();
    if(NParam==5) StartDRS();
  }  
  PrintMessage(CmdFromSocket ? MsgToSocket:MsgToConsole|MsgToLog, "==START== %d %.2f %.2f ",kNumberOfBins+2,DRSFreq[atoi(Param[1])],drs->GetBoard(atoi(Param[1]))->GetPrecision());
  for (int k=0; k<kNumberOfBins; k++) PrintMessage(CmdFromSocket ? MsgToSocket:MsgToConsole|MsgToLog, "%.1f ", 
   	(float) WaveForm[atoi(Param[1])][atoi(Param[2])][atoi(Param[3])][(k+TriggerCell[atoi(Param[1])][atoi(Param[2])])%kNumberOfBins]);
  PrintMessage(CmdFromSocket ? MsgToSocket:MsgToConsole|MsgToLog, "==END==");
  PrintMessage(CmdFromSocket ? MsgToSocket:MsgToConsole|MsgToLog, "\n");
} 

// Set Domino mode
void DAQReadout::cmd_mode() {
  if (Match(Param[1],"continuous")) SetDOMINOMode(1);
  else if (Match(Param[1],"single")) SetDOMINOMode(0);
  else PrintUsage();
} 

// Set Domino readout mode
void DAQReadout::cmd_rmode() {
  if (Match(Param[1],"1")) SetDOMINOReadMode(1);
  else if (Match(Param[1],"0")) SetDOMINOReadMode(0);
  else PrintUsage();
} 

// Set Domino wave mode
void DAQReadout::cmd_wmode() {
  if (Match(Param[1],"1")) SetDOMINOWaveMode(1);
  else if (Match(Param[1],"0")) SetDOMINOWaveMode(0);
  else PrintUsage();
} 

// Switch delayed start on/off
void DAQReadout::cmd_delayed() {
  if (Match(Param[1],"on")) SetDelayedStart(1);
  else if (Match(Param[1],"off")) SetDelayedStart(0);
  else PrintUsage();
} 

// Set trigger mode
void DAQReadout::cmd_trigger() {
  if (Match(Param[1],"on")) HWTrigger(1);
  else if (Match(Param[1],"off")) HWTrigger(0);
  else PrintUsage();
} 

// Set serial number of board
void DAQReadout::cmd_serial() {    
  if (NParam==4 && Match(Param[3], "expert")) {
    if ((atoi(Param[1]) < FirstBoard) || (atoi(Param[1]) > LastBoard))
      PrintMessage("Board number out of range (%d...%d)!\n",FirstBoard,LastBoard);
    else if (atoi(Param[2]) < 100 || atoi(Param[2]) >= 1000) 
      PrintMessage("Serial number out of range (100...999)!\n");
    else {
      PrintMessage("Flashing EEPROM of board %d...\n", atoi(Param[1]));
      (drs->GetBoard(atoi(Param[1])))->FlashEEPROM(atoi(Param[2]));
    }
  }
  else PrintMessage("You are not allowed to change the serial number!\n");
} 

// Do internal calibration
void DAQReadout::cmd_calib() {
  if (NParam==4 && atof(Param[1]) && atof(Param[2]))
    CalibrateDRS((char *) Param[3],atof(Param[1]),atof(Param[2])); 
  else if (NParam==3 && atof(Param[1]) && atof(Param[2]))
    CalibrateDRS(NULL,atof(Param[1]),atof(Param[2])); 
  else PrintUsage();
}

// Set DRS sampling frequency
void DAQReadout::cmd_freq() {    
  if (NParam>=2 && atof(Param[1])) {
    SetDRSFrequency(atof(Param[1]), NParam==2 ? false : true);
    CalibrationRead = false;
  }
  else PrintUsage();
} 

// Set LED
void DAQReadout::cmd_led() {
  if (Match(Param[1], "on") || Match(Param[1], "off"))
    for (int i=FirstBoard; i<=LastBoard; i++)
      (drs->GetBoard(i))->SetLED(Match(Param[1], "on") ? 1 : 0);     
  else PrintUsage();
}

// Print status
void DAQReadout::cmd_status() {
  
  double freq;
  
  if(NParam==1 || Match(Param[1],"daq")) {
    PrintMessage("********** DAQ STATUS **********\n"
 	        " DAQ: %s\n"
  	        " Run number: %d\n"
 	        " Run type: %s\n"
	        " Event: %d\n"
      	        " Requested events per run: %d\n"
 	        " Storage directory: %s\n"
 	        " Disk space: %lu MByte\n"
 	      	" Socket state: %s\n"                              
              	" Total number of CMC boards: %d\n"
	        " Active CMC boards: %d\n",
      daq_state_str[daq_state], daq_state==active ? (int) RunNumber:-1,
      daq_runtype_str[daq_runtype], NumEvents,
      NumEventsRequested, fRawDataPath,
      CheckDisk(fRawDataPath), Socket==-1 ? "disconnected":"connected",
      NumBoards, LastBoard - FirstBoard + 1);

    for (int i=FirstBoard;i<=LastBoard;i++)
      PrintMessage(" Frequency of board %d set: %s\n",i,(DRSFreq[i]!=0 ? "yes":"no"));
  }
  
  if(NParam==1 || Match(Param[1],"drs")) {
    PrintMessage("\n********** DRS STATUS **********\n");
    if (NumBoards) {
      for (int i=FirstBoard; i<=LastBoard; i++) {

	PrintMessage(" Mezz. board index:    %d\n"
                    " Slot:                 %d %s\n",i,((drs->GetBoard(i))->GetSlotNumber() >> 1)+2,((drs->GetBoard(i))->GetSlotNumber() & 1)==0 ? "upper":"lower");
	PrintMessage(" Chip version:         DRS%d\n"
                    " Board version:        %d\n"
                    " Serial number:        %d\n"
                    " Firmware revision:    %d\n"
                    " Temperature:          %1.1lf C\n"
                    " Status reg.:          0X%08X\n", 
		    (drs->GetBoard(i))->GetChipVersion(),
		    (drs->GetBoard(i))->GetCMCVersion(),
		    (drs->GetBoard(i))->GetCMCSerialNumber(),
		    (drs->GetBoard(i))->GetFirmwareVersion(),
		    (drs->GetBoard(i))->GetTemperature(),
		    (drs->GetBoard(i))->GetStatusReg());


	if ((drs->GetBoard(i))->GetStatusReg() & BIT_RUNNING)
	  PrintMessage("   Domino wave running\n");
	if ((drs->GetBoard(i))->GetStatusReg() & BIT_NEW_FREQ1)
	  PrintMessage("   New Freq1 ready\n");
	if ((drs->GetBoard(i))->GetStatusReg() & BIT_NEW_FREQ2)
	  PrintMessage("   New Freq2 ready\n");

	PrintMessage(" Control reg.:         0X%08X\n", (drs->GetBoard(i))->GetCtrlReg());
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_AUTOSTART)
	  PrintMessage("   AUTOSTART enabled\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_DMODE)
	  PrintMessage("   DMODE circular\n");
	else
	  PrintMessage("   DMODE single shot\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_LED)
          PrintMessage("   LED\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_TCAL_EN)
	  PrintMessage("   TCAL enabled\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_ZERO_SUPP)
	  PrintMessage("   ZERO_SUPP enabled\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_FREQ_AUTO_ADJ)
	  PrintMessage("   FREQ_AUTO_ADJ enabled\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_ENABLE_TRIGGER)
	  PrintMessage("   ENABLE_TRIGGER\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_LONG_START_PULSE)
	  PrintMessage("   LONG_START_PULSE\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_DELAYED_START)
	  PrintMessage("   DELAYED_START\n");
	if ((drs->GetBoard(i))->GetCtrlReg() & BIT_ACAL_EN)
	  PrintMessage("   ACAL enabled\n");
	PrintMessage(" Trigger bus:          0X%08X\n", (drs->GetBoard(i))->GetTriggerBus());
	if ((drs->GetBoard(i))->IsBusy()) {
	  (drs->GetBoard(i))->ReadFrequency(0, &freq);
	  PrintMessage(" Frequency0:           %1.4lf GHz\n", freq);
	  (drs->GetBoard(i))->ReadFrequency(1, &freq);
	  PrintMessage(" Frequency1:           %1.4lf GHz\n", freq);
	} 
	else PrintMessage(" Domino wave stopped\n");
      }
    }
    else PrintMessage("No DRS boards available!\n\n");
  }
}

// Adress DRS boards
void DAQReadout::cmd_board() {
  if (Match(Param[1],"all")) {
    FirstBoard = 0;
    LastBoard = drs->GetNumberOfBoards()-1;
  } 
  else if (NParam==2 && atoi(Param[1])>=0 && atoi(Param[1])<NumBoards) {
    FirstBoard = atoi(Param[1]);
    LastBoard = FirstBoard;
  }
  else if (NParam==3 && atoi(Param[1])>=0 && atoi(Param[1])<NumBoards && 
           atoi(Param[2])>0 && atoi(Param[2])<NumBoards) {
    FirstBoard = atoi(Param[1]);
    LastBoard = atoi(Param[2]);
  }
  else PrintMessage("Cannot address board(s), out of range.\n");
  CalibrationRead = false;
} 

// Print help (only to console or socket, not to log file)
void DAQReadout::cmd_help() {
  char Buffer[MAX_COM_SIZE];
  for(unsigned int i=0; i<sizeof(CommandList)/sizeof(CL_Struct); i++) {
    snprintf(Buffer, sizeof(Buffer), "%s %s", CommandList[i].Name, CommandList[i].Parameters);
    PrintMessage(CmdFromSocket ? MsgToSocket:MsgToConsole,"%-28s%s\n", Buffer, CommandList[i].Help);
  }     
  PrintMessage(CmdFromSocket ? MsgToSocket:MsgToConsole,".<command>                  Execute shell command\n\n"
   "Items in <> are mandatory, in [] optional, | indicates mutual exclusive or.\n"
   "Test data can also be written if no DRS boards are available.\n"
   "Strings containing spaces have to be enclosed in \"double quotes\".\n"); 
}

// Exit programm
void DAQReadout::cmd_exit() {
  if (CmdFromSocket) {
     PrintMessage("Exit command not allowed over socket.\n");
     return;
  }  	
  if (daq_state==active) PrintMessage("Issue \"stop\" first to stop daq\n");
  else {
    Exit = true;
    if(SocketThread != NULL) pthread_kill(*SocketThread, SIGUSR1);
  }
}

// Set/get mode of feedback
void DAQReadout::cmd_fmode() {
  if(Match(Param[1],"off")) HVFB->SetFBMode(FB_Off);
  if(Match(Param[1],"active")) HVFB->SetFBMode(FB_Active);
  if(Match(Param[1],"targets")) HVFB->SetFBMode(FB_Targets);
  HVFB->GetFBMode();
}

// Set/get current number of events
void DAQReadout::cmd_faverage() {
  if(NParam==1) PrintMessage("Current number of feedback events: %u   (acting when %u events are reached)\n",
                       HVFB->GetCurrentCount(), HVFB->GetNumAverages());
  else if(atoi(Param[1])>=0) HVFB->SetNumAverages(atoi(Param[1]));
  else PrintUsage();
}

// Set/get feedback gain
void DAQReadout::cmd_fgain() {
  if(NParam==2) HVFB->SetGain(atof(Param[1]));
  PrintMessage("Feedback gain is %.2f\n", HVFB->GetGain());
}

// Set/get target value
void DAQReadout::cmd_ftarget() {
  if(NParam==1) HVFB->GetTargets();
  else if(NParam!=5) PrintUsage();
  else for (int i=FirstBoard; i<=LastBoard; i++)
         for (int j=0; j<kNumberOfChips; j++)
           for (int k=0; k<kNumberOfChannels; k++)
             if ((atoi(Param[1])==i || Match(Param[1],"all")) &&
	         (atoi(Param[2])==j || Match(Param[2],"all")) &&
	         (atoi(Param[3])==k || Match(Param[3],"all")))
      	      HVFB->SetTarget(i,j,k,atof(Param[4]));
}

// Start response measurement
void DAQReadout::cmd_fresponse() {
  if(NParam==1) HVFB->GetResponse();
  else if(atof(Param[1])) HVFB->MeasureResponse(atof(Param[1]));
  else PrintUsage();
}

// Print feedback configuration
void DAQReadout::cmd_fconfig() {
  HVFB->PrintConfig();
}

// ----------------------------------------------
// *****  Utility function for DRS control  *****
// ----------------------------------------------

// Start domino wave
void DAQReadout::StartDRS() {
  for (int i=FirstBoard; i<=LastBoard; i++) drs->GetBoard(i)->StartDomino();
}

// Stop domino wave
void DAQReadout::StopDRS() {
  for (int i=FirstBoard; i<=LastBoard; i++) drs->GetBoard(i)->SoftTrigger();
}

// Transfer data to memory
void DAQReadout::ReadCalibratedDRSData() {

  for (int i=FirstBoard; i<=LastBoard; i++) {
    (drs->GetBoard(i))->TransferWaves(kNumberOfChannels*kNumberOfChips); 
    for (int j=0; j<kNumberOfChannels; j++) {
       drs->GetBoard(i)->GetWave(0, j, WaveForm[i][0][j], true); // Chip #1
       TriggerCell[i][0] = drs->GetBoard(i)->GetTriggerCell((unsigned int) 0);
       drs->GetBoard(i)->GetWave(1, j, WaveForm[i][1][j], true); // Chip #2
       TriggerCell[i][1] = drs->GetBoard(i)->GetTriggerCell((unsigned int) 1);
    }
  }
}

// Read calibration file
bool DAQReadout::ReadCalibration() {

  char dir[MAX_COM_SIZE];

  getcwd(dir, sizeof(dir));
  strcat(dir,"/calib");
  for (int i=FirstBoard; i<=LastBoard; i++) {
    (drs->GetBoard(i))->SetCalibrationDirectory(dir);
    PrintMessage("Reading response calibration file for board %d from: \"%s\"\n", i, dir);
    for (int Chip=0; Chip<kNumberOfChips; Chip++)
      if (drs->GetBoard(i)->GetResponseCalibration()->ReadCalibration(Chip)==false) {
        CalibrationRead = false;
	return false;
      }
  }
  CalibrationRead = true;
  return true;
}

// Check if calibration file has been read
bool DAQReadout::IsCalibrationRead() {

  for (int i=FirstBoard; i<=LastBoard; i++) {
    for (int Chip=0; Chip<kNumberOfChips; Chip++) 
      if (!(drs->GetBoard(i)->GetResponseCalibration()->IsRead(Chip))) {
	PrintMessage("Warning: Response calibration of board %d chip %d not yet read!\n",i,Chip);
	return false;
      }
  }
  return true;
}

// Stop DAQ
void DAQReadout::StopRun() {

  if(daq_state != active) PrintMessage("DAQ is not active.\n");
  else {
    Stop = true;
    PrintMessage("DAQ will stop.\n");
  }
}

// Set DOMINO mode 
void DAQReadout::SetDOMINOMode(int mode) {
 
  if (NumBoards) 
    for (int i=FirstBoard; i<=LastBoard; i++) {
      (drs->GetBoard(i))->SetDominoMode(mode==1 ? 1:0);
      PrintMessage("Domino mode of board %d switched to %s.\n",i,mode==1 ? "continuous":"single shot");
    } 
  else PrintMessage("No DRS boards available\n");
}

// Set DOMINO readout mode 
void DAQReadout::SetDOMINOReadMode(int mode) {

  if (NumBoards) 
    for (int i=FirstBoard; i<=LastBoard; i++) {
      (drs->GetBoard(i))->SetReadoutMode(mode==1 ? 1:0);
      PrintMessage("Start readout of board %d from %s.\n",i,mode==1 ? "first bin":"stop position");
    } 
  else PrintMessage("No DRS boards available\n");
}

// Set DOMINO wave mode 
void DAQReadout::SetDOMINOWaveMode(int mode) {

  if (NumBoards) 
    for (int i=FirstBoard; i<=LastBoard; i++) {
      (drs->GetBoard(i))->SetDominoActive(mode==1 ? 1:0);
      PrintMessage("Domino wave of board %d is %s during readout\n",i,mode==1 ? "running":"stopped");
    } 
  else PrintMessage("No DRS boards available\n");
}

// Delayed start on/off 
void DAQReadout::SetDelayedStart(int mode) {

  if (NumBoards) 
    for (int i=FirstBoard; i<=LastBoard; i++) {
      (drs->GetBoard(i))->SetDelayedStart(mode==1 ? 1:0);
       PrintMessage("Delayed start of board %d is %s\n",i,mode==1 ? "on":"off");
    } 
  else PrintMessage("No DRS boards available\n");
}

// Enable hardware trigger of all boards 
void DAQReadout::HWTrigger(int mode) {

  if (NumBoards) 
    for (int i=FirstBoard; i<=LastBoard; i++) {
      drs->GetBoard(i)->EnableTrigger(mode==1 ? 1:0);
      PrintMessage("Hardware trigger of board %d %s\n",i,mode==1 ? "enabled":"disabled");
    }
  else PrintMessage("No DRS boards available\n");
}

// Set DRS sampling frequency
void DAQReadout::SetDRSFrequency(double freq, bool Regulation) {

  double currentfreq;

  PrintMessage("Setting frequency %s regulation:\n",Regulation ? "with":"without");
  for (int i=FirstBoard; i<=LastBoard; i++) { 
    drs->GetBoard(i)->SetDebug(1);

    if (Regulation ? drs->GetBoard(i)->RegulateFrequency(freq) : drs->GetBoard(i)->SetFrequency(freq)) {
      drs->GetBoard(i)->ReadFrequency(0, &currentfreq); 
      DRSFreq[i] = freq;
      PrintMessage("Domino wave of board %d is running at %1.3lf GHz\n",i,currentfreq);
    } else {
      DRSFreq[i] = 0;
      PrintMessage("Warning: Domino wave of board %d has changed but not reached the requested value\n",i);
    }
  }
}

// Do internal calibration
void DAQReadout::CalibrateDRS(char *dir, double trigfreq, double calibfreq) {

  int i,j;
  char str[MAX_COM_SIZE];
  DIR *pdir;
      
  if (NumBoards) {
    if(dir!=NULL) {
      if ((pdir=opendir(str))==0){
	PrintMessage("Error: target directory \"%s\" does not exist!\n",str);
        return;
      }
      closedir(pdir);
      snprintf(str,sizeof(str),"%s",dir);
      PrintMessage("Target: \"%s\"\n",str);
    }
    else {
      getcwd(str, sizeof(str));
      strcat(str,"/calib");
      PrintMessage("Taking default target: \"%s/\"\n",str);
    }
        
    for (i=FirstBoard; i<=LastBoard; i++) {
      drs->GetBoard(i)->Init();
      drs->GetBoard(i)->SetFrequency(calibfreq);
      drs->GetBoard(i)->SoftTrigger();
            
      PrintMessage("Creating calibration of board %d\n", drs->GetBoard(i)->GetCMCSerialNumber());

      drs->GetBoard(i)->EnableTcal(1);
      PrintMessage("Tcal enabled");

      if (drs->GetBoard(i)->GetChipVersion() == 3)
	drs->GetBoard(i)->GetResponseCalibration()->SetCalibrationParameters(1,21,0,20,0,0,0,0,0);
      else
	drs->GetBoard(i)->GetResponseCalibration()->SetCalibrationParameters(1,36,110,20,19,40,15,trigfreq,0);
            
      drs->GetBoard(i)->SetCalibrationDirectory(str);
      PrintMessage("Storage directory \"%s\"\n",str);

      for (j=0;j<2;j++) {
	drs->GetBoard(i)->GetResponseCalibration()->ResetCalibration();
	PrintMessage("Calibration reset done.\n");

	while (!drs->GetBoard(i)->GetResponseCalibration()->RecordCalibrationPoints(j)) {}
	PrintMessage("Record calibration points done.\n");
	while (!drs->GetBoard(i)->GetResponseCalibration()->FitCalibrationPoints(j)) {}
	PrintMessage("Calibration points fitted.\n");
	while (!drs->GetBoard(i)->GetResponseCalibration()->OffsetCalibration(j)) {}
	PrintMessage("Offset calibration done.\n");

	if (!drs->GetBoard(i)->GetResponseCalibration()->WriteCalibration(j))
	  break;
      }        
      drs->GetBoard(i)->Init(); // Reset linear range -0.2 ... 0.8 V
    } // Loop over boards   
  }
  else PrintMessage("No DRS boards available\n");
}

// Check if DRS is sampling
bool DAQReadout::IsDRSBusy() {

  for (int i=FirstBoard; i<=LastBoard; i++)     
    if ((drs->GetBoard(i))->IsBusy()) return true;
  return false;
}

// Check if DRS frequency is set
bool DAQReadout::IsDRSFreqSet() {

  for (int i=FirstBoard;i<=LastBoard;i++)
    if (DRSFreq[i]==0) {
      PrintMessage("DRS sampling frequency of board %d not set!\n",i);
      return false;
    }
  return true;
}

// Open new raw data file
bool DAQReadout::OpenRawFile() {

  time_t rawtime;
  struct tm *timeinfo;
  char RunDate[MAX_COM_SIZE], Buffer[MAX_COM_SIZE];
  
  // Write run date to status structure (if after 13:00, use next day)
  time(&rawtime);
  timeinfo = gmtime(&rawtime);
  if(timeinfo->tm_hour>=13) rawtime += 12*60*60;
  timeinfo = gmtime(&rawtime);
  snprintf(RunDate,sizeof(RunDate),"%d%02d%02d",timeinfo->tm_year+1900,timeinfo->tm_mon + 1,timeinfo->tm_mday);

  // Create direcory if not existing (ignore error if already existing) and change to it
  snprintf(Buffer, sizeof(Buffer), "%s/%s", fRawDataPath, RunDate);
  if(mkdir(Buffer, S_IRWXU|S_IRWXG)==-1 && errno!=EEXIST) {
    PrintMessage("\rError: Could not create direcory \"%s\" (%s)\n", Buffer, strerror(errno));
    return false;
  }
  
  // Generate filename
  snprintf(FileName,sizeof(FileName),"%s/%s/%s_D1_%.8u.%.3u_%c_%s.raw", fRawDataPath, RunDate,
    RunDate,RunNumber,FileNumber,toupper(daq_runtype_str[daq_runtype][0]),RHeader->Description);
 
  //  Open file with rwx right for owner and group, never overwrite file
  Rawfile = open(FileName,O_WRONLY|O_CREAT|O_EXCL, S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP);
  if(Rawfile==-1) {
    PrintMessage("\rError: Could not open file \"%s\" (%s)\n", FileName, strerror(errno));
    return false;
  }
  return true;
}

// Write run header and board structures (revision number is zero for svn modified working copy)
bool DAQReadout::WriteRunHeader() {

  struct timeval Time;

  RHeader->MagicNum = MAGICNUM_OPEN;
  RHeader->DataFormat = DATA_FORMAT;
  RHeader->SoftwareRevision = atoi(REVISION) * (strchr(REVISION, 'M')==NULL ? 1:-1);

  RHeader->RunHeaderSize = sizeof(RunHeader);
  RHeader->EventHeaderSize = sizeof(EventHeader);
  RHeader->BoardStructureSize = sizeof(BoardStructure);

  RHeader->Identification = IDENTIFICATION;
  RHeader->Type = daq_runtype;
  RHeader->RunNumber  = RunNumber;
  RHeader->FileNumber = FileNumber;

  gettimeofday(&Time, NULL);
  RHeader->StartSecond = Time.tv_sec;
  RHeader->StartMicrosecond = Time.tv_usec;
  
  RHeader->NBoards   = NumBoards==0 && daq_runtype==test ? 1 : (LastBoard - FirstBoard) + 1;  
  RHeader->NChips    = kNumberOfChips;
  RHeader->NChannels = kNumberOfChannels;
  RHeader->NBytes    = sizeof(short);

  RHeader->Offset  = fFirstSample;
  RHeader->Samples = fSamples;

  if(write(Rawfile, RHeader, sizeof(RunHeader)) != sizeof(RunHeader)) {
    PrintMessage("Error: Could not write run header, terminating run (%s)\n", strerror(errno));
    return false;
  }
  
  for (int i=FirstBoard; i<=LastBoard; i++) {
    BStruct[i].SerialNo    = drs->GetBoard(i)->GetCMCSerialNumber();	  
    BStruct[i].BoardTemp   = drs->GetBoard(i)->GetTemperature();
    BStruct[i].NomFreq     = DRSFreq[i];
    BStruct[i].ScaleFactor = drs->GetBoard(i)->GetPrecision();
  }

  // In case no boards are available, dummy data is written for one board structure   
  if (NumBoards == 0) {
    BStruct[0].NomFreq     = 1;
    BStruct[0].ScaleFactor = 0.1;
  }    

  if(write(Rawfile, &BStruct[FirstBoard], sizeof(BoardStructure)*(LastBoard-FirstBoard+1+(NumBoards==0))) != (ssize_t) sizeof(BoardStructure)*(LastBoard-FirstBoard+1+(NumBoards==0))) {
    PrintMessage("Error: Could not write (all) board structures, terminating run (%s)\n", strerror(errno));
    return false;
  }
  return true;
}

// Update run header before closing file
bool DAQReadout::UpdateRunHeader(unsigned int Events, bool Error) {

  struct timeval Time;
  
  RHeader->MagicNum = Error==false ? MAGICNUM_CLOSED:MAGICNUM_ERROR;
  RHeader->Events   = Events;

  gettimeofday(&Time, NULL);
  RHeader->EndSecond = Time.tv_sec;
  RHeader->EndMicrosecond = Time.tv_usec;

  if(lseek(Rawfile,0,SEEK_SET)==-1) {
    PrintMessage("Error: Could not rewind file to write updated run header, terminating run (%s)\n", strerror(errno));
    return false;
  }

  if(write(Rawfile, RHeader, sizeof(RunHeader)) != sizeof(RunHeader)) {
    PrintMessage("Error: Could not write updated run header, terminating run (%s)\n", strerror(errno));
    return false;
  }
  return true;
}

// Write event header
bool DAQReadout::WriteEventHeader() {

  struct timeval Time;

  gettimeofday(&Time, NULL);
  
  EHeader->EventNumber = NumEvents;
  EHeader->TriggerType = (daq_runtype == data) ? 1 : 0;
  EHeader->Second = Time.tv_sec;
  EHeader->Microsecond = Time.tv_usec;
  EHeader->EventSize = sizeof(short)*RHeader->NBoards*RHeader->NChips*RHeader->NChannels*RHeader->Samples +
    	    	       sizeof(int)*RHeader->NBoards*RHeader->NChips;

  if(write(Rawfile, EHeader, sizeof(EventHeader)) != sizeof(EventHeader)) {
    PrintMessage("Error: Could not write event header, terminating run (%s)\n", strerror(errno));
    return false;
  }
  return true;
}

// Write event data (It is required that at least three chunks can be written with writev(), therefore
// IOV_MAX>=3 is checked at startup
bool DAQReadout::WriteEventData() {

  unsigned int Start, Count = 0;
  ssize_t WriteResult, Size = 0;
  struct iovec DataPart[IOV_MAX];

  // First chunk: trigger cells
  DataPart[Count].iov_base = TriggerCell + FirstBoard*kNumberOfChips;
  DataPart[Count++].iov_len = RHeader->NBoards * kNumberOfChips * sizeof(int);
  Size += DataPart[Count-1].iov_len;

  // Remaining chunks: ADC data (two chucks per channel if wrap around of pipeline occurred)
  for (int i=FirstBoard; (i<=LastBoard + (NumBoards==0)); i++) {
    for (unsigned int k=0; k<RHeader->NChips; k++) {
      Start = (TriggerCell[i][k]-fFirstSample+kNumberOfBins) % kNumberOfBins;  // Start bin for this chip
      for (unsigned int l=0; l<RHeader->NChannels; l++) {
	DataPart[Count].iov_base = &WaveForm[i][k][l][Start];
	DataPart[Count++].iov_len = (Start+fSamples<kNumberOfBins ? fSamples:(kNumberOfBins-Start)) * sizeof(short);
	Size += DataPart[Count-1].iov_len;
	// In case second part of waveform still missing, write now
	if(DataPart[Count-1].iov_len < fSamples * sizeof(short)) {
    	  DataPart[Count].iov_base = &WaveForm[i][k][l][0];
	  DataPart[Count++].iov_len = (fSamples-(kNumberOfBins-Start)) * sizeof(short);
    	  Size += DataPart[Count-1].iov_len;
	}

	// Write to disk if either maximum size of DataPart[] array or last loop interation is reached 
	// Possibly 2 chunks are entered in array in the previous lines of code, therefore IOV_MAX-1
	if (Count>=IOV_MAX-1 || (l==(RHeader->NChannels-1) && k==(RHeader->NChips-1) && i==(LastBoard+(NumBoards==0)))) {
	  if ((WriteResult=writev(Rawfile, DataPart, Count)) != (int) Size) {
            if (WriteResult==-1) PrintMessage("Error: Could not write event data, terminating run (%s)\n", strerror(errno));
            else PrintMessage("Error: Could only write %u out of %u bytes of event data, terminating run\n", WriteResult,Count*DataPart[0].iov_len);
	    return false;
	  }
	  Count = 0;   Size = 0;
	}
      } // Channels
    } // Chips
  } // Boards
  return true;
}

// Print usage text for command
void DAQReadout::PrintUsage() {
  PrintMessage("Usage: %s %s\n", CommandList[CmdNumber].Name, CommandList[CmdNumber].Parameters);
}
	 
// Print message to selected target
void DAQReadout::PrintMessage(int Target, const char *Format, ...) {
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format); 
  PrintMessage(Target, Format, ArgumentPointer);
  va_end(ArgumentPointer);
}

// Print message to log file, and screen or socket (depending on command origin)
void DAQReadout::PrintMessage(const char *Format, ...) {
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format);
  if(CmdFromSocket) PrintMessage(MsgToSocket|MsgToLog, Format, ArgumentPointer);
  else PrintMessage(MsgToConsole|MsgToLog, Format, ArgumentPointer);
  va_end(ArgumentPointer);
}

// Function doing the actual printing work
void DAQReadout::PrintMessage(int Target, const char *Format, va_list ArgumentPointer) {

  char Textbuffer[MAX_COM_SIZE];

  memset(Textbuffer, 0, sizeof(Textbuffer));  
  vsnprintf(Textbuffer, sizeof(Textbuffer), Format, ArgumentPointer);
  
  // Print to console
  if(Target & MsgToConsole) {
    if(strlen(Textbuffer)>0 && Textbuffer[strlen(Textbuffer)-1]=='\n') {
      printf("\r%s%s", Textbuffer, Prompt);   // New prompt
      fflush(stdout);
    }
    else printf("%s", Textbuffer);
  }
  // Print to log file
  if((Target & MsgToLog) && Logfile!=NULL) {
    fprintf(Logfile, "%s", Textbuffer);
    fflush(Logfile);
  }
  // Print to socket
  if((Target & MsgToSocket) && Socket!=-1) write(Socket, Textbuffer, strlen(Textbuffer));
}


// ---------------------------------------
// *****  Various utility functions  *****
// ---------------------------------------

// Check if two strings match (min 1 character must match)
bool Match(const char *str, const char *cmd) {
  return strncasecmp(str,cmd,strlen(str)==0 ? 1:strlen(str)) ? false:true;
}

// Return current available storage space in given directory
int CheckDisk(char *Directory) {
  struct statfs FileSystemStats;

  statfs(Directory, &FileSystemStats);
  return FileSystemStats.f_bavail / 1024 * (FileSystemStats.f_bsize / 1024);
}

// Parse command line for white space and double-quote separated tokens 
int ParseInput(char* Command, const char *Param[]) {
  int Count=0;
   
  while(Count<MAX_NUM_TOKEN) {
    while (isspace(*Command)) Command++; // Ignore initial white spaces
    if(*Command=='\0') break;
    if (*Command == '\"') {
      Param[Count] = ++Command;
      while(*Command!='\"' && *Command!='\0') Command++;
    }
    else {
      Param[Count] = Command;
      while(!isspace(*Command) && *Command!='\0') Command++;
    }
    if(*Command != '\0') *Command++ = '\0';
    Count++;
  }
  return Count;
}

// ReadCard function (original version by F. Goebel)
// Data is read into an array if MaxNum is larger than 1
bool ReadCard(const char *card_flag, void *store, char Type, FILE *File, unsigned int MaxNum) {
  
  char *card_name, *card_val, Buffer[MAX_COM_SIZE];
  unsigned int Count=0;
  
  rewind(File);

  while (fgets(Buffer, sizeof(Buffer), File) != NULL) {    // Read line by line
    card_name = strtok(Buffer," \t\n");
    
     // Ignore empty lines, comments, and skip if card name does not match
    if (card_name==NULL || card_name[0]=='#' || strcmp(card_name, card_flag)!=0) continue;

    // Read numbers of given type (if MaxNum>1 read array)
    while ((card_val=strtok(NULL," \t\n")) != NULL && Count++<MaxNum) {
      switch (Type) {
	case 'I': *(((int *&) store)++) = (int) strtol(card_val, NULL, 10);
	      	  break;
	case 'i': *(((short *&) store)++) = (short) strtol(card_val, NULL, 10);
	      	  break;
	case 'U': *(((unsigned int *&) store)++) = (unsigned int) strtoul(card_val, NULL, 10);
	      	  break;
	case 'u': *(((unsigned short *&) store)++) = (unsigned short) strtoul(card_val, NULL, 10);
	      	  break;
	case 'f': *(((float *&) store)++) = atof(card_val);
	      	  break;
	case 'd': *(((double *&) store)++) = atof(card_val);
	      	  break;
	case 's': sprintf((char *) store, "%s", card_val);
	      	  break;
	case 'c': *((char *) store) = card_val[0];
	      	  break;
	default:  fprintf(stderr,"Warning: Unknown type '%c' for reading of configuration file\n", Type);
	      	  return false;
      }
    }
    return true;  // Finished reading data for card name
    
  }
  fprintf(stderr,"Warning: Configuration value %s not found\n", card_flag);
  return false;
}


/********************************************************************\

  DAQ Thread

  This thread takes data until the requested number of events is reached,
  until no more disk space is available or until data taking is stopped.
  No mutex mechanism is used since variables will never be written 
  simultaneoously by two threads.
  
\********************************************************************/

void DAQ(DAQReadout *m) {

  struct timeval StartTime, StopTime;
  unsigned int EventsInFile;
  unsigned long long RunSize = 0;
  bool WriteError = false;
  off_t FileSize;

  m->NumEvents = 0;
  m->FileNumber = 0;
  m->HVFB->ClearAverages();    
  gettimeofday(&StartTime, NULL);
  m->PrintMessage("\rStarting run #%d (%s) on \"%s\" with %u event(s)\n",m->RunNumber,daq_runtype_str[m->daq_runtype],m->RHeader->Description,m->NumEventsRequested);

  do {
    // Check if enough disk space is left
    if (CheckDisk(m->fRawDataPath) <= m->fMinDiskSpaceMB+m->fMaxFileSizeMB) {	  
      m->PrintMessage("\rError: Disk space after next file (max. %d MByte) below %d MByte\n",m->fMaxFileSizeMB,m->fMinDiskSpaceMB);
      break;
    }

    // Init run header, open raw file, write run header
    if (!m->OpenRawFile()) break;
    m->PrintMessage("\rData file \"%s\" opened.\n",m->FileName);
    EventsInFile = 0;
    FileSize = 0;

    WriteError |= !m->WriteRunHeader();
    
    if (m->daq_runtype != test) m->StartDRS(); 

    // Take data until finished, stopped or file too large   
    while ((m->NumEvents<m->NumEventsRequested || m->NumEventsRequested==0) &&
      !m->Stop && FileSize/1024/1024<m->fMaxFileSizeMB && !WriteError) {

      if (m->daq_runtype == data) while (m->IsDRSBusy());  // Wait for hardware trigger (if DAQ stopped, DRS will not be busy anymore)
      else if (m->daq_runtype == pedestal) m->StopDRS();   // ..or for software trigger

      EventsInFile++;      m->NumEvents++;	
      WriteError |= !m->WriteEventHeader();

      // Read event data via VME or generate test data (for one board if no boards available)
      if (m->daq_runtype != test) {
	m->ReadCalibratedDRSData();
        m->StartDRS();  // Restart here: writing data is in parallel to waiting for next trigger
      }
      else {
	double Period = ((double) rand())/RAND_MAX*20;
	for (long unsigned int i=0; i<m->RHeader->NBoards*kNumberOfChips*kNumberOfChannels*kNumberOfBins; i++)
	  *((short *) m->WaveForm+i) = (short) (sin(i/Period)*1000);
      }

      // Write data to disk and update file size
      WriteError |= !m->WriteEventData();

      if((FileSize = lseek(m->Rawfile, 0, SEEK_CUR)) == -1) {
    	m->PrintMessage("Error: Could not determine file size, terminating run (%s)\n", strerror(errno));
	WriteError = true;
      }

      // Call feedback to process event
      m->HVFB->ProcessEvent();
    }

    // Write updated run header, close file
    RunSize += FileSize;
    WriteError |= !m->UpdateRunHeader(EventsInFile, WriteError);
    if(close(m->Rawfile)==-1) m->PrintMessage("Error: Could not close data file (%s)\n", strerror(errno));
    else m->PrintMessage("Data file closed (size %lu MByte).\n", FileSize/1024/1024);

    m->FileNumber += 1; 
  } while((m->NumEvents<m->NumEventsRequested || m->NumEventsRequested==0) && !m->Stop && !WriteError);

  // Print run summary to slow data file and to screen
  m->StopDRS();
  if(!WriteError) {
    m->PrintMessage("\rRun #%d (%s) %s, %d events\n",m->RunNumber,daq_runtype_str[m->daq_runtype],(m->NumEvents == m->NumEventsRequested) ? "completed":"stopped",m->NumEvents);
    m->SlowDataClass->NewEntry("Runinfo");
    m->SlowDataClass->AddToEntry("%d %s %s %d %d", m->RunNumber, daq_runtype_str[m->daq_runtype], m->RHeader->Description, m->NumEvents, m->FileNumber);
    if(m->SlowDataClass->ErrorCode != 0) {
      m->PrintMessage("Error, could not write DAQ data to file (%s), file closed\n", strerror(m->SlowDataClass->ErrorCode));
    }

  }
  else m->PrintMessage("\rRun #%d (%s) aborted due to error after %d events\n",m->RunNumber,daq_runtype_str[m->daq_runtype],m->NumEvents);

  // Print run statistics
  if (m->NumEvents>0 && !WriteError) {
    gettimeofday(&StopTime, NULL);
    float RunTime = StopTime.tv_sec-StartTime.tv_sec + (StopTime.tv_usec-StartTime.tv_usec)*1e-6;
    m->PrintMessage("Time for run %.2f seconds, trigger rate %.2f Hz.\n", RunTime, m->NumEvents/RunTime);
    m->PrintMessage("Run size %llu MByte, data rate %.1f MByte/s.\n", RunSize/1024/1024, RunSize/1024.0/1024/RunTime);
  }
  
  m->daq_state = stopped;
}
