#ifndef MARS_MGeomPix
#define MARS_MGeomPix

#ifndef MARS_MParContainer
#include "MParContainer.h"
#endif

class MGeomCam;

class MGeomPix : public MParContainer
{ 
public:
    static const Float_t gsTan60; // tan(60/kRad2Deg);
    static const Float_t gsTan30; // tan(30/kRad2Deg);

    enum {
        kRightTop,
        kRight,
        kRightBottom,
        kLeftBottom,
        kLeft,
        kLeftTop
    };

private:
    enum {
        kIsInOutermostRing = BIT(22),
        kIsInOuterRing     = BIT(23),
        kUserBits          = 0x1fc000 // 14-21 are allowed
    };

    Float_t fX;            // [mm]   the x coordinate of the center
    Float_t fY;            // [mm]   the y coordinate of the center
    Float_t fD;            // [mm]   the d coordinate of the pixel (dist between two parallel sides)
    Float_t fA;            // [mm^2] Area of the pixel

    Byte_t  fNumNeighbors; // number of valid neighbors
    Short_t fNeighbors[6]; // the IDs of the pixel next to it (we are assuming an hexagonal geometry)

    UInt_t fSector;        // Number of sector the pixels corresponds to
    UInt_t fAidx;          // Area index of the pixel

public:
    MGeomPix(Float_t x=0, Float_t y=0, Float_t d=1, UInt_t s=0, UInt_t aidx=0);

    void Copy(TObject &obj) const
    {
        MGeomPix &pix = (MGeomPix&)obj;
        pix.fX = fX;
        pix.fY = fY;
        pix.fD = fD;
        pix.fA = fA;
        pix.fNumNeighbors = fNumNeighbors;
        pix.fSector = fSector;
        pix.fAidx = fAidx;
        for (int i=0; i<6; i++)
            pix.fNeighbors[i] = fNeighbors[i];
    }

    void Print(Option_t *opt=NULL) const;

    void Set(Float_t x, Float_t y, Float_t d=1, UInt_t s=0, UInt_t aidx=0) { fX=x; fY=y; fD=d; fA=d*d*gsTan60/2; fSector=s; fAidx=aidx; }

    void SetNeighbors(Short_t i0=-1, Short_t i1=-1, Short_t i2=-1,
                      Short_t i3=-1, Short_t i4=-1, Short_t i5=-1);

    void CheckOuterRing(const MGeomCam &cam);

    Float_t GetX() const  { return fX; }
    Float_t GetY() const  { return fY; }
    Float_t GetD() const  { return fD; }
    Float_t GetL() const  { return fD*gsTan30; } // Length of one of the parallel sides
    UInt_t  GetSector() const { return fSector; }

    Float_t GetDist() const;
    Float_t GetDist(const MGeomPix &pix) const;
    Float_t GetAngle(const MGeomPix &pix) const;

    Float_t GetA() const    { return fA; /*fD*fD*gsTan60/2;*/ }
    Int_t   GetAidx() const { return fAidx; }

    Byte_t  GetNumNeighbors() const { return fNumNeighbors; }
    Short_t GetNeighbor(Byte_t i) const { return fNeighbors[i]; }

    Bool_t IsInOutermostRing() const { return TestBit(kIsInOutermostRing); }
    Bool_t IsInOuterRing() const     { return TestBit(kIsInOuterRing); }

    Bool_t IsInside(Float_t px, Float_t py) const;
    Int_t  GetDirection(const MGeomPix &pix) const;

    /*
     //
     // These function are for future usage. They are not virtual in
     // root by now.
     //
     void SetBit(UInt_t f, Bool_t set) { set ? TObject::SetBit(f) : TObject::ResetBit(f); }
     void SetBit(UInt_t f)    { TObject::SetBit(f & kUserBits); }
     void ResetBit(UInt_t f)  { TObject::ResetBit(f & kUserBits); }
     void InvertBit(UInt_t f) { TObject InvertBit(f & kUserBits); }
     */

    ClassDef(MGeomPix, 3) // Geometry class describing the geometry of one pixel
};

#endif
