/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz 11/2005 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2005-2007
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MJOptimizeBase
//
// Base class for classes training a random forest
//
// The order when reading a file is:
//   1) Execution of PreTasks  (set by user)
//   2) Execution of PreCuts   (set by user)
//   3) Selector               (calculated from number of requested events)
//   4) Splitter               (if sample is split automatically in test/train)
//   5) PostTasks              (set by user)
//
// The split into Pre- and PostTasks is done for speed reason. So, if
// you calculate a vlue which is not needed for your PreCuts, you can
// calculate it afterwards, which will speed up execution.
//
/////////////////////////////////////////////////////////////////////////////
#include "MJOptimizeBase.h"

#include <TFile.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MFDataPhrase.h"
#include "MParameterCalc.h"

#include "MStatusDisplay.h"

ClassImp(MJOptimizeBase);

using namespace std;

//------------------------------------------------------------------------
//
// Add a cut which is used to fill the matrix, eg "MMcEvt.fPartId<1.5"
// (The rule is applied, not inverted: The matrix is filled with
// the events fullfilling the condition)
//
void MJOptimizeBase::AddCut(TList &l, const char *rule)
{
    MFilter *f = new MFDataPhrase(rule);
    f->SetBit(kCanDelete); //FIXME!!!! Why does not any other list delete it???
    Add(l, f);
}

//------------------------------------------------------------------------
//
// Add an additional parameter (MParameterCalc), eg "0.5", "MWeight"
// The default container name is "MWeight"
//
void MJOptimizeBase::AddPar(TList &l, const char *rule, const char *pname)
{
    TString tname(pname);
    tname += "Calc";

    MParameterCalc *par = new MParameterCalc(rule, tname);
    par->SetNameParameter(pname);
//    par->SetBit(kCanDelete);  //FIXME!!!! MTaskList is deleting it
    Add(l, par);
}

//------------------------------------------------------------------------
//
// Add a task/cut which is used to fill the matrix. If kCanDelete is set
// MJOptimize takes the ownership.
//
void MJOptimizeBase::Add(TList &l, MTask *f)
{
    l.Add(f);
}

//------------------------------------------------------------------------
//
// Add a parameter used in your filters (see AddFilter) The parameter
// index is returned,
//
//   Int_t idx = AddParameter("log10(MHillas.fSize)");
//
// The indices are starting with 0 always.
//
Int_t MJOptimizeBase::AddParameter(const char *rule)
{
    fRules.Add(new TNamed(rule, ""));
    return fRules.GetSize()-1;
}
