//=//////////////////////////////////////////////////////////////////////
//=
//= creadparam            
//=
//= @file        creadparam.cxx
//= @desc        Reading of parameters file
//= @author      J C Gonzalez
//= @email       gonzalez@mppmu.mpg.de
//= @date        Thu May  7 16:24:22 1998
//=
//=----------------------------------------------------------------------
//=
//= Created: Thu May  7 16:24:22 1998
//= Author:  Jose Carlos Gonzalez
//= Purpose: Program for reflector simulation
//= Notes:   See files README for details
//=    
//=----------------------------------------------------------------------
//=
//= $RCSfile: creadparam.cxx,v $
//= $Revision: 1.11 $
//= $Author: magicsol $ 
//= $Date: 2001-07-25 11:38:00 $
//=
//=//////////////////////////////////////////////////////////////////////

// @T \newpage

//!@section Source code of |creadparam.cxx|.

/*!@"
  
  This section describes briefly the source code for the file
  |creadparam.cxx|. This file is very closely related to the file
  |readparams.cxx| from the |reflector| program. Actually, this later
  file was the ancestror of the file you are looking at.

  All the defines it uses are located in the file |creadparam.h|. In
  the first one we can see the definitions of the commands available
  for the parameters file. We describe these commands in a later
  section.
  
  @"*/

//!@subsection Includes and Global variables definition.

/*!@"
  
  All the defines are located in the file {\tt creadparam.h}.

  @"*/
 
//!@{

#include "creadparam.h"

//!@}

//!@subsection Definition of global variables.

/*!@"

  Here we define the global variables where the values from the
  parameters file are stored.
  
  @"*/

//!@{

static char Input_filename[PATH_MAX_LENGTH];  //@< input filename
static char Starfield_filename[PATH_MAX_LENGTH]; //@< starfield input filename
static char Data_filename[PATH_MAX_LENGTH];   //@< data filename
static char ROOT_filename[PATH_MAX_LENGTH];   //@< data filename
static char Loop_filename[PATH_MAX_LENGTH];   //@< special data filename
static char CT_filename[PATH_MAX_LENGTH];     //@< name of the CT def. file
static char NSB_directory[PATH_MAX_LENGTH];  //@< database for NSB
static int simulateNSB = TRUE;                //@< Will we simulate NSB?
static int anaPixels = -1;      //@< number of pixels for the analysis
static float meanNSB;           //@< NSB mean value (per pixel)
static int nphe2NSB=0;            //@< Number of phe from shower to do NSB simulation 
static float qThreshold[TRIGGER_PIXELS];     //@< Threshold values
static int Individual_Thres = FALSE;
static float qTailCut;          //@< Tail Cut value
static int nIslandsCut;         //@< Islands Cut value
static int countIslands = TRUE; //@< Will we count the islands?
static long int Seeds[2]; 
static int *Skip;
static int nSkip=0;
static int Data_From_STDIN = FALSE;
static int Read_Phe = FALSE;
static int Read_Phe_All = FALSE;
static int Write_All_Images = FALSE;
static int Write_McEvt  = TRUE;
static int Write_McTrig = FALSE;
static int Write_McFadc = FALSE;
static int Write_RawEvt = FALSE;
static int Write_All_Data = FALSE;
static int Select_Energy = TRUE;
static float Select_Energy_le = 0.0;           //@< GeV
static float Select_Energy_ue = 100000.0;      //@< GeV
static float Trigger_Radius;
static int Set_Trigger_Radius=FALSE;
static float fCorrection;
static int Apply_Correction=FALSE;
static int Trigger_Scan = FALSE;
static int FADC_Scan = FALSE;
static int Trigger_Loop = FALSE;
static float Trigger_gate_length = 3.0;
static float Trigger_over_time = 0.25;
static float Trigger_response_ampl = 1.0;
static float Trigger_response_fwhm = 2.0;
static float Trigger_threshold = 7.0;
static int Trigger_multiplicity = 4;
static int Trigger_topology = 2;
static int Trigger_loop_lthres = 0;
static int Trigger_loop_uthres = 10;
static int Trigger_loop_lmult = 2;
static int Trigger_loop_umult = 10;
static int Trigger_loop_ltop = 0;
static int Trigger_loop_utop = 2;
static float FADC_response_ampl = MFADC_RESPONSE_AMPLITUDE;
static float FADC_response_fwhm = MFADC_RESPONSE_FWHM;
//!@}

//!@subsection The function |readparam()|.

//!-----------------------------------------------------------
// @name  creadparam
//                                                
// @desc  read parameters from the stdin / parameters file
//
// @var   *filename  Name of the parameters file (NULL->STDIN)
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function

//!@{ 
void 
readparam(char * filename)
{
  char sign[] = GLUE_postp( PROGRAM, VERSION ); //@< initialize sign
  char line[LINE_MAX_LENGTH];    //@< line to get from the stdin
  char token[ITEM_MAX_LENGTH];   //@< a single token
  int i, j, k;                      //@< dummy counters
  float aux;                    //@< auxiliar variable
  ifstream ifile;

  // use cin or ifile (reading from STDIN or from parameters file?
  if ( filename != NULL )
    ifile.open( filename );

  // get signature
  if ( filename != NULL )
    ifile.getline(line, LINE_MAX_LENGTH);
  else
    cin.getline(line, LINE_MAX_LENGTH);
  line[strlen(SIGNATURE)] = '\0';
  strcpy(line, sign);
  if (strcmp(sign, SIGNATURE) != 0) {
    cerr << "ERROR: Signature of parameters file is not correct\n";
    cerr << '"' << sign << '"' << '\n';
    cerr << "should be: " << SIGNATURE << '\n';
    exit(1);
  }

  // loop till the "end" directive is reached
  int is_end = FALSE;
  while (! is_end) {          

    // get line from file or stdin
    if ( filename != NULL )
      ifile.getline(line, LINE_MAX_LENGTH);
    else
      cin.getline(line, LINE_MAX_LENGTH);

    // skip comments (start with '#')
    if (line[0] == '#')
      continue;

    // show user comments (start with '>')
    if (line[0] == '>') {
      cout << line << endl << flush;
      continue;
    }

    // look for each item at the beginning of the line
    for (i=0; i<=end_file; i++) 
      if (strstr(line, ITEM_NAMES[i]) == line)
        break;
        
    // if it is not a valid line, just ignore it
    if (i == end_file+1) {
      cerr << "Skipping unknown token in [" << line << "]\n";
      continue;
    }

    // case block for each directive
    switch ( i ) {

    case input_file:          //@< name of the input file
          
      // get the name of the input_file from the line
      sscanf(line, "%s %s", token, Input_filename);

      break;

    case starfield_file:         //@< name of the output file
          
      // get the name of the output_file from the line
      sscanf(line, "%s %s", token, Starfield_filename);

      break;

    case data_file:           //@< name of the data file
          
      // get the name of the data_file from the line
      sscanf(line, "%s %s", token, Data_filename);

      break;

    case root_file:          //@< name of the ROOT file
          
      // get the name of the data_file from the line
      sscanf(line, "%s %s", token, ROOT_filename);
      cout << '[' << ROOT_filename << ']' << endl << flush;

      break;

    case ct_file:             //@< name of the telescope file
          
      // get the name of the ct_file from the line
      sscanf(line, "%s %s", token, CT_filename);

      break;

    case nsb_on:              //@< simulate NSB?
          
      // we will simulate NSB
      simulateNSB = TRUE;
          
      break;

    case nsb_off:             //@< simulate NSB?
          
      // we will NOT simulate NSB
      simulateNSB = FALSE;
          
      break;

    case nsb_mean:            //@< value of <NSB> per pixel
          
      // get value of <NSB> (in photons)
      sscanf(line, "%s %f %d", token, &meanNSB, &nphe2NSB);

      break;

    case nsb_directory:         //@< name of the output file
          
      // get the name of the output_file from the line
      sscanf(line, "%s %s", token, NSB_directory);

      break;

    case ana_pixels:          //@< number of pixels for analysis
          
      // number of pixels for analysis
      sscanf(line, "%s %d", token, &anaPixels);

      break;

    case pixel_thres:           //@< value of threshold for trigger (q0)
          
      // get value of threshold (in ph.e.)
      sscanf(line, "%s %i %f", token,&k, &aux);
      qThreshold[k]=aux;
      Individual_Thres = TRUE;

      break;

    case tail_cut:            //@< value of tail_cut (t0)
          
      // get value of tail_cut (in ph.e.)
      sscanf(line, "%s %f", token, &qTailCut);

      break;

    case islands_on:          //@< DO count islands
          
      // DO count islands
      countIslands = TRUE;
          
      break;

    case islands_off:         //@< do NOT count islands
          
      // do NOT count islands
      countIslands = FALSE;
          
      break;

    case islands_cut:         //@< value of islands_cut (i0)
          
      // get value of islands_cut (in ph.e.)
      sscanf(line, "%s %d", token, &nIslandsCut);
      countIslands = TRUE;

      break;

    case seeds:               //@< values of seeds for random numbers
          
      // get seeds
      sscanf(line, "%s %ld %ld", token, &Seeds[0], &Seeds[1]);

      break;

    case skip:                //@< skip pathological showers
          
      // get showers to skip
      cin >> nSkip;
      Skip = new int[nSkip];
      for (j=0; j<nSkip; ++j) {
        cin >> Skip[j];
        cout << Skip[j] << endl << flush;
      }

      break;

    case data_from_stdin:     //@< to read data from stdin
          
      // change boolean value
      Data_From_STDIN = TRUE;

      break;

    case read_phe:            //@< to read PHE files
          
      // change boolean value
      Read_Phe = TRUE;

      break;

    case read_phe_all:        //@< to read PHE files from write_all_images
          
      // change boolean value
      Read_Phe_All = TRUE;

      break;

    case write_all_images:    //@< to write ALL the images
          
      // change boolean value
      Write_All_Images = TRUE;

      break;

    case nowrite_McEvt:    //@< do not write the McEvt info
          
      // change boolean value
      Write_McEvt = FALSE;

      break;

    case write_McTrig:    //@< to write the McTrig info
          
      // change boolean value
      Write_McTrig = TRUE;

      break;

    case write_McFadc:    //@< to write the McFadc info
          
      // change boolean value
      Write_McFadc = TRUE;

      break;

    case write_all_data:      //@< to write single pixel data
          
      // change boolean value
      Write_All_Data = TRUE;

      break;

    case select_energy:       //@< value of islands_cut (i0)
          
      // get energy range
      sscanf(line, "%s %f %f", token, &Select_Energy_le, &Select_Energy_ue);
      Select_Energy = TRUE;

      break;

    case trigger_radius:      //@< set radius of trigger area 
          
      // get trigger radius
      sscanf(line, "%s %f", token, &Trigger_Radius);
      Set_Trigger_Radius = TRUE;

      break;

    case correction:          //@< apply a kind of correction to pixel values
          
      // get trigger radius
      sscanf(line, "%s %f", token, &fCorrection);
      Apply_Correction = TRUE;

      break;

    case fadc_scan:

      // change boolean value
      FADC_Scan = TRUE;

      break;

    case fadc_prop:

      //  Get parameters for the fadc  response for one phe
      sscanf(line, "%s %f %f", token, &FADC_response_ampl,&FADC_response_fwhm);

      break;

    case trigger_scan:

      // change boolean value
      Trigger_Scan = TRUE;

      break;
    case trigger_prop:

      //  Get parameters for the diskriminator and the electronic
      //  response for one phe
      sscanf(line, "%s %f %f %f %f", token, &Trigger_gate_length,&Trigger_over_time, &Trigger_response_ampl,&Trigger_response_fwhm);

      break;

    case trigger_loop:

      //  Get loop's limits
      sscanf(line, "%s %d %d %d %d %d %d", token, &Trigger_loop_lthres, &Trigger_loop_uthres, &Trigger_loop_lmult, &Trigger_loop_umult, &Trigger_loop_ltop, &Trigger_loop_utop );

      // Set qThreshold to a global value
      for(k=0;k<TRIGGER_PIXELS;k++){
	qThreshold[k]=Trigger_loop_lthres;
      }

      // change boolean value
      Trigger_Loop = TRUE;
      Write_RawEvt = FALSE;
      Individual_Thres = FALSE;

      break;

    case trigger_single:

      //  Get trigger conditions
      sscanf(line, "%s %f %d %d", token, &Trigger_threshold, &Trigger_multiplicity, &Trigger_topology);

      // Set qThreshold to a global value
      for(k=0;k<TRIGGER_PIXELS;k++){
	qThreshold[k]=Trigger_threshold;
      }

      // change boolean value
      Trigger_Loop = FALSE;
      Write_RawEvt = TRUE;
      Individual_Thres = FALSE;

      break;

    case Trigger_Loop_Output_Only_Specialists:

      // get the name of the data_file from the line
      sscanf(line, "%s %s", token, Loop_filename);
      cout << '[' << Loop_filename << ']' << endl << flush;
          
      // change boolean value
      Write_RawEvt = TRUE;
      
      break;

    case end_file:            //@< end of the parameters file

      // show a short message
      is_end = TRUE;

      break;

    } // switch ( i ) 

  } // while (! is_end)

  // after the loop is finished, return to the main function
  return;
}
//!@}


//!-----------------------------------------------------------
// @name get_input_filename
//                                                
// @desc get name of the input file
//
// @return   Name of the Input file
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
char *
get_input_filename(void)
{
  return (Input_filename);
}
//!@}


//!-----------------------------------------------------------
// @name get_starfield_filename
//                                                
// @desc get name of the starfield input file
//
// @return   Name of the starfield file
//
// @date Tue Feb 15 16:02:18 CET 2000
//------------------------------------------------------------
// @function 

//!@{
char *
get_starfield_filename(void)
{
  return (Starfield_filename);
}
//!@}


//!-----------------------------------------------------------
// @name get_data_filename
//                                                
// @desc get name of the data file
//
// @return   Name of the Data file
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
char *
get_data_filename(void)
{
  return (Data_filename);
}
//!@}


//!-----------------------------------------------------------
// @name get_root_filename
//                                                
// @desc get name of the ROOT file
//
// @return   Name of the ROOT file
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
char *
get_root_filename(void)
{
  return (ROOT_filename);
}
//!@}
//!-----------------------------------------------------------
// @name get_loop_filename
//                                                
// @desc get name of the special ROOT file
//
// @return   Name of the special ROOT file
//
// @date Fri Jun 23 17:34:19 CEST 2000
//------------------------------------------------------------
// @function 

//!@{

char *
get_loop_filename(void)
{
  return (Loop_filename);
}

//!-----------------------------------------------------------
// @name get_ct_filename
//                                                
// @desc get name of CT definition file
//
// @return   Name of the CT definition file
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
char *
get_ct_filename(void)
{
  return (CT_filename);
}
//!@}

//!-----------------------------------------------------------
// @name get_nsb_directory
//                                                
// @desc get name of the directory where the database for NSB is
//
// @return   Name of the NSB directory
//
// @date Tue Jan 30 12:07:56 MET 2001
//------------------------------------------------------------
// @function 

//!@{
char *
get_nsb_directory(void)
{
  return (NSB_directory);
}
//!@}



//!-----------------------------------------------------------
// @name get_nsb
//                                                
// @desc are we going to simulate NSB ?
//
// @var  *n  Mean value for the NSB (ph.e./pixel)
// @return   TRUE: we'll simulate NSB; FALSE: we won't
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_nsb(float *n, int *m)
{
  *n = meanNSB;
  *m = nphe2NSB;
  return ( simulateNSB );
}
//!@}


//!-----------------------------------------------------------
// @name get_threshold   
//                                                
// @desc get threshold value
//
// @return   Value of the threshold q$_0$
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
void 
get_threshold(float *t)
{
  for(int i=0;i<TRIGGER_PIXELS;i++)
    t[i]=qThreshold[i];
}
//!@}


//!-----------------------------------------------------------
// @name get_indi_thres_pixel
//                                                
// @desc get boolean information about global (FALSE) or 
// @desc individual (TRUE) pixel trigger threshold
//
// @return   Value for the Individual_Thres
//
// @date Wed Jul 18 16:29:43 CEST 2001
//------------------------------------------------------------
// @function 

//!@{
int get_indi_thres_pixel(void){
  return  Individual_Thres;
}
//!@}


//!-----------------------------------------------------------
// @name get_tail_cut   
//                                                
// @desc get tail cut value
//
// @return   Value for the Tail-cut
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
float 
get_tail_cut(void)
{
  return( qTailCut );
}
//!@}


//!-----------------------------------------------------------
// @name get_islands_cut
//                                                
// @desc are we going to count the islands ?
//
// @var  *n  Cut on islands number
// @return   TRUE: we'll count the islands; FALSE: we won't
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_islands_cut(int *n)
{
  *n = nIslandsCut;
  return ( countIslands );
}
//!@}


//!-----------------------------------------------------------
// @name get_ana_pixels
//                                                
// @desc number of pixels for the analysis
//
// @return  Number of pixels to use in the image parameters
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_ana_pixels(void)
{
  return ( anaPixels );
}
//!@}


//!-----------------------------------------------------------
// @name get_seeds
//                                                
// @desc are we going to count the islands ?
//
// @var  *n  Number of the seed
// @return   N-th random-number Seed
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
long int
get_seeds(int n)
{
  return ( Seeds[n] );
}
//!@}


//!-----------------------------------------------------------
// @name get_skip_showers
//                                                
// @desc get list of showers to skip
//
// @var *s1  Pointer to a vector of number of showers
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
void 
get_skip_showers( int *s )
{
  int i;
  for (i=0; i<nSkip; ++i)
    s[i] = Skip[i];
}
//!@}


//!-----------------------------------------------------------
// @name get_nskip_showers
//                                                
// @desc get number of showers to skip
//
// @return  Number of showers to be skipped
//
// @date Mon Sep 14 13:27:56 MET DST 1998
//------------------------------------------------------------
// @function 

//!@{
int 
get_nskip_showers( void )
{
  return( nSkip );
}
//!@}


//!-----------------------------------------------------------
// @name get_data_from_stdin
//                                                
// @desc get whether we will read the data from the STDIN
//
// @return  TRUE: we'll read data from STDIN; FALSE: we won't
//
// @date Wed Nov 25 13:21:00 MET 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_data_from_stdin(void)
{
  return ( Data_From_STDIN );
}
//!@}


//!-----------------------------------------------------------
// @name get_read_phe
//                                                
// @desc get whether we will read PHE files
//
// @return  TRUE: we'll read PHE files; FALSE: we won't
//
// @date Wed Nov 25 13:21:00 MET 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_read_phe(void)
{
  return ( Read_Phe );
}
//!@}


//!-----------------------------------------------------------
// @name get_read_phe_all
//                                                
// @desc get whether we will read PHE files, with write_all_images
//
// @return  TRUE: we'll do it; FALSE: we won't
//
// @date Wed Nov 25 13:21:00 MET 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_read_phe_all(void)
{
  return ( Read_Phe_All );
}
//!@}


//!-----------------------------------------------------------
// @name write_all_images
//                                                
// @desc write all images to .phe, even those without trigger
//
// @return  TRUE: we'll write everything; FALSE: we won't
//
// @date Wed Nov 25 13:21:00 MET 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_write_all_images(void)
{
  return ( Write_All_Images );
}
//!@}


//!-----------------------------------------------------------
// @name write_all_data
//                                                
// @desc write single pixel data to file .dat
//
// @return  TRUE: we'll write everything; FALSE: we won't
//
// @date Wed Nov 25 13:21:00 MET 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_write_all_data(void)
{
  return ( Write_All_Data );
}
//!@}

//!-----------------------------------------------------------
// @name write_McEvt
//                                                
// @desc write the McEvt class for each event to the .root file 
//
// @return  TRUE: we'll write it;  FALSE: we won't
//
//------------------------------------------------------------
// @function 

//!@{
int
get_write_McEvt(void)
{
  return ( Write_McEvt );
}
//!@}

//!-----------------------------------------------------------
// @name write_McTrig
//                                                
// @desc write the McTrig class for each event to the .root file 
//
// @return  TRUE: we'll write it;  FALSE: we won't
//
//------------------------------------------------------------
// @function 
 
//!@{
int
get_write_McTrig(void)
{
  return ( Write_McTrig );
}
//!@}

//!-----------------------------------------------------------
// @name write_McFadc
//                                                
// @desc write the McFadc class for each event to the .root file 
//
// @return  TRUE: we'll write it;  FALSE: we won't
//
//------------------------------------------------------------
// @function 
 
//!@{
int
get_write_McFadc(void)
{
  return ( Write_McFadc );
}
//!@}

//!-----------------------------------------------------------
// @name write_RawEvt
//                                                
// @desc write the RawEvt class for each event to the .root file 
//
// @return  TRUE: we'll write it;  FALSE: we won't
//
//------------------------------------------------------------
// @function 

//!@{
int
get_write_RawEvt(void)
{
  return ( Write_RawEvt );
}
//!@}

//!-----------------------------------------------------------
// @name get_select_energy                                      
//
// @desc return energy range allowed for showers from .phe file
//
// @var *le  Lower limit in the allowed energy range
// @var *ue  Lower limit in the allowed energy range
// @return  TRUE: we make selection on the energy; FALSE: we don't
//
// @date Wed Nov 25 13:21:00 MET 1998
//------------------------------------------------------------
// @function 

//!@{
int
get_select_energy(float *le, float *ue)
{
  *le = Select_Energy_le;
  *ue = Select_Energy_ue;
  return ( Select_Energy );
}
//!@}


//!-----------------------------------------------------------
// @name get_trigger_radius                                  
//
// @desc return the radius of the trigger area in the camera (if set)
//
// @var *radius  Radius of the trigger area in the camera
// @return  TRUE: we choose a given radius for the trigger area
//
// @date Fri May  7 11:07:43 MET DST 1999
//------------------------------------------------------------
// @function 

//!@{
int get_trigger_radius(float *radius)
{
  *radius = Trigger_Radius;
  return ( Set_Trigger_Radius );
}
//!@}


//!-----------------------------------------------------------
// @name get_trigger_radius                                  
//
// @desc return the radius of the trigger area in the camera (if set)
//
// @var *radius  Radius of the trigger area in the camera
// @return  TRUE: we choose a given radius for the trigger area
//
// @date Fri May  7 11:07:43 MET DST 1999
//------------------------------------------------------------
// @function 

//!@{
int get_correction(float *corr)
{
  *corr = fCorrection;
  return ( Apply_Correction );
}
//!@}

//!-----------------------------------------------------------
// @name FADC_Scan
//                                                
// @desc show the FADC signal for each event in the screen 
//
// @return  TRUE: we'll show it;  FALSE: we won't
//
//------------------------------------------------------------
// @function 

//!@{
int
get_FADC_Scan(void)
{
  return ( FADC_Scan );
}
//!@}

//!-----------------------------------------------------------
// @name Trigger_Scan
//                                                
// @desc show the Trigger signal for each event in the screen 
//
// @return  TRUE: we'll show it;  FALSE: we won't
//
//------------------------------------------------------------
// @function 

//!@{
int
get_Trigger_Scan(void)
{
  return ( Trigger_Scan );
}
//!@}

//!-----------------------------------------------------------
// @name Fadc_Propeties
//                                                
// @desc fix properties of the FADC response 
//
//------------------------------------------------------------
// @function 

//!@{
void
get_FADC_properties(float *ra, float *rf)
{
  *ra=FADC_response_ampl;
  *rf=FADC_response_fwhm;

}
//!@}

//!-----------------------------------------------------------
// @name Trigger_Propeties
//                                                
// @desc fix properties of the diskriminator and amplifier for Trigger 
//
//------------------------------------------------------------
// @function 

//!@{
void
get_Trigger_properties(float *gl, float *ot, float *ra, float *rf)
{
  *gl=Trigger_gate_length;
  *ot=Trigger_over_time;
  *ra=Trigger_response_ampl;
  *rf=Trigger_response_fwhm;

}
//!@}

//!-----------------------------------------------------------
// @name Trigger_Loop
//                                                
// @desc make a loop over Trigger conditions 
//
// @return  TRUE: we'll make it;  FALSE: we won't
//
//------------------------------------------------------------
// @function 

//!@{
int
get_Trigger_Loop(int *lt, int *ut, int *lm, int *um, int *lg, int *ug)
{
  *lt=Trigger_loop_lthres;
  *ut=Trigger_loop_uthres;
  *lm=Trigger_loop_lmult;
  *um=Trigger_loop_umult;
  *lg=Trigger_loop_ltop;
  *ug=Trigger_loop_utop;
  return ( Trigger_Loop );
}
//!@}

//!-----------------------------------------------------------
// @name Trigger_Single
//                                                
// @desc fix Trigger conditions 
//
//------------------------------------------------------------
// @function 

//!@{
void
get_Trigger_Single(float *t, int *m, int *g)
{
  for(int i=0;i<TRIGGER_PIXELS;i++)
    t[i]=qThreshold[i];
  *m=Trigger_multiplicity;
  *g=Trigger_topology;

}
//!@}

//=------------------------------------------------------------
//!@subsection Log of this file.

//!@{
//
// $Log: not supported by cvs2svn $
// Revision 1.10  2001/07/19 09:28:30  blanch
// A new command, which allows to set individual trigger threshod for each pixel,
// has been added.
//
// Revision 1.9  2001/03/05 10:43:18  blanch
// New input commands have been added:
//
// 	- write_McFadc: to write the FADC simulation information.
// 	- fadc_prop: gives the shape of the single phe response of the FADC
//
// And new value has been added in the mean_NSB command that tells how many phes
// have to come from the shower to do the NSB simulation in that shower (it speeds
// up the simulation).
//
// Revision 1.8  2001/02/23 10:55:43  magicsol
// An input commmand that talls the path for the NSB database has been added.
//
// Revision 1.7  2001/01/15 12:37:48  magicsol
// It has been introduced the option to read from the input card the overlaping
// time.
//
// Revision 1.6  2000/09/21 11:47:33  harald
// Oscar found some smaller errors in the calculation of the pixel shape and
// corrected it.
//
// Revision 1.5  2000/07/04 14:13:02  MagicSol
// It reads from the general input card the parameters for the
// trigger analysi.
//
// Revision 1.4  2000/05/11 14:22:33  blanch
// New input card option has been introduced:
// 	trigger_loop lt ut lm um lg ug
// It forces the camera program to study several condition trigger implementations. Integers after key word fix limits of loop over thershold, multiplicity and topology.
//
// Revision 1.3  2000/03/24 18:14:05  blanch
// Parameters that tell as if we are going to see the diskriminator and/or FADC signal have been included.
//
// Revision 1.2  2000/02/18 17:45:43  petry
// This version belongs to camera.cxx 1.5.
// It has been put in the repository in order to be
// able to share the further development with others.
//
// If you need something working, wait or take an earlier one.
// See file README
//
// Revision 1.1.1.1  1999/11/05 11:59:34  harald
// This the starting point for CVS controlled further developments of the
// camera program. The program was originally written by Jose Carlos. 
// But here you can find a "rootified" version to the program. This means 
// that there is no hbook stuff in it now. Also the output of the
// program changed to the MagicRawDataFormat. 
//
// The "rootification" was done by Dirk Petry and Harald Kornmayer. 
//
// In the following you can see the README file of that version:
//
// ==================================================
//
// Fri Oct 22  1999   D.P.
//
// The MAGIC Monte Carlo System
//
// Camera Simulation Programme
// ---------------------------
//
// 1) Description
//
// This version is the result of the fusion of H.K.'s
// root_camera which is described below (section 2)
// and another version by D.P. which had a few additional
// useful features.
//
// The version compiles under Linux with ROOT 2.22 installed
// (variable ROOTSYS has to be set).
//
// Compile as before simply using "make" in the root_camera
// directory.
//
// All features of H.K.'s root_camera were retained.
//
// Additional features of this version are:
//
//   a) HBOOK is no longer used and all references are removed.
//
//   b) Instead of HBOOK, the user is given now the possibility of 
//      having Diagnostic data in ROOT format as a complement
//      to the ROOT Raw data.
//
//      This data is written to the file which is determined by
//      the new input parameter "diag_file" in the camera parameter
//      file.
//
//      All source code file belonging to this part have filenames
//      starting with "MDiag".
//
//      The user can read the output file using the following commands
//      in an interactive ROOT session:
//
//        	root [0] .L MDiag.so
// 	root [1] new TFile("diag.root");
// 	root [2] new TTreeViewer("T");
// 	
//      This brings up a viewer from which all variables of the
//      TTree can be accessed and histogrammed. This example
//      assumes that you have named the file "diag.root", that
//      you are using ROOT version 2.22 or later and that you have
//      the shared object library "MDiag.so" which is produced
//      by the Makefile along with the executable "camera".
//        
//  !   The contents of the so-called diag file is not yet fixed.
//  !   At the moment it is what J.C.G. used to put into the HBOOK
//  !   ntuple. In future versions the moments calculation can be
//  !   removed and the parameter list be modified correspondingly.
//
//   c) Now concatenated reflector files can be read. This is useful
//      if you have run the reflector with different parameters but
//      you want to continue the analysis with all reflector data
//      going into ONE ROOT outputfile.
//
//      The previous camera version contained a bug which made reading 
//      of two or more concatenated reflector files impossible.
//
//   d) The reflector output format was changed. It is now version
//      0.4 .
//      The change solely consists in a shortening of the flag
//      definition in the file 
//
//            include-MC/MCCphoton.hxx  
//
//  !   IF YOU WANT TO READ REFLECTOR FORMAT 0.3, you can easily
//  !   do so by recompiling camera with the previous version of
//  !   include-MC/MCCphoton.hxx.
//
//      The change was necessary for saving space and better
//      debugging. From now on, this format can be frozen.
//
//  !   For producing reflector output in the new format, you
//  !   of course have to recompile your reflector with the
//  !   new include-MC/MCCphoton.hxx .
//
//   e) A first version of the pixelization with the larger
//      outer pixels is implemented. THIS IS NOT YET FULLY
//      TESTED, but first rough tests show that it works
//      at least to a good approximation.
//
//      The present version implements the camera outline
//      with 18 "gap-pixels" and 595 pixels in total as
//      shown in 
//
//         http://sarastro.ifae.es/internal/home/hardware/camera/numbering.ps
//
//      This change involved 
//
// 	(i) The file pixels.dat is no longer needed. Instead
//           the coordinates are generated by the program itself
//           (takes maybe 1 second). In the file 
//
// 		pixel-coords.txt
//
// 	  in the same directory as this README, you find a list
//           of the coordinates generated by this new routine. It
//           has the format
//
//               number   i   j   x  y  size-factor
//
//           where i and j are J.C.G.'s so called biaxis hexagonal
//           coordinates (for internal use) and x and y are the
//           coordinates of the pixel centers in the standard camera
//           coordinate system in units of centimeters. The value
//           of "size-factor" determines the linear size of the pixel
//           relative to the central pixels. 
//
//         (ii) The magic.def file has two additional parameters
//           which give the number of central pixels and the
//           number of gap pixels
//
//         (iii) In camera.h and camera.cxx several changes were 
//           necessary, among them the introduction of several
//           new functions 
//
//      The newly suggested outline with asymmetric Winston cones
//      will be implemented in a later version.
//
//   f) phe files can no longer be read since this contradicts
//      our philosophy that the analysis should be done with other
//      programs like e.g. EVITA and not with "camera" itself.
//      This possibility was removed. 
//
//   g) ROOT is no longer invoked with an interactive interface.
//      In this way, camera can better be run as a batch program and
//      it uses less memory.
//
//   h) small changes concerning the variable "t_chan" were necessary in
//      order to avoid segmentation faults: The variable is used as an
//      index and it went sometimes outside the limits when camera
//      was reading proton data. This is because the reflector files
//      don't contain the photons in a chronological order and also
//      the timespread can be considerably longer that the foreseen
//      digitisation timespan. Please see the source code of camera.cxx
//      round about line 1090.
//
//   j) several unused variables were removed, a few warning messages
//      occur when you compile camera.cxx but these can be ignored at
//      the moment.
//
// In general the program is of course not finished. It still needs
// debugging, proper trigger simulation, simulation of the asymmetric
// version of the outer pixels, proper NSB simulation, adaption of
// the diag "ntuple" contents to our need and others small improvements.
//
// In the directory rfl-files there is now a file in reflector format 0.4
// containing a single event produced by the starfiled adder. It has
// a duration of 30 ns and represents the region around the Crab Nebula.
// Using the enclosed input parameter file, camera should process this
// file without problems.
//
// 2) The README for the previous version of root_camera
//
// README for a preliminary version of the 
// root_camera program. 
//
// root_camera is based on the program "camera"of Jose Carlos
// Gonzalez. It was changed in the way that only the pixelisation 
// and the distibution of the phe to the FADCs works in a 
// first version. 
//
// Using the #undef command most possibilities of the orignal 
// program are switched of. 
//
// The new parts are signed by 
//
// - ROOT or __ROOT__ 
//   nearly all  important codelines for ROOT output are enclosed 
//   in structures like 
//   #ifdef __ROOT__ 
//   
//     code 
//
//   #endif __ROOT__ 
//
//   In same case the new lines are signed by a comment with the word 
//   ROOT in it. 
//
//   For timing of the pulse some variable names are changed. 
//   (t0, t1, t  -->  t_ini, t_fin, t_1st, t_chan,...) 
//   Look also for this changes. 
//
//   For the new root-file is also a change in readparm-files
//
//
// - __DETAIL_TRIGGER__
//
//   This is for the implementation of the current work on trigger 
//   studies. Because the class MTrigger is not well documented it 
//   isnt a part of this tar file. Only a dummy File exists. 
//
//
//
// With all files in the archive, the root_camera program should run. 
//
// A reflector file is in the directory rfl-files
//
// ==================================================
//
// From now on, use CVS for development!!!!
//
//
//
// Revision 1.2  1999/10/22 15:01:28  petry
// version sent to H.K. and N.M. on Fri Oct 22 1999
//
// Revision 1.1.1.1  1999/10/21 16:35:11  petry
// first synthesised version
//
// Revision 1.6  1999/03/15  14:59:08  gonzalez
// camera-1_1
//
// Revision 1.5  1999/03/02  09:56:12  gonzalez
// *** empty log message ***
//
// Revision 1.4  1999/01/14  17:32:41  gonzalez
// Added to camera the STDIN input option (data_from_input)
//
//!@}

//=EOF
