
/********************************************************************\

  Name:         HV.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Main class for HV supply
                
\********************************************************************/

#include "HV.h"


// Constructor
HVBoard::HVBoard(int DeviceNumber, char *DeviceName, bool TestMode, float TimeOut, bool verbose) {
   
  char Buffer[BUFFER_LENGTH];
  struct termios tio;
 
  fTestMode = TestMode;
  fTestModeWrap = 0;

  SetTimeOut(TimeOut);
  BoardNumber = DeviceNumber;
  BoardName = DeviceName;
  
  for (int i=0; i<NUM_CHAINS; i++) Overcurrent[i] = false;
  ResetButton = false;
  WrapOK = true;
  LastWrapCount = -1;
  
  if (fTestMode) {
    fDescriptor = 99;
    return;
  }

  ClearVoltageArrays();

  // Open device (do not warn on non-existing device)
  snprintf(Buffer, BUFFER_LENGTH, "/dev/%s",DeviceName);
  if ((fDescriptor = open(Buffer, O_RDWR|O_NOCTTY|O_NDELAY)) == -1) {
    if(errno != 2) printf("Error: Could not open device %d/%s (%s)\n", DeviceNumber,DeviceName, strerror(errno));
    return;
  }

  // Get current serial port settings
  if (tcgetattr(fDescriptor, &tio) == -1) {
    printf("tcgetattr(...) failed (%d/%s)\n", errno, strerror(errno));
    return;   
  }

  // Set baudrate and raw mode
  if (cfsetspeed(&tio, BAUDRATE) == -1) printf("Error: Could not set baud rate (%s)\n", strerror(errno));
  cfmakeraw(&tio);
  if (tcsetattr(fDescriptor, TCSANOW, &tio ) == -1) printf("Errsor with tcsetattr(...) (%s)\n", strerror(errno));

  //  Synchronize HV board (if fails, closes device and sets fDescriptor to -2) 
  unsigned char wbuf = REG_STATUS;
  int trial = 0, ret;
  
  while(++trial<=3) {
    if((ret = Communicate(stdout, &wbuf, 1, verbose)) == 1) {
      Reset(stdout, verbose);
      return;
    }
    if (ret==0) break;
    wbuf = 0;
  }
  close(fDescriptor);
  fDescriptor = -2;
  return;
}

// Destructor (Resetting board)
HVBoard::~HVBoard() {
  if(fDescriptor >= 0 && !fTestMode) {
    Reset(stdout, false);
    close(fDescriptor);
  }
}


// Communicate: Write and read from HV Board until fTimeOut has been reached 
//
// Returns: 0 read error, 1 success, -1 fTimeOut exceeded
int HVBoard::Communicate(FILE* fptr, unsigned char* wbuf, int Bytes, bool verbose) {

  unsigned char rbuf;
  int ret;
  fd_set SelectDescriptor;
  
  // === Write data ===
  if (write(fDescriptor, wbuf, Bytes) < Bytes) {
    fprintf(fptr," Error: could not write (all) data to HV board\n");
    return 0;
  }
  if (verbose) {
    fprintf(fptr,"%d bytes written:\n", Bytes);
    for (int i=0; i<Bytes; i++) fprintf(fptr," Byte %d: %#.2x\n",i,wbuf[i]);
  }

  // === Try to read until time-out ===
  FD_ZERO(&SelectDescriptor);   FD_SET(fDescriptor, &SelectDescriptor);
  struct timeval WaitTime = {(long) fTimeOut, (long) ((fTimeOut-(long) fTimeOut)*1e6)};
  if (select(fDescriptor+1, &SelectDescriptor, NULL, NULL, &WaitTime)==-1) {
    printf("Error with select() (%d/%s)\n", errno,strerror(errno));
    return 0;
  }
  // Time-out expired?
  if (!FD_ISSET(fDescriptor, &SelectDescriptor)) {
    if (verbose) printf("Time-out of %.2f seconds expired while reading\n", fTimeOut);
    return -1;
  }
  // Read error?    
  if ((ret = read(fDescriptor, &rbuf, 1)) == -1) {
    fprintf(stderr, " Read error (%s)\n", strerror(errno));
    return 0;
  }

  // === Update status information ===
  if (LastWrapCount != -1) {
    if ((LastWrapCount+1)%8 == (rbuf & 0X07)) WrapOK = true;
    else WrapOK = false;
  }
  LastWrapCount = rbuf & 0X07;  
  for (int i=0; i<NUM_CHAINS; i++) Overcurrent[i]=(rbuf & (0X08 << i));
  ResetButton = (bool) (rbuf & 0X80);

  if (verbose && ret==1) fprintf(fptr," 1 byte read: %#.2x\n", rbuf);  
  
  return 1;
}


/* Reset HV board - uses TRead() and has same return values */
int HVBoard::Reset(FILE* fptr, bool verbose) {
  
  if (fTestMode) return 1; 

  unsigned char wbuf[] = {REG_RESET,0,0};
  int ret;
  
  if((ret = Communicate(fptr, wbuf, 3, verbose)) == 1) ClearVoltageArrays();  
  return ret;
}


/* Read status register - uses TRead() and has same return values */
int HVBoard::GetStatus(FILE* fptr, bool verbose) {

  if (fTestMode) return 1;

  unsigned char wbuf[] = {REG_STATUS,0,0};
  
  return Communicate(fptr, wbuf, 3, verbose);
}


/* Set high voltage - uses TRead() and has same return values */
int HVBoard::SetHV(FILE* fptr, int chain, unsigned int channel, unsigned int hv, bool verbose) {

  if (fTestMode){
    printf("Test mode: Nothing to be done. \n");
    return 1;
  }
  
  unsigned char wbuf[] = {0,0,0};

  if (!(hv>=0 && hv<=0x3FFF)) {
    fprintf(fptr," Error: HV beyond limits [0 - 0x3FFF]\n"); 
    return 0;
  }
  
  switch (chain) {
    case 0: wbuf[0] = REG_HV0; break;
    case 1: wbuf[0] = REG_HV1; break;
    case 2: wbuf[0] = REG_HV2; break;
    case 3: wbuf[0] = REG_HV3; break;

    default : fprintf(fptr," Error: chain %d does not exist\n",chain); return 0;
  }
 
  // Assemble bytes
  wbuf[0] |= (unsigned char)((channel >> 2) & 0X00000007);  // Add address [A4-A3]
  wbuf[1] |= (unsigned char)((hv >> 8) & 0X000000FF);       // Add [D13-D8]
  wbuf[1] |= (unsigned char)((channel << 6)  & 0X000000C0); // Add [A1-A0]
  wbuf[2] |= (unsigned char)(hv & 0X000000FF);              // Add [D7-D0]
   
  return Communicate(fptr, wbuf, 3, verbose);
}


// Set all voltages of board to zero
void HVBoard::ClearVoltageArrays() {

  for (int j=0; j<NUM_CHAINS; j++) {
    for (int k=0; k<NUM_CHANNELS; k++){
      HV[j][k] = 0;
      HVV[j][k] = 0.0;
    }
  }
}


// Set time-out to wait for read
void HVBoard::SetTimeOut(double t) {

  if (t >= MIN_TIMEOUT && t <= MAX_TIMEOUT) fTimeOut  = t;
  else fTimeOut  = 1.0;
}
