
/********************************************************************\

  Name:         HVConfig.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Class reading the HV utility configuration file

\********************************************************************/

#include "HVConfig.h"


HVConfig::HVConfig(FILE* fptr, char *configfile) {

  fLogFile   = new char[BUFFER_LENGTH];
  fUSBDevice = new char*[MAX_NUM_HVBOARDS];

  fPixMapTable = new char[BUFFER_LENGTH];
    
  for (int i=0; i<MAX_NUM_HVBOARDS; i++) {
    fUSBDevice[i]      = new char[BUFFER_LENGTH];
    USBDeviceNumber[i] = 0;

    for (int j=0; j<NUM_CHAINS; j++) {
      for (int k=0; k<2; k++) Coef[i][j][k] = 0.;
    }
  }

  TestMode           = false;
  NumHVBoards        = 0;
  FileName           = configfile;
  fStatusRefreshRate = 1.;
  fTimeOut           = 1.;
  DACMin             = 11008;
  DACMax             = 12496;
  fHVCalibOffset      = -.8;
  fHVCalibSlope       = 0.0064;
  fHVMaxDiff          = 1;  

  if (configfile != NULL) {
    if (!ReadHVConfig(fptr, configfile)) {
      fprintf(fptr, "Error (HVConfig): could not configure HV control\n");
      exit(1);
    }
  }
}


HVConfig::~HVConfig() {

  delete [] fLogFile;	delete [] fPixMapTable;
  for (int i=0; i<MAX_NUM_HVBOARDS; i++) delete [] fUSBDevice[i];
  delete [] fUSBDevice;
}


int HVConfig::ReadHVConfig(FILE* fptr, char *configfile) {

  FILE *f;
  char str[MAX_COM_SIZE], dev[MAX_COM_SIZE];
  int j = 0;

  if ((f = fopen(configfile,"r")) == NULL) {
    fprintf(fptr,"Could not open configuration file: %s\n", configfile);
    return 0;
  }
  else fprintf(fptr,"Opening configuration file: %s\n", configfile);
 
  ReadCard("LogFile",    fLogFile,    's',f);
  ReadCard("PixMapTable",fPixMapTable,'s',f);
  ReadCard("TestMode",   &str,        's',f);
  if (!strcmp(str,"TRUE")) TestMode = true;

  for (int i=0; i<MAX_NUM_HVBOARDS; i++) {
    sprintf(str,"Board%d",i);

    if (ReadCard(str, dev, 's', f)) {
      USBDeviceNumber[j] = i;
      sprintf(fUSBDevice[j++],"%s",dev);
      NumHVBoards++;     
    }
  }
  
  ReadCard("TimeOut",           &fTimeOut,           'f', f);
  ReadCard("StatusRefreshRate", &fStatusRefreshRate, 'f', f);
  ReadCard("CCPort",            &fCCPort,            'I', f);
  ReadCard("DACMin",            &DACMin,             'I', f);
  ReadCard("DACMax",            &DACMax,             'I', f);
  ReadCard("HVCalibOffset",     &fHVCalibOffset,     'f', f);
  ReadCard("HVCalibSlope",      &fHVCalibSlope,      'f', f);
  ReadCard("HVMaxDiff",         &fHVMaxDiff,         'U', f);

  fclose(f);
  return 1;
}


int HVConfig::PrintHVConfig(FILE *fptr) {
  
  fprintf(fptr," \n HV control configuration (%s):\n\n", FileName);
  fprintf(fptr," Log file:          %s\n", fLogFile);
  fprintf(fptr," Pixel map table:   %s\n", fPixMapTable);
  fprintf(fptr," Test mode:         %s\n", TestMode ? "yes" : "no");
  fprintf(fptr," %d USB devices:\n", NumHVBoards);
  
  for (int i=0;i<NumHVBoards;i++) 
    fprintf(fptr," Board%d: %s\n", USBDeviceNumber[i], fUSBDevice[i]);
  
  fprintf(fptr,"\n");
  fprintf(fptr," TimeOut:           %.2f s\n",   fTimeOut);
  fprintf(fptr," StatusRefreshRate: %.2f Hz\n",fStatusRefreshRate);
  fprintf(fptr," CCPort:            %d\n",       fCCPort);
  fprintf(fptr," DACMin value:      %d\n",     DACMin);
  fprintf(fptr," DACMax value:      %d\n",     DACMax);
  fprintf(fptr," HVCalibOffset :    %f\n",     fHVCalibOffset);
  fprintf(fptr," HVCalibSlope :     %f\n",     fHVCalibSlope);
  fprintf(fptr," HVMaxDiff :        %u\n",     fHVMaxDiff);
 
  return 1;
}

// ReadCard function (original version by F. Goebel)
// Data is read into an array if MaxNum is larger than 1
bool ReadCard(const char *card_flag, void *store, char Type, FILE *File, unsigned int MaxNum) {
  
  char *card_name, *card_val, Buffer[MAX_COM_SIZE];
  unsigned int Count=0;
  
  rewind(File);

  while (fgets(Buffer, sizeof(Buffer), File) != NULL) {    // Read line by line
    card_name = strtok(Buffer," \t\n");
    
     // Ignore empty lines, comments, and skip if card name does not match
    if (card_name==NULL || card_name[0]=='#' || strcmp(card_name, card_flag)!=0) continue;

    // Read numbers of given type (if MaxNum>1 read array)
    while ((card_val=strtok(NULL," \t\n")) != NULL && Count++<MaxNum) {
      switch (Type) {
	case 'I': *(((int *&) store)++) = (int) strtol(card_val, NULL, 10);
	      	  break;
	case 'i': *(((short *&) store)++) = (short) strtol(card_val, NULL, 10);
	      	  break;
	case 'U': *(((unsigned int *&) store)++) = (unsigned int) strtoul(card_val, NULL, 10);
	      	  break;
	case 'u': *(((unsigned short *&) store)++) = (unsigned short) strtoul(card_val, NULL, 10);
	      	  break;
	case 'f': *(((float *&) store)++) = atof(card_val);
	      	  break;
	case 'd': *(((double *&) store)++) = atof(card_val);
	      	  break;
	case 's': sprintf((char *) store, "%s", card_val);
	      	  break;
	case 'c': *((char *) store) = card_val[0];
	      	  break;
	default:  fprintf(stderr,"Warning: Unknown type '%c' for reading of configuration file\n", Type);
	      	  return false;
      }
    }
    return true;  // Finished reading data for card name  
  }
  return false;
}

