#ifndef MARS_MSequence
#define MARS_MSequence

#ifndef ROOT_TArrayI
#include <TArrayI.h>
#endif

#ifndef MARS_MTime
#include "MTime.h"
#endif

class MDirIter;

class MSequence : public MParContainer
{
    friend class MSequenceSQL;
public:
    enum LightCondition_t { kNA, kNoMoon, kTwilight, kMoon, kDay };
    enum FileType_t {
        kRawDat, kRawPed, kRawCal, kRawAll,
        kRootDat, kRootPed, kRootCal, kRootAll,
        kCalibrated, kImages
    };

private:
    TString fFileName;         // Expanded file name
    TString fDataPath;         // Path to data files

    UShort_t fTelescope;       // Telescope number
    UInt_t   fSequence;        // Sequence number

    MTime  fStart;             // Start time of sequence

    UInt_t fLastRun;           // Last run in sequence (necessary?)
    UInt_t fNumEvents;         // Number of events in sequence

    UInt_t fPeriod;            // Observation period of the sequence
    MTime  fNight;             // Night (day of sunrise) of the sequence

    LightCondition_t fLightCondition;

    TString fProject;
    TString fSource;
    TString fTriggerTable;
    TString fHvSettings;
    TString fComment;

    TArrayI fRuns;            // Run numbers
    TArrayI fRunsSub;         // Sub runs (files)

    TArrayI fCalRuns;         // Numbers of calibration runs/files
    TArrayI fCalRunsSub;      // Sub runs (files) of calibration runs

    TArrayI fPedRuns;         // Numbers of pedestal runs/files
    TArrayI fPedRunsSub;      // Sub runs (files) of pedestal runs

    TArrayI fDatRuns;         // Numbers of data runs/files
    TArrayI fDatRunsSub;      // Sub runs (files) of data runs

    TArrayI fExclRuns;        // Numbers of excluded runs/files
    TArrayI fExclRunsSub;     // Sub runs (files) of excluded runs

    Bool_t fMonteCarlo;       // Flag for Monte Carlo sequences

    // Helper for interpretation
    void    AddEntry(Int_t run, Int_t file, TArrayI &data, TArrayI &sub) const;
    void    EvalEntry(const TEnv *env, const TString &prefix, const TString &num, TArrayI &data, TArrayI &sub) const;

    void    Split(TString &runs, TArrayI &data, TArrayI &sub, const TEnv *env=0, const TString prefix="") const;
    void    Split(const TEnv &env, const TString &prefix, const char *name, TArrayI &data, TArrayI &sub) const;

    LightCondition_t GetLightCondition(const TString &str) const;

    // Helper for file setup
    TString GetPathName(TString d, FileType_t type) const;
    TString GetFileName(UInt_t num, const TArrayI &arr, const TArrayI &sub, FileType_t type) const;
    UInt_t  SetupRuns(MDirIter &iter, const TArrayI &arr, const TArrayI &file, FileType_t type, const char *path=0) const;

    // Helper for Print()
    TString GetNumSequence(Int_t pos, Int_t n, const TArrayI &f) const;
    TString GetNumSequence(Int_t &pos, const TArrayI &n, const TArrayI &f) const;

    void    PrintRunsClassic(ostream &out, const char *pre, const char *name, const TArrayI &r) const;
    TString PrintRuns(ostream &out, const char *pre, const char *name, const TArrayI &r, const TArrayI &f) const;

    // General helper
    Bool_t  HasSubRuns() const { return fRunsSub.GetSize()!=0 || fDatRunsSub.GetSize()!=0 || fPedRunsSub.GetSize()!=0 || fCalRunsSub.GetSize()!=0 || fExclRunsSub.GetSize()!=0; }
    Bool_t  IsContained(const TArrayI &arr, const TArrayI &sub, UInt_t num, UInt_t file) const;

    // Some helpers to handle the arrays
    static Int_t SortArraySub(Int_t p, Int_t n, Int_t *arr1, Int_t *arr2);
    static void  SortArrays(TArrayI &arr1, TArrayI &arr2);
    static Int_t GetSubArray(Int_t p, Int_t n, Int_t *arr1);

public:
    MSequence() : fTelescope(0), fSequence((UInt_t)-1), fLastRun((UInt_t)-1),
        fNumEvents((UInt_t)-1), fPeriod((UInt_t)-1), fLightCondition(kNA), fMonteCarlo(kFALSE)
    {
        fName  = "MSequence";
        fTitle = "Sequence file";
    }
    MSequence(const char *fname, const char *path="", UInt_t id=(UInt_t)-1);
    MSequence(const char *fname, UInt_t seq, UShort_t tel=1);
    MSequence(const MSequence &s) : MParContainer(s),
        fFileName(s.fFileName), fDataPath(s.fDataPath),
        fTelescope(s.fTelescope),
        fSequence(s.fSequence), fStart(s.fStart),
        fLastRun(s.fLastRun), fNumEvents(s.fNumEvents), fPeriod(s.fPeriod),
        fNight(s.fNight), fLightCondition(s.fLightCondition),
        fProject(s.fProject), fSource(s.fSource),
        fTriggerTable(s.fTriggerTable), fHvSettings(s.fHvSettings),
        fRuns(s.fRuns), fRunsSub(s.fRunsSub), fCalRuns(s.fCalRuns),
        fCalRunsSub(s.fCalRunsSub), fPedRuns(s.fPedRuns),
        fPedRunsSub(s.fPedRunsSub), fDatRuns(s.fDatRuns),
        fDatRunsSub(s.fDatRunsSub), fExclRuns(s.fExclRuns),
        fExclRunsSub(s.fExclRunsSub), fMonteCarlo(s.fMonteCarlo) { }

    Bool_t IsSimilar(const MSequence &s) const;
    Bool_t IsIdentical(const MSequence &s) const;
    Bool_t operator==(const MSequence &s) const;

    // I/O
    Bool_t WriteFile(const char *filename, const Option_t *o) const;
    Bool_t WriteFile(const char *filename) const { return WriteFile(filename,""); } //*MENU *ARGS={filename=>fBaseName}

    // TObject
    void Print(ostream &out, Option_t *o) const;
    void Print(Option_t *o) const;
    void Print() const { Print(""); } //*MENU*

    // Genaral interface
    Bool_t IsValid() const { return fTelescope>0 && fSequence!=(UInt_t)-1; }
    Bool_t IsMonteCarlo() const { return fMonteCarlo; }
    Bool_t IsExcluded(UInt_t run, UInt_t file) const { return IsContained(fExclRuns, fExclRunsSub, run, file); }
    Bool_t IsContained(UInt_t run, UInt_t file) const { return IsContained(fCalRuns, fCalRunsSub, run, file) || IsContained(fPedRuns, fPedRunsSub, run, file) || IsContained(fDatRuns, fDatRunsSub, run, file); }

    // Setter
    void SetNight(const char*night);

    void AddRun(UInt_t run, char type='*') { AddFile(run, 0, type); }
    void AddRuns(UInt_t run1, UInt_t run2, char type='*') { for (UInt_t i=run1; i<=run2; i++) AddFile(i, 0, type); }
    void AddFile(UInt_t run, UInt_t file, char type='*');
    void AddFiles(UInt_t run, UInt_t f1, UInt_t f2, char type='*') { for (UInt_t i=f1; i<=f2; i++) AddFile(run, i, type); }

    void ExcludeFile(UInt_t num, UInt_t file=0);
    void ExcludeRuns(TString runs);

    void SetMonteCarlo(Bool_t ismc=kTRUE) { fMonteCarlo=ismc; }

    // Getter
    UInt_t GetNumExclRuns() const { return fExclRuns.GetSize(); }

    UShort_t GetTelescope() const { return fTelescope; }
    UInt_t GetSequence() const { return fSequence; }
    UInt_t GetLastRun() const  { return fLastRun; }
    UInt_t GetPeriod() const   { return fPeriod; }
    Bool_t HasMoon() const { return fLightCondition==kMoon; }

    LightCondition_t GetLightCondition() const { return fLightCondition; }

    const MTime   &GetStart() const { return fStart; }
    const MTime   &GetNight() const { return fNight; }
    const TString &GetSource() const { return fSource; } 

    const TString GetExcludedRuns() const;

    // Filesystem interface
    UInt_t SetupPedRuns(MDirIter &iter, const char *path=0, Bool_t raw=kFALSE) const;
    UInt_t SetupDatRuns(MDirIter &iter, const char *path=0, Bool_t raw=kFALSE) const;
    UInt_t SetupAllRuns(MDirIter &iter, const char *path=0, Bool_t raw=kFALSE) const;
    UInt_t SetupCalRuns(MDirIter &iter, const char *path=0, Bool_t raw=kFALSE) const;
    UInt_t SetupDatRuns(MDirIter &iter, FileType_t type, const char *path=0) const;

    // Filesystem getter
    const char    *GetBaseName() const;
    const char    *GetFilePath() const;
    const TString &GetFileName() const { return fFileName; }
    const TString &GetDataPath() const { return fDataPath; }

    const TString  GetStandardPath() const { return GetStandardPath(fMonteCarlo); }

    // Static function for external access
    static TString GetStandardPath(Bool_t mc) { return mc?"/magic/montecarlo/":"/magic/data/"; }
    static Bool_t  InflateSeq(TString &seq, /*FIXME: TELNUM*/ Bool_t ismc=kFALSE);

    static TString InflateRunName(const MTime &night, UShort_t tel, Int_t run, Int_t file, Int_t type=kRawAll);
    static TString InflateRunName(const MTime &night, Int_t run, Int_t type=kRawAll) { return InflateRunName(night, 1, run, 0, type); }

    static TString InflateRunPath(const MTime &night, UShort_t tel, Int_t run, Int_t file=0, Int_t type=kRawAll);
    static TString InflateRunPath(const MTime &night, Int_t run, Int_t type=kRawAll) { return InflateRunPath(night, 1, run, 0, type); }

    static TString InflateRunPath(const MTime &night, Bool_t mc=kFALSE);

    ClassDef(MSequence, 6) // Describes a sequences, reads and writes sequence files
};

#endif
