//
// This File contains the definition of the MGCoordinates-class
//
//   Author: Thomas Bretz
//   Version: V1.0 (1-8-2000)

#include "MGSkyPosition.h"

#include <iostream.h>  // cout

#include <TArc.h>
#include <TLine.h>
#include <TList.h>
#include <TCanvas.h>

#include "coord.h"
#include "SlaPlanets.h"

void MGSkyPosition::InitPlanets()
{
    //
    // Magnitudes:
    // -----------
    //   Moon     -12.8
    //   Jupiter   -4.6
    //   Venus     -4.0
    //   Mars      -2.0
    //   Merkur     0.0
    //   Saturn     0.7
    //   Uranus     5.5
    //   Neptun     7.8
    //
    fSlalib = new SlaPlanets;

    //
    // Colors: black .. white:
    //
    //   1, 12, 13, 14, 15, 16, 17, 18, 19, 10
    //
    fMars = new TArc(0, 0, 1);
    fMars->SetFillStyle(1001);     // solid      (s. TAttFill)
    fMars->SetFillColor(13);       // light grey (s. TAttFill)
    fMars->SetLineColor(12);
    fMars->Draw();

    fVenus = new TArc(0, 0, 2);
    fVenus->SetFillStyle(1001);    // solid      (s. TAttFill)
    fVenus->SetFillColor(15);      // light grey (s. TAttFill)
    fVenus->SetLineColor(14);
    fVenus->Draw();

    fJupiter = new TArc(0, 0, 2);
    fJupiter->SetFillStyle(1001);  // solid      (s. TAttFill)
    fJupiter->SetFillColor(17);    // light grey (s. TAttFill)
    fJupiter->SetLineColor(16);
    fJupiter->Draw();

    fMoon = new TArc(0, 0, 3, 290, 70);
    fMoon->SetFillStyle(1001);     // solid      (s. TAttFill)
    fMoon->SetFillColor(19);       // light grey (s. TAttFill)
    fMoon->SetLineColor(15);
    fMoon->Draw();

    for (int i=0; i<4; i++)
    {
        fSunL[i] = new TLine(0,0,0,0);
        fSunL[i]->SetLineColor(10);    // white gray
        fSunL[i]->SetLineStyle(1);     // solid  (s. TAttLine)
        fSunL[i]->Draw();

        fList->Add(fSunL[i]);
    }

    fSun = new TArc(0, 0, 2);
    fSun->SetFillStyle(1001);     // solid      (s. TAttFill)
    fSun->SetFillColor(10);       // white      (s. TAttFill)
    fSun->SetLineColor(19);
    fSun->Draw();

    fList->Add(fMars);
    fList->Add(fVenus);
    fList->Add(fJupiter);
    fList->Add(fMoon);
    fList->Add(fSun);

    //
    // Reset 'static' positions of planets
    //
    for (int i=0; i<9; i++)
    {
        fX[i] = ~0;
        fY[i] = ~0;
    }
}

void MGSkyPosition::DrawCoordinateSystem()
{
    TArc arc;
    arc.SetFillStyle(4000);  // transparent
    arc.SetLineColor(13);    // dark gray
    arc.SetLineStyle(1);     // solid  (s. TAttLine)
    arc.DrawArc(0, 0, 90);
    arc.DrawArc(0, 0, 67.5);
    arc.DrawArc(0, 0, 45);
    arc.DrawArc(0, 0, 22.5);

    TLine line;
    line.SetLineColor(1);  // black
    line.SetLineStyle(1);  // solid  (s. TAttLine)
    line.DrawLine(-95,   0, 95,  0);
    line.DrawLine(  0, -95,  0, 95);
}

void MGSkyPosition::InitPosition()
{
    fLin1a = new TLine(0, 0, 0, 0);
    fLin1b = new TLine(0, 0, 0, 0);
    fLin2  = new TLine(0, 0, 0, 0);

    fLin1a->SetLineColor(5);  // yellow (s. TAttFill)
    fLin1b->SetLineColor(5);  // yellow
    fLin2 ->SetLineColor(10); // white
    fLin1a->SetLineStyle(1);  // solid  (s. TAttLine)
    fLin1b->SetLineStyle(2);  // dashed

    fLin1a->Draw();
    fLin1b->Draw();
    fLin2->Draw();

    fList->Add(fLin1a);
    fList->Add(fLin1b);
    fList->Add(fLin2);
}

MGSkyPosition::MGSkyPosition(const TGWindow* p, const Int_t w, const Int_t h)
    : TRootEmbeddedCanvas("SkyPosition", p, w-1, h-1, kRaisedFrame),//, 0) //234, 76, kFixedSize)
    fModified(kFALSE)
{
    fCanvas = GetCanvas();

    fCanvas->SetFillColor(39); // s. TAttFill
    fCanvas->Range(-100, -100, 100, 100);

    fList = new TList;
    fList->SetOwner();

    DrawCoordinateSystem();
    InitPosition();
    InitPlanets();

    Resize(w, h); //GetDefaultSize()); // ???
    MapSubwindows();
}

MGSkyPosition::~MGSkyPosition()
{
    delete fList;
    delete fSlalib;

    cout << "MGSkyPosition destroyed." << endl;
}

void MGSkyPosition::SetLin1(int x, int y)
{
    fLin1a->SetX2(x);
    fLin1a->SetY2(y);
    fLin1b->SetX2(-x);
    fLin1b->SetY2(-y);
}

void MGSkyPosition::SetLin2(int x1, int y1, int x2, int y2)
{
    fLin2->SetX1(x1);
    fLin2->SetX2(x2);
    fLin2->SetY1(y1);
    fLin2->SetY2(y2);
}

void MGSkyPosition::DrawPosition(Float_t zd, Float_t az)
{
    static int X=~0;
    static int Y=~0;

    const float rad = D2PI*az/360.0;

    const float s = sin(rad);
    const float c = cos(rad);

    const int x = (int)(s*zd);
    const int y = (int)(c*zd);

    if (X==x && Y==y)
        return;

    X = x;
    Y = y;

    const int dx = (int)(s*4);
    const int dy = (int)(c*4);

    SetLin1(s*95, c*95);
    SetLin2(x+dy, y-dx, x-dy, y+dx);

    fModified = kTRUE;
}

void MGSkyPosition::Update(ZdAz &pos)
{
    //
    // calculate actual time for planet positions
    //
    fSlalib->SetMjd2Now();

    DrawPlanet(kESun,     fSun);
    DrawPlanet(kEMoon,    fMoon);
    DrawPlanet(kEJupiter, fJupiter);
    DrawPlanet(kEVenus,   fVenus);
    DrawPlanet(kEMars,    fMars);

    DrawPosition(pos.Zd(), pos.Az());

    if (!fModified)
        return;

    // FIXME: Sometimes (if the canvas couldn't be created correctly:
    // X11 Pixmap error) Update hangs the Gui system.

    fCanvas->Modified();
    fCanvas->Update();

    fModified = kFALSE;
}

void MGSkyPosition::DrawPlanet(Int_t planet, TArc *arc)
{
    //
    // calculate actual position of planet
    //
    fSlalib->UpdatePlanetPos((ePlanets_t)planet);

    ZdAz zdaz = fSlalib->GetPlanetPos((ePlanets_t)planet);

    //
    // check whether we have to change the position where it is drawn
    //
    const float s = sin(zdaz.Az());
    const float c = cos(zdaz.Az());

    const float deg = 360.0/D2PI*zdaz.Zd();

    const int x = (int)(s*deg);
    const int y = (int)(c*deg);

    if (fX[planet]==x && fY[planet]==y)
        return;

    fX[planet] = x;
    fY[planet] = y;

    //
    // Set new position
    //
    arc->SetX1(x);
    arc->SetY1(y);

    fModified = kTRUE;

    if (planet!=kESun)
        return;

    cout << "Sun: x=" << x << " y=" << y;
    cout << "   Zd=" << deg << " Az=" << 360./D2PI*zdaz.Az() << endl;

    fSunL[0]->SetX1(x-4); fSunL[0]->SetX2(x+4);
    fSunL[1]->SetX1(x-3); fSunL[1]->SetX2(x+3);
    fSunL[2]->SetX1(x-3); fSunL[2]->SetX2(x+3);
    fSunL[3]->SetX1(x  ); fSunL[3]->SetX2(x  );

    fSunL[0]->SetY1(y  ); fSunL[0]->SetY2(y  );
    fSunL[1]->SetY1(y-3); fSunL[1]->SetY2(y+3);
    fSunL[2]->SetY1(y+3); fSunL[2]->SetY2(y-3);
    fSunL[3]->SetY1(y-4); fSunL[3]->SetY2(y+4);
}
