#ifndef COORD_H
#define COORD_H

#include <math.h>          // floor

#include "slalib/slamac.h" // D2PI

class Deg
{
protected:
    double fDeg;

public:
    Deg(const double d) : fDeg(d) {}
    Deg(const Deg &c) { fDeg = c.fDeg; }

    void Set(double d) { fDeg=d; }

    double operator()() { return fDeg; }

    operator double() const { return fDeg*D2PI/360.0; }
};

class XY
{
protected:
    double fX;
    double fY;

public:
    XY(double x=0, double y=0) : fX(x), fY(y) {}
    XY(const XY &c) { fX = c.fX; fY = c.fY; }

    void Set(double x, double y) { fX=x; fY=y; }

    double X() const { return fX; }
    double Y() const { return fY; }

    void operator/=(double c) { fX/=c; fY/=c; }
    void operator*=(double c) { fX*=c; fY*=c; }

    XY operator/(double c) const { return XY(fX/c, fY/c); }
    XY operator*(double c) const { return XY(fX*c, fY*c); }
    XY operator*(const XY &c) const { return XY(fX*c.fX, fY*c.fY); }
    XY operator/(const XY &c) const { return XY(fX/c.fX, fY/c.fY); }
    XY operator+(const XY &c) const { return XY(fX+c.fX, fY+c.fY); }
    XY operator-(const XY &c) const { return XY(fX-c.fX, fY-c.fY); }

    double Sqr()   const { return fX*fX + fY*fY; }
    double Sqrt()  const { return sqrt(Sqr()); }
    double Ratio() const { return fX/fY; }
    void Round()         { fX=(int)(floor(fX+.5)); fY=(int)(floor(fY+.5)); }
};

class AltAz : public XY
{
public:
    AltAz(double alt=0, double az=0) : XY(alt, az) {}

    double Alt() const { return fX; }
    double Az()  const { return fY; }

    void Alt(double d) { fX=d; }
    void Az(double d)  { fY=d; }
    void operator*=(const XY &c)    { fX*=c.X(); fY*=c.Y(); }
    void operator-=(const AltAz &c) { fX-=c.fX; fY-=c.fY; }

    AltAz operator/(double c) const { return AltAz(fX/c, fY/c); }
    AltAz operator*(double c) const { return AltAz(fX*c, fY*c); }
    AltAz operator*(const XY &c) const { return AltAz(fX*c.X(), fY*c.Y()); }
    AltAz operator/(const XY &c) const { return AltAz(fX/c.X(), fY/c.Y()); }
    AltAz operator+(const AltAz &c) const { return AltAz(fX+c.fX, fY+c.fY); }
    AltAz operator-(const AltAz &c) const { return AltAz(fX-c.fX, fY-c.fY); }
};

class ZdAz : public XY
{
public:
    ZdAz(double zd=0, double az=0) : XY(zd, az) {}
    ZdAz(const ZdAz &c) : XY(c) {}

    double Zd() const { return fX; }
    double Az() const { return fY; }

    void Zd(double d) { fX=d; }
    void Az(double d) { fY=d; }
    void operator*=(const XY &c)   { fX*=c.X(); fY*=c.Y(); }
    void operator-=(const ZdAz &c) { fX-=c.fX; fY-=c.fY; }
    void operator+=(const ZdAz &c) { fX+=c.fX; fY+=c.fY; }

    ZdAz operator/(double c) const { return ZdAz(fX/c, fY/c); }
    ZdAz operator*(double c) const { return ZdAz(fX*c, fY*c); }
    ZdAz operator*(const XY &c) const { return ZdAz(fX*c.X(), fY*c.Y()); }
    ZdAz operator/(const XY &c) const { return ZdAz(fX/c.X(), fY/c.Y()); }
    ZdAz operator+(const ZdAz &c) const { return ZdAz(fX+c.fX, fY+c.fY); }
    ZdAz operator-(const ZdAz &c) const { return ZdAz(fX-c.fX, fY-c.fY); }
};

class RaDec : public XY
{
public:
    RaDec(double ra=0, double dec=0) : XY(ra, dec) {}

    double Ra()  const { return fX; }
    double Dec() const { return fY; }

    RaDec operator/(double c) const { return RaDec(fX/c, fY/c); }
    RaDec operator*(double c) const { return RaDec(fX*c, fY*c); }
    RaDec operator*(const XY &c) const { return RaDec(fX*c.X(), fY*c.Y()); }
    RaDec operator+(const RaDec &c) const { return RaDec(fX+c.fX, fY+c.fY); }
    RaDec operator-(const RaDec &c) const { return RaDec(fX-c.fX, fY-c.fY); }
};

inline double Rad2Deg(double rad)
{
    return 360.0/D2PI*rad;
}

inline double Deg2Rad(double rad)
{
    return D2PI/360.0*rad;
}
#endif
