/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  1/2009 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2009
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MSimPointingPos
//
// This task is meant to simulate the pointing position (mirror orientation).
// This depends on the direction from which the shower is coming but also
// on the user request (e.g. Wobble mode).
//
// WARNING: Currently the telescope orientation is just fixed to the
//          direction in the run-header (if a view cone was given) or
//          the direction in the evt-header (if no view cone given)
//
//  Input Containers:
//   MCorsikaRunHeader
//   MCorsikaEvtHeader
//
//  Output Containers:
//   MPointingPos
//
//////////////////////////////////////////////////////////////////////////////
#include "MSimPointingPos.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MCorsikaEvtHeader.h"
#include "MCorsikaRunHeader.h"

#include "MPointingPos.h"

ClassImp(MSimPointingPos);

using namespace std;

// --------------------------------------------------------------------------
//
//  Default Constructor.
//
MSimPointingPos::MSimPointingPos(const char* name, const char *title)
    : fRunHeader(0), fEvtHeader(0), fPointing(0)
{
    fName  = name  ? name  : "MSimPointingPos";
    fTitle = title ? title : "Task to simulate the pointing position (mirror orientation)";
}


// --------------------------------------------------------------------------
//
// Search for all necessary containers
//
Int_t MSimPointingPos::PreProcess(MParList *pList)
{
    fPointing = (MPointingPos*)pList->FindCreateObj("MPointingPos");
    if (!fPointing)
        return kFALSE;

    fRunHeader = (MCorsikaRunHeader*)pList->FindObject("MCorsikaRunHeader");
    if (!fRunHeader)
    {
        *fLog << err << "MCorsikaRunHeader not found... aborting." << endl;
        return kFALSE;
    }

    fEvtHeader = (MCorsikaEvtHeader*)pList->FindObject("MCorsikaEvtHeader");
    if (!fEvtHeader)
    {
        *fLog << err << "MCorsikaEvtHeader not found... aborting." << endl;
        return kFALSE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
Int_t MSimPointingPos::Process()
{
    // If a view cone is given use the fixed telescope orientation
    const Bool_t fixed = fRunHeader->HasViewCone();

    // Local sky coordinates (direction of telescope axis)
    const Double_t zd = fixed ? fRunHeader->GetZdMin() : fEvtHeader->GetZd()*TMath::RadToDeg();  // x==north
    const Double_t az = fixed ? fRunHeader->GetAzMin() : fEvtHeader->GetAz()*TMath::RadToDeg();  // y==west

    // Setup the pointing position
    fPointing->SetLocalPosition(zd, az);

    return kTRUE;
}

/*
Int_t MSimPointingPos::ReadEnv(const TEnv &env, TString prefix, Bool_t print)
{
    Bool_t rc = kFALSE;
    if (IsEnvDefined(env, prefix, "FileName", print))
    {
        rc = kTRUE;
        SetFileName(GetEnvValue(env, prefix, "FileName", fFileName));
    }

    if (IsEnvDefined(env, prefix, "UseTheta", print))
    {
        rc = kTRUE;
        SetUseTheta(GetEnvValue(env, prefix, "UseTheta", fUseTheta));
    }

    return rc;
}
*/
