/* ======================================================================== *\
!
! *
! * This file is part of CheObs, the Modular Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appears in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz,  1/2009 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: CheObs Software Development, 2000-2009
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//
//  MSimPointingPos
//
// This task is meant to simulate the pointing position (mirror orientation).
// This depends on the direction from which the shower is coming but also
// on the user request (e.g. Wobble mode).
//
// WARNING: Currently the telescope orientation is just fixed to the
//          direction in the run-header (if a view cone was given) or
//          the direction in the evt-header (if no view cone given)
//
//  Input Containers:
//   MCorsikaRunHeader
//   MCorsikaEvtHeader
//
//  Output Containers:
//   MPointingPos
//   PointingCorsika [MPointingPos]
//
//////////////////////////////////////////////////////////////////////////////
#include "MSimPointingPos.h"

#include <TRandom.h>
#include <TVector3.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

#include "MCorsikaEvtHeader.h"
#include "MCorsikaRunHeader.h"

#include "MPointingPos.h"

ClassImp(MSimPointingPos);

using namespace std;

// --------------------------------------------------------------------------
//
//  Default Constructor.
//
MSimPointingPos::MSimPointingPos(const char* name, const char *title)
    : fRunHeader(0), fEvtHeader(0), fPointingCorsika(0), fPointingLocal(0),
    fOffTargetDistance(-1), fOffTargetPhi(-1)

{
    fName  = name  ? name  : "MSimPointingPos";
    fTitle = title ? title : "Task to simulate the pointing position (mirror orientation)";
}


// --------------------------------------------------------------------------
//
// Search for all necessary containers
//
Int_t MSimPointingPos::PreProcess(MParList *pList)
{
    fPointingCorsika = (MPointingPos*)pList->FindCreateObj("MPointingPos", "PointingCorsika");
    if (!fPointingCorsika)
        return kFALSE;

    fPointingLocal = (MPointingPos*)pList->FindCreateObj("MPointingPos");
    if (!fPointingLocal)
        return kFALSE;

    fRunHeader = (MCorsikaRunHeader*)pList->FindObject("MCorsikaRunHeader");
    if (!fRunHeader)
    {
        *fLog << err << "MCorsikaRunHeader not found... aborting." << endl;
        return kFALSE;
    }

    fEvtHeader = (MCorsikaEvtHeader*)pList->FindObject("MCorsikaEvtHeader");
    if (!fEvtHeader)
    {
        *fLog << err << "MCorsikaEvtHeader not found... aborting." << endl;
        return kFALSE;
    }

    // FIXED offset
    // Diffuse?  ( FOV of camera folded with mirror diameter as Corsika input? )
    // Hour angle!
    // Angle to magnetic field!

    if (IsOffTargetObservation())
    {
        *fLog << inf;
        *fLog << "Off target observations switched on with" << endl;
        *fLog <<"   a pointing distance of " << GetOffTargetDistance() << "deg ";
        if (fOffTargetPhi<0)
            *fLog << "randomly distributed in phi." << endl;
        else
            *fLog << "and phi=" << GetOffTargetPhi() << "deg." << endl;
    }

    return kTRUE;
}

/*
Bool_t MSimPointingPos::ReInit(MParList *pList)
{
    // FIXME: Check also the enlightened region on the ground!
    return kTRUE;
}
*/

// --------------------------------------------------------------------------
//
Int_t MSimPointingPos::Process()
{

    // If a view cone is given use the fixed telescope orientation
    const Bool_t fixed = fRunHeader->HasViewCone();

    // Local sky coordinates (direction of telescope axis)
    /*const*/ Double_t zd = fixed ? fRunHeader->GetZdMin() : fEvtHeader->GetZd()*TMath::RadToDeg();  // x==north
    /*const*/ Double_t az = fixed ? fRunHeader->GetAzMin() : fEvtHeader->GetAz()*TMath::RadToDeg();  // y==west

    if (!fixed && IsOffTargetObservation())
    {
        const Double_t theta = zd*TMath::DegToRad();
        const Double_t phi   = az*TMath::DegToRad();

        /*const*/ TVector3 source;
        source.SetMagThetaPhi(1, theta, phi);

        /*const*/ TVector3 point;
        point.SetMagThetaPhi(1, theta+fOffTargetDistance, phi);

        const Double_t delta = fOffTargetPhi>0 ? fOffTargetPhi :
            gRandom->Uniform(TMath::TwoPi());

        point.Rotate(delta, source);

        zd = point.Theta()*TMath::RadToDeg();
        az = point.Phi()*TMath::RadToDeg();
    }

    // Setup the pointing position
    fPointingCorsika->SetLocalPosition(zd, az);
    fPointingLocal->SetLocalPosition(zd, az+fRunHeader->GetMagneticFieldAz());

    // Calculate incident angle between magnetic field direction
    // and pointing direction ( phi and theta? )

    return kTRUE;
}

Int_t MSimPointingPos::ReadEnv(const TEnv &env, TString prefix, Bool_t print)
{
    Bool_t rc = kFALSE;
    if (IsEnvDefined(env, prefix, "OffTargetDistance", print))
    {
        rc = kTRUE;
        SetOffTargetDistance(GetEnvValue(env, prefix, "OffTargetDistance", GetOffTargetDistance()));
    }

    if (IsEnvDefined(env, prefix, "OffTargetPhi", print))
    {
        rc = kTRUE;
        SetOffTargetPhi(GetEnvValue(env, prefix, "OffTargetPhi", GetOffTargetPhi()));
    }

    return rc;
}

