/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 1/2009 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2009
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
//  MJSimulation
//
//
// Force reading a corsika file even if the footer (RUNE-section) is missing
// by setting fForceMode to kTRUE or from the resource file by
//
//    ForceMode: Yes
//
//
// To switch off the simulation of the camera electronics, use:
//
//    Camera: Off
//
// Note, that the border of the camera and the propertied of the cones
// are still simulated (simply because it is fast). Furthermore, this
// switches off the trigger for the output, i.e. all data which deposits
// at least one photon in at least one pixel is written to the output file.
//
//
// In case of a pedestal or calibration run the artificial trigger can
// be "switched off" and the cosmics trrigger "switched on" by setting
// fForceTrigger to kTRUE or from the resource file by
//
//    ForceTrigger: Yes
//
//
/////////////////////////////////////////////////////////////////////////////
#include "MJSimulation.h"

#include <TEnv.h>
#include <TCanvas.h>

// Core
#include "MLog.h"
#include "MLogManip.h"

#include "MArgs.h"
#include "MDirIter.h"
#include "MParList.h"
#include "MTaskList.h"
#include "MEvtLoop.h"

#include "MStatusDisplay.h"

// Tasks
#include "MCorsikaRead.h"
#include "MContinue.h"
#include "MFillH.h"
#include "MGeomApply.h"
#include "MParameterCalc.h"
#include "MHillasCalc.h"
#include "MImgCleanStd.h"
#include "MWriteRootFile.h"

#include "MSimMMCS.h"
#include "MSimAbsorption.h"
#include "MSimAtmosphere.h"
#include "MSimReflector.h"
#include "MSimPointingPos.h"
#include "MSimPSF.h"
#include "MSimGeomCam.h"
#include "MSimSignalCam.h"
#include "MSimAPD.h"
#include "MSimExcessNoise.h"
#include "MSimCamera.h"
#include "MSimTrigger.h"
#include "MSimReadout.h"
#include "MSimRandomPhotons.h"
#include "MSimBundlePhotons.h"
#include "MSimCalibrationSignal.h"

// Histograms
#include "MBinning.h"

#include "MHn.h"
#include "MHCamera.h"
#include "MHCamEvent.h"
#include "MHPhotonEvent.h"

// Container
#include "MRawRunHeader.h"
#include "MParameters.h"
#include "MReflector.h"
#include "MParEnv.h"
#include "MSpline3.h"
#include "MParSpline.h"
#include "MGeomCam.h"

#include "MPedestalCam.h"
#include "MPedestalPix.h"

ClassImp(MJSimulation);

using namespace std;

// --------------------------------------------------------------------------
//
// Default constructor. 
//
// Sets fRuns to 0, fExtractor to NULL, fDataCheck to kFALSE
//
MJSimulation::MJSimulation(const char *name, const char *title)
    : fForceMode(kFALSE), fCamera(kTRUE), fForceTrigger(kFALSE),
    fOperationMode(kModeData)
{
    fName  = name  ? name  : "MJSimulation";
    fTitle = title ? title : "Standard analysis and reconstruction";
}

Bool_t MJSimulation::CheckEnvLocal()
{
    fForceMode    = GetEnv("ForceMode",    fForceMode);
    fForceTrigger = GetEnv("ForceTrigger", fForceTrigger);
    fCamera       = GetEnv("Camera",       fCamera);

    return kTRUE;
}
/*
TString MJSimulation::GetOutFile(const MSequence &seq) const
{
    return seq.IsValid() ? Form("ceres%08d.root", seq.GetSequence()) : "ceres.root";
}
*/

Bool_t MJSimulation::WriteResult(const MParList &plist, const MSequence &seq)
{
    if (fPathOut.IsNull())
    {
        *fLog << inf << "No output path specified via SetPathOut - no output written." << endl;
        return kTRUE;
    }

    TObjArray cont;
    cont.Add(const_cast<TEnv*>(GetEnv()));
    if (seq.IsValid())
        cont.Add(const_cast<MSequence*>(&seq));

    cont.Add(plist.FindObject("PulseShape"));
    cont.Add(plist.FindObject("Reflector"));
    cont.Add(plist.FindObject("MGeomCam"));
    cont.Add(plist.FindObject("GeomCones"));

    TNamed cmdline("CommandLine", fCommandLine.Data());
    cont.Add(&cmdline);

    if (fDisplay)
    {
        TString title = "--  Ceres";
        if (seq.IsValid())
        {
            title += ": ";
            title += seq.GetSequence();
        }
        title += "  --";
        fDisplay->SetTitle("Ceres", kFALSE);

        cont.Add(fDisplay);
    }

    const TString name = seq.IsValid() ? Form("ceres%08d.root", seq.GetSequence()) : "ceres.root";
    return WriteContainer(cont, name, "RECREATE");
}

void MJSimulation::SetupHist(MHn &hist) const
{
    hist.AddHist("MCorsikaEvtHeader.fTotalEnergy");
    hist.InitName("Energy");
    hist.InitTitle("Energy;E [GeV]");
    hist.SetLog(kTRUE, kTRUE, kFALSE);

    hist.AddHist("MPhotonEvent.GetNumExternal");
    hist.InitName("Size");
    hist.InitTitle("Size;S [#]");
    hist.SetLog(kTRUE, kTRUE, kFALSE);

    /*
     hist.AddHist("-MCorsikaEvtHeader.fX/100","-MCorsikaEvtHeader.fY/100");
     hist.SetDrawOption("colz");
     hist.InitName("Impact;Impact;Impact");
     hist.InitTitle("Impact;West <--> East [m];South <--> North [m]");
     hist.SetAutoRange();
     */

    hist.AddHist("MCorsikaEvtHeader.GetImpact/100");
    hist.InitName("Impact");
    hist.InitTitle("Impact;Impact [m]");
    hist.SetAutoRange();

    hist.AddHist("MCorsikaEvtHeader.fFirstInteractionHeight/100000");
    hist.InitName("Height");
    hist.InitTitle("FirstInteractionHeight;h [km]");

    hist.AddHist("(MCorsikaEvtHeader.fAz+MCorsikaRunHeader.fMagneticFieldAz)*TMath::RadToDeg()", "MCorsikaEvtHeader.fZd*TMath::RadToDeg()");
    hist.InitName("SkyOrigin;Az;Zd");
    hist.InitTitle("Sky Origin;Az [\\deg];Zd [\\deg]");
    hist.SetDrawOption("colz");
    hist.SetAutoRange();

    hist.AddHist("IncidentAngle.fVal");
    hist.InitName("ViewCone");
    hist.InitTitle("Incident Angle;\\alpha [\\deg]");
}

void MJSimulation::SetupCommonFileStructure(MWriteRootFile &write) const
{
    // Common run headers
    write.AddContainer("MMcCorsikaRunHeader", "RunHeaders", kFALSE);
    write.AddContainer("MCorsikaRunHeader",   "RunHeaders", kFALSE);
    write.AddContainer("MRawRunHeader",       "RunHeaders");
    write.AddContainer("MGeomCam",            "RunHeaders");
    write.AddContainer("MMcRunHeader",        "RunHeaders");

    // Common events
    write.AddContainer("MCorsikaEvtHeader",   "Events", kFALSE);
    write.AddContainer("MRawEvtHeader",       "Events");
    write.AddContainer("MMcEvt",              "Events");
    write.AddContainer("IncidentAngle",       "Events", kFALSE);
}

Bool_t MJSimulation::Process(const MArgs &args, const MSequence &seq)
{
    /*
    if (!fSequence.IsValid())
    {
        *fLog << err << "ERROR - Sequence invalid!" << endl;
        return kFALSE;
    }
    */

//    if (!HasWritePermission(CombinePath(fPathOut, GetOutFile(seq))))
//        return kFALSE;

    *fLog << inf;
    fLog->Separator(GetDescriptor());

    if (!CheckEnv())
        return kFALSE;

    if (seq.IsValid())
        *fLog << fSequence.GetFileName() << endl;
    else
        *fLog << args.GetNumArguments() << "-files" << endl;
    *fLog << endl;

    MDirIter iter;
    if (seq.IsValid() && seq.GetRuns(iter, MSequence::kCorsika)<=0)
    {
        *fLog << err << "ERROR - Sequence valid but without files." << endl;
        return kFALSE;
    }

    // --------------------------------------------------------------------------------

    // Setup Parlist
    MParList plist;
    plist.AddToList(this); // take care of fDisplay!

    // setup TaskList
    MTaskList tasks;
    plist.AddToList(&tasks);

    // --------------------------------------------------------------------------------

    // FIXME: Allow empty GeomCones!
    MParEnv env0("Reflector");
    MParEnv env1("GeomCones");
    MParEnv env2("MGeomCam");
    env0.SetClassName("MReflector");
    env1.SetClassName("MGeomCam");
    env2.SetClassName("MGeomCam");
    plist.AddToList(&env0);
    plist.AddToList(&env1);
    plist.AddToList(&env2);

    plist.FindCreateObj("MPedPhotCam", "MPedPhotFromExtractorRndm");

    MParSpline shape("PulseShape");
    plist.AddToList(&shape);

    // *** FIXME *** FIXME *** FIXME ***
    plist.FindCreateObj("MMcRunHeader");

    MRawRunHeader header;
    header.SetValidMagicNumber();
    //header.InitFadcType(3);

    switch (fOperationMode)
    {
    case kModeData:
        header.SetRunType(MRawRunHeader::kRTMonteCarlo|MRawRunHeader::kRTData);
        break;

    case kModePed:
        header.SetRunType(MRawRunHeader::kRTMonteCarlo|MRawRunHeader::kRTPedestal);
        header.SetSourceInfo("Pedestal");
        break;

    case kModeCal:
        header.SetRunType(MRawRunHeader::kRTMonteCarlo|MRawRunHeader::kRTCalibration);
        header.SetSourceInfo("Calibration");
        break;

    case kModePointRun:
        header.SetRunType(MRawRunHeader::kRTMonteCarlo|MRawRunHeader::kRTPointRun);
        break;
    }

    // FIXME: Move to MSimPointingPos, MSimCalibrationSignal
    //        Can we use this as input for MSimPointingPos?
    header.SetObservation("On", "MonteCarlo");
    plist.AddToList(&header);
    // ++++++++ FIXME FIXME FIXME +++++++++++++

    /*
    MPedestalCam pedcam;
    pedcam.Init(geomcam.GetNumPixels());
    for (UInt_t i=0; i<geomcam.GetNumPixels(); i++)
        pedcam[i].Set(128./header.GetScale(), 22.5/header.GetScale());
    plist.AddToList(&pedcam);
    */

    // -------------------------------------------------------------------

    MCorsikaRead read;
    read.SetForceMode(fForceMode);

    if (!seq.IsValid())
    {
        for (int i=0; i<args.GetNumArguments(); i++)
            read.AddFile(args.GetArgumentStr(i));
    }
    else
        read.AddFiles(iter);

    MContinue precut("", "PreCut");
    precut.IsInverted();
    precut.SetAllowEmpty();

    MSimMMCS simmmcs;

    MParSpline splinepde("PhotonDetectionEfficiency");
    MParSpline splinemirror("MirrorReflectivity");
    MParSpline splinecones("ConesAngularAcceptance");
    plist.AddToList(&splinepde);
    plist.AddToList(&splinemirror);
    plist.AddToList(&splinecones);

    const TString sin2 = "sin(MCorsikaEvtHeader.fZd)*sin(MCorsikaRunHeader.fZdMin*TMath::DegToRad())";
    const TString cos2 = "cos(MCorsikaEvtHeader.fZd)*cos(MCorsikaRunHeader.fZdMin*TMath::DegToRad())";
    const TString cos  = "cos(MCorsikaEvtHeader.fAz-MCorsikaRunHeader.fAzMin*TMath::DegToRad())";

    const TString form = "acos("+sin2+"*"+cos+"+"+cos2+")*TMath::RadToDeg()";

    MParameterCalc calcangle(form, "CalcIncidentAngle");
    calcangle.SetNameParameter("IncidentAngle");

    MSimAtmosphere simatm;
    MSimAbsorption absapd("SimPhotonDetectionEfficiency");
    MSimAbsorption absmir("SimMirrorReflectivity");
    MSimAbsorption cones("SimConesAngularAcceptance");
    absapd.SetParName("PhotonDetectionEfficiency");
    absmir.SetParName("MirrorReflectivity");
    cones.SetParName("ConesAngularAcceptance");
    cones.SetUseTheta();

    MSimPointingPos pointing;

    MSimReflector reflect;
    reflect.SetNameGeomCam("GeomCones");
//  MSimStarField stars;

    MContinue cont1("MPhotonEvent.GetNumPhotons<1", "ContEmpty1");
    MContinue cont2("MPhotonEvent.GetNumPhotons<1", "ContEmpty2");
    MContinue cont3("MPhotonEvent.GetNumPhotons<2", "ContEmpty3");

    // -------------------------------------------------------------------

    MBinning binse( 120,     1,  1000000, "BinningEnergy",    "log");
    MBinning binsth( 60,   0.9,   900000, "BinningThreshold", "log");
    MBinning binsee( 36,   0.9,   900000, "BinningEnergyEst", "log");
    MBinning binss( 100,     1, 10000000, "BinningSize",      "log");
//    MBinning binsi( 100,  -500,      500, "BinningImpact");
    MBinning binsi(  32,     0,      800, "BinningImpact");
    MBinning binsh( 150,     0,       50, "BinningHeight");
    MBinning binsaz(720,  -360,      360, "BinningAz");
    MBinning binszd( 70,     0,       70, "BinningZd");
    MBinning binsvc( 35,     0,        7, "BinningViewCone");
    MBinning binsel(150,     0,       50, "BinningTotLength");
    MBinning binsew(150,     0,       15, "BinningMedLength");
    MBinning binsd("BinningDistC");
    MBinning binsd0("BinningDist");
    MBinning binstr("BinningTrigPos");

    plist.AddToList(&binsee);
    plist.AddToList(&binse);
    plist.AddToList(&binsth);
    plist.AddToList(&binss);
    plist.AddToList(&binsi);
    plist.AddToList(&binsh);
    plist.AddToList(&binszd);
    plist.AddToList(&binsaz);
    plist.AddToList(&binsvc);
    plist.AddToList(&binsel);
    plist.AddToList(&binsew);
    plist.AddToList(&binstr);
    plist.AddToList(&binsd);
    plist.AddToList(&binsd0);

    MHn mhn1, mhn2, mhn3, mhn4;
    SetupHist(mhn1);
    SetupHist(mhn2);
    SetupHist(mhn3);
    SetupHist(mhn4);

    MH3 mhtp("TriggerPos.fVal-IntendedPulsePos.fVal-PulseShape.GetWidth");
    mhtp.SetName("TrigPos");
    mhtp.SetTitle("Trigger position w.r.t. the first photon hitting a detector");

    MH3 mhew("MPhotonStatistics.fLength");
    mhew.SetName("TotLength");
    mhew.SetTitle("Time between first and last photon hitting a detector;L [ns]");

    MH3 mhed("MPhotonStatistics.fTimeMedDev");
    mhed.SetName("MedLength");
    mhed.SetTitle("Median deviation (1\\sigma);L [ns]");

    MFillH fillh1(&mhn1, "", "FillCorsika");
    MFillH fillh2(&mhn2, "", "FillH2");
    MFillH fillh3(&mhn3, "", "FillH3");
    MFillH fillh4(&mhn4, "", "FillH4");
    MFillH filltp(&mhtp, "", "FillTriggerPos");
    MFillH fillew(&mhew, "", "FillEvtWidth");
    MFillH filled(&mhed, "", "FillMedDev");
    fillh1.SetNameTab("Corsika",    "Distribution as simulated by Corsika");
    fillh2.SetNameTab("Detectable", "Distribution of events hit the detector");
    fillh3.SetNameTab("Triggered",  "Distribution of triggered events");
    fillh4.SetNameTab("Cleaned",    "Distribution after cleaning and cuts");
    filltp.SetNameTab("TrigPos",    "TriggerPosition w.r.t the first photon");
    fillew.SetNameTab("EvtWidth",   "Time between first and last photon hitting a detector");
    filled.SetNameTab("MedDev",     "Time between first and last photon hitting a detector");

    MHPhotonEvent planeG(1, "HPhotonEventGround");     // Get from MaxImpact
    MHPhotonEvent plane0(2, "HMirrorPlane0");     // Get from MReflector
    //MHPhotonEvent plane1(2, "HMirrorPlane1");
    MHPhotonEvent plane2(2, "HMirrorPlane2");
    MHPhotonEvent plane3(2, "HMirrorPlane3");
    MHPhotonEvent plane4(2, "HMirrorPlane4");
    MHPhotonEvent planeF1(3, "HPhotonEventCamera");   // Get from MGeomCam
    MHPhotonEvent planeF2(header.IsPointRun()?4:3, "HPhotonEventCones"); // Get from MGeomCam

    plist.AddToList(&planeG);
    plist.AddToList(&plane0);
    plist.AddToList(&plane2);
    plist.AddToList(&plane3);
    plist.AddToList(&plane4);
    plist.AddToList(&planeF1);
    plist.AddToList(&planeF2);;

    //MHPSF psf;

    MFillH fillG(&planeG,   "MPhotonEvent", "FillGround");
    MFillH fill0(&plane0,   "MirrorPlane0", "FillReflector");
    //MFillH fill1(&plane1,   "MirrorPlane1", "FillCamShadow");
    MFillH fill2(&plane2,   "MirrorPlane2", "FillCandidates");
    MFillH fill3(&plane3,   "MirrorPlane3", "FillReflected");
    MFillH fill4(&plane4,   "MirrorPlane4", "FillFocal");
    MFillH fillF1(&planeF1, "MPhotonEvent", "FillCamera");
    MFillH fillF2(&planeF2, "MPhotonEvent", "FillCones");
    //MFillH fillP(&psf,      "MPhotonEvent", "FillPSF");

    fillG.SetNameTab("Ground",     "Photon distribution at ground");
    fill0.SetNameTab("Reflector",  "Photon distribution at reflector plane w/o camera shadow");
    //fill1.SetNameTab("CamShadow",  "Photon distribution at reflector plane w/ camera shadow");
    fill2.SetNameTab("Candidates", "'Can hit' photon distribution at reflector plane w/ camera shadow");
    fill3.SetNameTab("Reflected",  "Photon distribution after reflector projected to reflector plane");
    fill4.SetNameTab("Focal",      "Photon distribution at focal plane");
    fillF1.SetNameTab("Camera",    "Photon distribution which hit the detector");
    fillF2.SetNameTab("Cones",     "Photon distribution after cones");
    //fillP.SetNameTab("PSF",        "Photon distribution after cones for all mirrors");

    // -------------------------------------------------------------------

    const char *fmt = Form("s/cer([0-9]+)/%%s\\/00$1_%c_MonteCarlo.root/", header.GetRunTypeChar());

    // FIXME: Pedestal and Calibration runs should get P and C
    const TString rule1(Form("s/cer([0-9]+)/%s\\/00$1_R_MonteCarlo.root/", Esc(fPathOut).Data()));
    const TString rule2(Form("s/cer([0-9]+)/%s\\/00$1_Y_MonteCarlo.root/", Esc(fPathOut).Data()));
    const TString rule3(Form(fmt, Esc(fPathOut).Data()));

    MWriteRootFile write3a( 2, rule3, fOverwrite?"RECREATE":"NEW", "Camera file");
    MWriteRootFile write3b( 2, rule3, fOverwrite?"RECREATE":"NEW", "Camera file");
    MWriteRootFile write2a( 2, rule2, fOverwrite?"RECREATE":"NEW", "Signal file");
    MWriteRootFile write2b( 2, rule2, fOverwrite?"RECREATE":"NEW", "Signal file");
    MWriteRootFile write1a( 2, rule1, fOverwrite?"RECREATE":"NEW", "Reflector file");
    MWriteRootFile write1b( 2, rule1, fOverwrite?"RECREATE":"NEW", "Reflector file");

    write1a.SetName("WriteRefData");
    write1b.SetName("WriteRefMC");
    write2a.SetName("WriteSigData");
    write2b.SetName("WriteSigMC");
    write3a.SetName("WriteCamData");
    write3b.SetName("WriteCamMC");

    SetupCommonFileStructure(write1a);
    SetupCommonFileStructure(write2a);
    SetupCommonFileStructure(write3a);

    // R: Dedicated file structureedicated events
    write1a.AddContainer("MPhotonEvent", "Events");

    // I: Dedicated file structureedicated events
    write2a.AddContainer("MPedPhotFromExtractorRndm", "RunHeaders"); // FIXME: Needed for the signal files to be display in MARS
    write2a.AddContainer("MSignalCam", "Events");

    // D: Dedicated file structureedicated events
    write3a.AddContainer("ElectronicNoise", "RunHeaders");
    write3a.AddContainer("MRawEvtData", "Events");

    // Basic MC data
    write1b.AddContainer("MMcEvtBasic", "OriginalMC");
    write2b.AddContainer("MMcEvtBasic", "OriginalMC");
    write3b.AddContainer("MMcEvtBasic", "OriginalMC");

    // -------------------------------------------------------------------

    MGeomApply apply;

    MSimPSF simpsf;

    MSimGeomCam simgeom;
    simgeom.SetNameGeomCam("GeomCones");

    MSimCalibrationSignal simcal;
    simcal.SetNameGeomCam("GeomCones");

    // Sky Quality Meter: 21.82M = 2.02e-4cd/m^2
    // 1cd = 12.566 lm / 4pi sr

    // FIXME: Simulate photons before cones and QE!

    MSimRandomPhotons  simnsb;
    simnsb.SetFreq(5.8, 0.004);  // ph/m^2/nm/sr/ns NSB, 4MHz dark count rate
    simnsb.SetNameGeomCam("GeomCones");

    // FIXME: How do we handle star-light? We need the rate but we also
    //        need to process it through the mirror!

    MSimAPD simapd;
    simapd.SetNameGeomCam("GeomCones");

    MSimExcessNoise   simexcnoise;
    MSimBundlePhotons simsum;
    MSimSignalCam     simsignal;
    MSimCamera        simcam;
    MSimTrigger       simtrig;
    MSimReadout       simdaq;

    MContinue conttrig("TriggerPos.fVal<0", "ContTrigger");

    MParameterD pulpos("IntendedPulsePos");
    // FIXME: Set a default which could be 1/3 of the digitization window
    //    pulpos.SetVal(40);  // [ns]
    plist.AddToList(&pulpos);

    MParameterD trigger("TriggerPos");
    trigger.SetVal(0);
    plist.AddToList(&trigger);

    // -------------------------------------------------------------------

    // Remove isolated pixels
    MImgCleanStd clean(0, 0);
    clean.SetCleanLvl0(0); // The level above which isolated pixels are kept
    clean.SetCleanRings(0);
    clean.SetMethod(MImgCleanStd::kAbsolute);

    //clean.SetNamePedPhotCam("MPedPhotFromExtractorRndm");

    MHillasCalc hcalc;
    hcalc.Disable(MHillasCalc::kCalcConc);

    MHCamEvent evt0a(/*10*/3, "Signal",    "Average signal (absolute);;S [ph]");
    MHCamEvent evt0c(/*10*/3, "MaxSignal", "Maximum signal (absolute);;S [ph]");
    MHCamEvent evt0d(/*11*/8, "ArrTm",     "Time after first photon;;T [ns]");
    evt0a.SetErrorSpread(kFALSE);
    evt0c.SetCollectMax();

    MContinue cut("", "Cut");

    MFillH fillx0a(&evt0a,             "MSignalCam",      "FillAvgSignal");
    MFillH fillx0c(&evt0c,             "MSignalCam",      "FillMaxSignal");
    MFillH fillx0d(&evt0d,             "MSignalCam",      "FillArrTm");
    MFillH fillx1("MHHillas",          "MHillas",         "FillHillas");
    MFillH fillx3("MHHillasSrc",       "MHillasSrc",      "FillHillasSrc");
    MFillH fillx4("MHNewImagePar",     "MNewImagePar",    "FillNewImagePar");
    MFillH fillth("MHThreshold",       "",                "FillThreshold");
    MFillH fillca("MHCollectionArea",  "",                "FillTrigArea");

    fillth.SetNameTab("Threshold");
    fillca.SetNameTab("TrigArea");

    // -------------------------------------------------------------------

    if (header.IsDataRun())
    {
        tasks.AddToList(&read);
        tasks.AddToList(&precut);
        tasks.AddToList(&pointing);
        tasks.AddToList(&simmmcs);
        if (!fPathOut.IsNull() && !HasNullOut())
        {
            tasks.AddToList(&write1b);
            tasks.AddToList(&write2b);
            if (fCamera)
                tasks.AddToList(&write3b);
        }
        //    if (header.IsPointRun())
        //        tasks.AddToList(&stars);
        if (1)
            tasks.AddToList(&simatm); // Here because before fillh1
        tasks.AddToList(&calcangle);
        tasks.AddToList(&fillh1);
        tasks.AddToList(&fillG);
        if (!header.IsPointRun())
        {
            tasks.AddToList(&absapd);
            tasks.AddToList(&absmir);
            if (0)
                tasks.AddToList(&simatm); // FASTER?
        }
        tasks.AddToList(&reflect);
        if (!header.IsPointRun())
        {
            tasks.AddToList(&fill0);
            //tasks.AddToList(&fill1);
            tasks.AddToList(&fill2);
            tasks.AddToList(&fill3);
            tasks.AddToList(&fill4);
            tasks.AddToList(&fillF1);
        }
        tasks.AddToList(&cones);
        //if (header.IsPointRun())
        //    tasks.AddToList(&fillP);
        tasks.AddToList(&cont1);
    }
    // -------------------------------
    tasks.AddToList(&apply);
    if (header.IsDataRun())
    {
        tasks.AddToList(&simpsf);
        tasks.AddToList(&simgeom);
        tasks.AddToList(&cont2);
        if (!header.IsPointRun())
        {
            tasks.AddToList(&fillF2);
            tasks.AddToList(&fillh2);
        }
        tasks.AddToList(&cont3);
    }
    if (fCamera)
    {
        if (header.IsPedestalRun() || header.IsCalibrationRun())
            tasks.AddToList(&simcal);
        tasks.AddToList(&simnsb);
        tasks.AddToList(&simapd);
        tasks.AddToList(&simexcnoise);
    }
    tasks.AddToList(&simsum);
    if (fCamera)
    {
        tasks.AddToList(&simcam);
        if (header.IsDataRun() || fForceTrigger)
            tasks.AddToList(&simtrig);
        tasks.AddToList(&conttrig);
        tasks.AddToList(&simdaq);
    }
    tasks.AddToList(&simsignal);  // What do we do if signal<0?
    if (!fPathOut.IsNull() && !HasNullOut())
    {
        tasks.AddToList(&write1a);
        if (!header.IsPedestalRun())
            tasks.AddToList(&write2a);
        if (fCamera)
            tasks.AddToList(&write3a);
    }
    // -------------------------------
    if (fCamera)
    {
        // FIXME: MHCollectionArea Trigger Area!
        if (header.IsDataRun())
            tasks.AddToList(&fillh3);
        tasks.AddToList(&filltp);
    }
    if (header.IsDataRun())
    {
        tasks.AddToList(&fillew);
        tasks.AddToList(&filled);
    }
    if (!header.IsPedestalRun())
    {
        tasks.AddToList(&fillx0a);
        tasks.AddToList(&fillx0c);
        if (!header.IsCalibrationRun())
            tasks.AddToList(&clean);
        tasks.AddToList(&hcalc);
        tasks.AddToList(&cut);
        tasks.AddToList(&fillx0d);
        tasks.AddToList(&fillx1);
        //tasks.AddToList(&fillx2);
        tasks.AddToList(&fillx3);
        tasks.AddToList(&fillx4);
        //tasks.AddToList(&fillx5);
    }
    if (header.IsDataRun())
    {
        tasks.AddToList(&fillh4);
        tasks.AddToList(&fillth);
        tasks.AddToList(&fillca);
    }
    //-------------------------------------------

    tasks.SetAccelerator(MTask::kAccDontReset|MTask::kAccDontTime);

    // Create and setup the eventloop
    MEvtLoop evtloop(fName);
    evtloop.SetParList(&plist);
    evtloop.SetDisplay(fDisplay);
    evtloop.SetLogStream(&gLog);
    if (!SetupEnv(evtloop))
        return kFALSE;

    // FIXME: If pedestal file given use the values from this file
    //-------------------------------------------

    MGeomCam *cam = static_cast<MGeomCam*>(env2.GetCont());

    if (binstr.IsDefault())
        binstr.SetEdgesLin(150, -shape.GetWidth(),
                           header.GetFreqSampling()/1000.*header.GetNumSamples()+shape.GetWidth());

    if (binsd.IsDefault() && cam)
        binsd.SetEdgesLin(100, 0, cam->GetMaxRadius()*cam->GetConvMm2Deg());

    if (binsd0.IsDefault() && cam)
        binsd0.SetEdgesLin(100, 0, cam->GetMaxRadius()*cam->GetConvMm2Deg());


    header.Print();

    // FIXME: Display acceptance curves!

    if (fDisplay)
    {
        TCanvas &c = fDisplay->AddTab("Info");
        c.Divide(2,2);

        c.cd(1);
        gPad->SetBorderMode(0);
        gPad->SetFrameBorderMode(0);
        gPad->SetGridx();
        gPad->SetGridy();
        gROOT->SetSelectedPad(0);
        shape.DrawClone()->SetBit(kCanDelete);

        if (env0.GetCont() && (header.IsDataRun() || header.IsPointRun()))
        {
            c.cd(3);
            gPad->SetBorderMode(0);
            gPad->SetFrameBorderMode(0);
            gPad->SetGridx();
            gPad->SetGridy();
            gROOT->SetSelectedPad(0);
            static_cast<MReflector*>(env0.GetCont())->DrawClone("line")->SetBit(kCanDelete);
        }

        if (fCamera)
        {
            if (env1.GetCont())
            {
                c.cd(2);
                gPad->SetBorderMode(0);
                gPad->SetFrameBorderMode(0);
                gPad->SetGridx();
                gPad->SetGridy();
                gROOT->SetSelectedPad(0);
                MHCamera *c = new MHCamera(static_cast<MGeomCam&>(*env1.GetCont()));
                c->SetStats(kFALSE);
                c->SetBit(MHCamera::kNoLegend);
                c->SetBit(kCanDelete);
                c->Draw();
            }

            if (cam)
            {
                c.cd(4);
                gPad->SetBorderMode(0);
                gPad->SetFrameBorderMode(0);
                gPad->SetGridx();
                gPad->SetGridy();
                gROOT->SetSelectedPad(0);
                MHCamera *c = new MHCamera(*cam);
                c->SetStats(kFALSE);
                c->SetBit(MHCamera::kNoLegend);
                c->SetBit(kCanDelete);
                c->Draw();
            }
        }

        TCanvas &d = fDisplay->AddTab("Info2");
        d.Divide(2,2);

        d.cd(1);
        gPad->SetBorderMode(0);
        gPad->SetFrameBorderMode(0);
        gPad->SetGridx();
        gPad->SetGridy();
        gROOT->SetSelectedPad(0);
        splinepde.DrawClone()->SetBit(kCanDelete);

        d.cd(2);
        gPad->SetBorderMode(0);
        gPad->SetFrameBorderMode(0);
        gPad->SetGridx();
        gPad->SetGridy();
        gROOT->SetSelectedPad(0);
        splinemirror.DrawClone()->SetBit(kCanDelete);

        d.cd(3);
        gPad->SetBorderMode(0);
        gPad->SetFrameBorderMode(0);
        gPad->SetGridx();
        gPad->SetGridy();
        gROOT->SetSelectedPad(0);
        splinecones.DrawClone()->SetBit(kCanDelete);

        d.cd(4);
        gPad->SetBorderMode(0);
        gPad->SetFrameBorderMode(0);
        gPad->SetGridx();
        gPad->SetGridy();
        gROOT->SetSelectedPad(0);
        MParSpline *all = (MParSpline*)splinepde.DrawClone();
        //all->SetTitle("Combined acceptance");
        all->SetBit(kCanDelete);
        all->Multiply(*splinemirror.GetSpline());
    }

    //-------------------------------------------

    // Execute first analysis
    if (!evtloop.Eventloop(fMaxEvents))
    {
        gLog << err << GetDescriptor() << ": Failed." << endl;
        return kFALSE;
    }

    //-------------------------------------------
    // FIXME: Display Spline     in tab
    // FIXME: Display Reflector  in tab
    // FIXME: Display Routing(?) in tab
    // FIXME: Display Camera(s)  in tab
    //-------------------------------------------

    if (!WriteResult(plist, seq))
        return kFALSE;

    *fLog << all << GetDescriptor() << ": Done." << endl << endl << endl;;

    return kTRUE;
}
