/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Harald Kornmayer 1/2001 (harald@mppmu.mpg.de)
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MGFadcDisp.h"

#include <stdlib.h>               // atoi
#include <iostream.h>             // cout for debugging

#include <TList.h>                // TList
#include <TCanvas.h>              // TCanvas
#include <TRootEmbeddedCanvas.h>  // TRootEmbeddedCanvas

#include <TGTab.h>                // TGTab
#include <TGLabel.h>              // TGLabel
#include <TGButton.h>             // TGPictureButton
#include <TGSlider.h>             // TGVSlider
#include <TGMsgBox.h>             // TGMsgBox
#include <TGListBox.h>            // TGListBox
#include <TGTextEntry.h>          // TGTextEntry

#include <TG3DLine.h>             // TGHorizontal3DLine
                                  // use TGSplitter instead for root<3.00
#include "MParList.h"
#include "MReadTree.h"
#include "MRawEvtData.h"
#include "MRawEvtPixelIter.h"

ClassImp(MGFadcDisp);

enum MGFadcDispCommand
{
    M_PIXELLIST = 4201,
    M_PREVEVT,
    M_NEXTEVT,
    M_EVTNUMBER,

    M_PREVPIXEL,
    M_NEXTPIXEL,

    M_PRINT,
    M_CLOSE
}; 

void MGFadcDisp::AddTopFramePart1(TGVerticalFrame *frame,
                                  const char *filename,
                                  const char *treename)
{
    //
    //  --- the top1 part of the window ---
    //
    TGHorizontalFrame *top1 = new TGHorizontalFrame(frame, 300, 100);
    fList->Add(top1);

    //
    // create gui elements
    //
    TGLabel *lfile = new TGLabel(top1, new TGString("File:"));
    TGLabel *file  = new TGLabel(top1, new TGString(filename));
    TGLabel *ltree = new TGLabel(top1, new TGString("Tree:"));
    TGLabel *tree  = new TGLabel(top1, new TGString(treename));

    fList->Add(lfile);
    fList->Add(file);
    fList->Add(ltree);
    fList->Add(tree);

    //
    // layout and add gui elements in/to frame
    //
    TGLayoutHints *laystd = new TGLayoutHints(kLHintsLeft, 10, 10, 10, 10);
    fList->Add(laystd);

    top1->AddFrame(lfile, laystd);
    top1->AddFrame(file,  laystd);
    top1->AddFrame(ltree, laystd);
    top1->AddFrame(tree,  laystd);

    //
    // layout and add frame
    //
    TGLayoutHints *laytop1 = new TGLayoutHints(kLHintsTop);
    fList->Add(laytop1);

    frame->AddFrame(top1, laytop1);
}

void MGFadcDisp::AddTopFramePart2(TGVerticalFrame *frame)
{
    //
    // --- the top2 part of the window ---
    //
    TGHorizontalFrame *top2 = new TGHorizontalFrame(frame, 300, 100);
    fList->Add(top2);

    //
    // Create the gui elements
    //
    TGTextButton *prevevt = new TGTextButton(top2, "<< Previous Event", M_PREVEVT);
    prevevt->Associate(this);

    TGLabel *evtnr = new TGLabel(top2, new TGString("Event: "));

    fTxtEvtNr = new TGTextEntry(top2, new TGTextBuffer(100), M_EVTNUMBER);
    fTxtEvtNr->Resize(60, fTxtEvtNr->GetDefaultHeight());
    fTxtEvtNr->Associate(this);

    char wortdummy[100];
    sprintf(wortdummy, "out of %d Events", fReadTree->GetEntries());
    TGLabel *totnr = new TGLabel(top2, new TGString(wortdummy));

    TGTextButton *nextevt = new TGTextButton (top2, "Next Event >>", M_NEXTEVT);
    nextevt->Associate(this);

    //
    // Add gui elements to 'atotodel'
    //
    fList->Add(prevevt);
    fList->Add(evtnr);
    fList->Add(fTxtEvtNr);
    fList->Add(totnr);
    fList->Add(nextevt);

    //
    // add the gui elements to the frame
    //
    TGLayoutHints *laystd    = new TGLayoutHints(kLHintsLeft, 10, 10, 10, 10);
    TGLayoutHints *laytentry = new TGLayoutHints(kLHintsNormal, 5, 5, 5, 5);

    fList->Add(laystd);
    fList->Add(laytentry);

    top2->AddFrame(prevevt,   laystd);
    top2->AddFrame(evtnr,     laystd);
    top2->AddFrame(fTxtEvtNr, laytentry);
    top2->AddFrame(totnr,     laystd);
    top2->AddFrame(nextevt,   laystd);

    frame->AddFrame(top2, new TGLayoutHints(kLHintsCenterX));
}

void MGFadcDisp::AddMidFrame(TGHorizontalFrame *frame)
{
    //
    // create tab control
    //
    TGTab *tabs = new TGTab(frame, 300, 300);

    //
    // Create Tab1
    //
    TGCompositeFrame *tab1 = tabs->AddTab("PixelList");
    tab1->ChangeOptions(kHorizontalFrame);

    //
    // Create first gui element for tab1
    //
    fPixelList = new TGListBox(tab1, M_PIXELLIST);
    fPixelList->Associate(this);
    fPixelList->Resize(80, 230);

    TGLayoutHints *layplist = new TGLayoutHints(kLHintsExpandY|kLHintsLeft, 5, 5, 5, 5);
    tab1->AddFrame(fPixelList, layplist);

    //
    // Crete second gui elemet for tab1 (TGVertical Frame)
    //
    TGVerticalFrame *mid1 = new TGVerticalFrame(tab1, 300, 100);

    //
    // Create gui elements for vertical frame
    //
    TGTextButton *prevpix = new TGTextButton(mid1, "<< Prev Pixel", M_PREVPIXEL);
    TGTextButton *nextpix = new TGTextButton(mid1, "Next Pixel >>", M_NEXTPIXEL);
    prevpix->Associate(this);
    nextpix->Associate(this);

    TGVSlider *slider = new TGVSlider(mid1, 200, kSlider1|kScaleBoth);
    slider->Associate(this);
    slider->SetRange(0, 576);

    //
    // Layout gui elements
    //
    TGLayoutHints *laybut    = new TGLayoutHints(kLHintsRight);
    TGLayoutHints *layslider = new TGLayoutHints(kLHintsCenterX|kLHintsExpandY);

    mid1->AddFrame(prevpix, laybut);
    mid1->AddFrame(slider,  layslider);
    mid1->AddFrame(nextpix, laybut);

    TGLayoutHints *laytab    = new TGLayoutHints(kLHintsRight|kLHintsExpandY, 5, 5, 5, 5);
    tab1->AddFrame(mid1, laytab);

    TGLayoutHints *laytabs   = new TGLayoutHints(kLHintsNormal|kLHintsExpandY, 10, 10, 10, 10);
    frame->AddFrame(tabs, laytabs);

    //
    // Create second part of frame
    // 
    TGTab *tabdisp = new TGTab(frame, 300, 300);

    TGCompositeFrame *tab2 = tabdisp->AddTab("Digital Scope");

    TRootEmbeddedCanvas *canvas = new TRootEmbeddedCanvas("Digi Scope", tab2, 400, 400);

    TGLayoutHints *laycanvas = new TGLayoutHints(kLHintsCenterX|kLHintsCenterY|kLHintsExpandX|kLHintsExpandY);
    tab2->AddFrame(canvas, laycanvas);

    fCanvas = canvas->GetCanvas();

    //
    // Add second part to frame
    //
    TGLayoutHints *laydisp = new TGLayoutHints(kLHintsNormal|kLHintsExpandY|kLHintsExpandX, 10, 10, 10, 10);
    frame->AddFrame(tabdisp, laydisp);

    //
    // Now add all gui elements to 'autodel'-list
    //
    fList->Add(tabdisp);
    fList->Add(canvas);
    fList->Add(laycanvas);
    fList->Add(laydisp);
    fList->Add(fPixelList);
    fList->Add(layplist);
    fList->Add(mid1);
    fList->Add(prevpix);
    fList->Add(laybut);
    fList->Add(slider);
    fList->Add(layslider);
    fList->Add(nextpix);
    fList->Add(laytab);
    fList->Add(laytabs);

}

void MGFadcDisp::AddLowFrame(TGHorizontalFrame *frame)
{
    TGTextButton *but1 = new TGTextButton(frame, "Print", M_PRINT);
    TGTextButton *but2 = new TGTextButton(frame, "Close", M_CLOSE);

    but1->Associate(this);
    but2->Associate(this);

    fList->Add(but1);
    fList->Add(but2);

    TGLayoutHints *laybut = new TGLayoutHints(kLHintsLeft, 10, 10, 10, 10);
    fList->Add(laybut);

    frame->AddFrame(but1, laybut);
    frame->AddFrame(but2, laybut);
}

void MGFadcDisp::CreateGui(const char *filename, const char *treename)
{
    //
    // Create the frame elements and add gui elements to it
    //
    TGVerticalFrame *frametop = new TGVerticalFrame(this, 300, 100);
    AddTopFramePart1(frametop, filename, treename);
    AddTopFramePart2(frametop);

    TGHorizontal3DLine *line1   = new TGHorizontal3DLine(this);

    TGHorizontalFrame *framemid = new TGHorizontalFrame(this, 300, 100);
    AddMidFrame(framemid);

    TGHorizontal3DLine *line2   = new TGHorizontal3DLine(this);

    TGHorizontalFrame *framelow = new TGHorizontalFrame(this, 300, 100);
    AddLowFrame(framelow);

    //
    // add frame elements to 'autodel'
    //
    fList->Add(frametop);
    fList->Add(line1);
    fList->Add(framemid);
    fList->Add(line2);
    fList->Add(framelow);

    //
    // Layout frame elements and add elements to frame
    //
    TGLayoutHints *laytop  = new TGLayoutHints(kLHintsTop|kLHintsCenterX);
    TGLayoutHints *layline = new TGLayoutHints(kLHintsTop|kLHintsExpandX);
    TGLayoutHints *laymid  = new TGLayoutHints(kLHintsExpandY|kLHintsExpandX);
    TGLayoutHints *laylow  = new TGLayoutHints(kLHintsTop);

    fList->Add(laytop);
    fList->Add(layline);
    fList->Add(laymid);
    fList->Add(laylow);

    AddFrame(frametop, laytop);
    AddFrame(line1,    layline);
    AddFrame(framemid, laymid);
    AddFrame(line2,    layline);
    AddFrame(framelow, laylow);
}

void MGFadcDisp::SetupFileAccess(const char *filename, const char *treename)
{
    //
    // Create the file access elements
    //
    fEvtData = new MRawEvtData();

    pList = new MParList();
    pList->AddToList(fEvtData);

    fReadTree = new MReadTree(treename, filename);
    fReadTree->PreProcess(pList);
    fReadTree->GetEvent();
}

MGFadcDisp::MGFadcDisp(char *filename, char *treename,
                       const TGWindow *p, const TGWindow *main,
                       UInt_t w, UInt_t h)
    : TGTransientFrame(p, main, w, h)
{

    //
    //  create an autodelete-list for the gui elements
    //
    fList = new TList;
    fList->SetOwner();

    SetupFileAccess(filename, treename);
    CreateGui(filename, treename);

    //
    //   Map the window, set up the layout, etc.
    //
    SetWMSizeHints(450, 400, 1000, 1000, 10, 10 );      // set the smallest and biggest size of the Main frame

    MapSubwindows();

    Layout();

    SetWindowName("FadcDisplay");
    SetIconName("FadcDisp");

    MapWindow();

    CreatePixelList();
    UpdateEventCounter();
}


MGFadcDisp::~MGFadcDisp()
{
    //
    // close the file
    //
    fReadTree->PostProcess();

    delete fEvtData;
    delete pList;

    delete fReadTree;

    //
    //   destruct the graphical members
    //
    delete fList;
}


void MGFadcDisp::CloseWindow()
{
   // Got close message for this MainFrame. Calls parent CloseWindow()
   // (which destroys the window) and terminate the application.
   // The close message is generated by the window manager when its close
   // window menu item is selected.

    delete this;
}
       

void MGFadcDisp::CreatePixelList(Int_t lastsel)
{
    //
    //   after a new event is read in one has to update
    //   the list of pixels in the fPixelList (TGListBox)
    //

    //
    //   put the selection of the last event in memory
    //
    MRawEvtPixelIter pixel(fEvtData);
    while (pixel.Next())
    {
        char wortdummy[100];
        sprintf(wortdummy, "%d", pixel.GetPixelId());
        fPixelList->AddEntry(wortdummy, pixel.GetPixelId());
    }

    fPixelList->MapSubwindows();
    fPixelList->Layout();

    //
    // check if the pixel from last event also occurs in this event
    //
    fCanvas->Clear();
    fCanvas->cd();

    if (lastsel<0 || !pixel.Jump(lastsel))
    {
        pixel.Reset();
        lastsel=pixel.GetPixelId();

    }

    char wortdummy[100];
    sprintf(wortdummy, "GRAPH%d", lastsel);
    fEvtData->Draw(wortdummy);
    fPixelList->Select(lastsel, kTRUE);

    fCanvas->Modified();
    fCanvas->Update();
}

void MGFadcDisp::UpdateEventCounter() 
{
    //     Update the event counter

    char wortdummy[256];

    sprintf(wortdummy, "%d", fReadTree->GetEventNum());

    fTxtEvtNr->SetText(wortdummy);
}

void MGFadcDisp::ReadinEvent(UInt_t iEvt)
{
    Int_t buttons = 4;
    Int_t retval  = 0;

    //  first check if the new event is in the range of possible events
  
    if (iEvt >= fReadTree->GetEntries())
    {
        new TGMsgBox(fClient->GetRoot(), this,
                     "WARNING!",
                     "The event number is out of range!!!",
                     kMBIconExclamation, buttons, &retval);
    }
    else
    {
        const Int_t lastsel = fPixelList->GetSelected();

        fPixelList->RemoveEntries(0, fEvtData->GetNumPixels());

        fReadTree->SetEventNum(iEvt);
        fReadTree->GetEvent();

        CreatePixelList(lastsel);
    }

    UpdateEventCounter();
}

void MGFadcDisp::DisplayPix(UInt_t i)
{
    char wortdummy[256];

    sprintf(wortdummy, "GRAPH%d", i);

    fCanvas->Clear();
    fCanvas->cd();

    fEvtData->Draw(wortdummy);

    fCanvas->Modified();
    fCanvas->Update();

    //
    // FIXME: too complicated!
    //
    fPixelList->RemoveEntries(0, fEvtData->GetNumPixels());
    CreatePixelList(i);
}

Bool_t MGFadcDisp::ProcessMessage(Long_t msg, Long_t parm1, Long_t parm2) 
{
    //------------------------------------------------------------------
    //
    //    ProcessMessage(Long_t msg, Long_t parm1, Long_t parm2)
    //
    // Processes information from all GUI items.
    // Selecting an item usually generates an event with 4 parameters.
    // The first two are packed into msg (first and second bytes).
    // The other two are parm1 and parm2.
    //
    //------------------------------------------------------------------

    switch(GET_MSG(msg))
    {
    case kC_COMMAND:
        switch(GET_SUBMSG(msg))
        {
        case kCM_BUTTON:
            switch (parm1)
            {
            case M_PREVEVT:
                ReadinEvent(fReadTree->GetEventNum()-1);
                return kTRUE;

            case M_NEXTEVT:
                ReadinEvent(fReadTree->GetEventNum()+1);
                return kTRUE;

            case M_PREVPIXEL:
                DisplayPix(fPixelList->GetSelected()-1);
                return kTRUE;

            case M_NEXTPIXEL:
                DisplayPix(fPixelList->GetSelected()+1);
                return kTRUE;

            case M_PRINT:
                cout << "Sorry, not yet implemented!" << endl;
                return kTRUE;

            case M_CLOSE:
                CloseWindow();
                return kTRUE;
            }
            return kTRUE;

        case kCM_LISTBOX:
            if (parm1 != M_PIXELLIST)
                return kTRUE;

            DisplayPix(fPixelList->GetSelected());
            return kTRUE;
        }
        return kTRUE;

    case kC_TEXTENTRY:
        if (GET_SUBMSG(msg) == kTE_ENTER)
            ReadinEvent(atoi(fTxtEvtNr->GetText()));

        return kTRUE;
    }

    return kTRUE;
}
