#include "MCamDisplay.h"

#include <math.h>

#include <TClonesArray.h>
#include <TCanvas.h>
#include <TStyle.h>
#include <TBox.h>
#include <TText.h>

#include "MHexagon.h"
#include "MGeomCam.h"

#include "MCerPhotEvt.h"

#define kItemsLegend 25

ClassImp(MCamDisplay);

// ------------------------------------------------------------------------
//
//  default constructor
//
MCamDisplay::MCamDisplay(MGeomCam *geom) : fAutoScale(kTRUE)
{
    //
    //    set the color palette
    //
    gStyle->SetPalette(1, 0);

    //
    //  create the hexagons of the display
    //
    fNumPixels = geom->GetNumPixels();
    fRange     = (Int_t)geom->GetMaxRadius();

    fPixels    = new TClonesArray("MHexagon", fNumPixels);

    //
    // Construct all hexagons. Use new-operator with placement
    //
    for (UInt_t i=0; i<fNumPixels; i++)
        new ((*fPixels)[i]) MHexagon((*geom)[i]);

    //
    // set the range to default
    //
    fMinPhe = -2.;
    fMaxPhe = 50.;

    //
    // set up the Legend
    //
    fLegend  = new TClonesArray("TBox",  kItemsLegend);
    fLegText = new TClonesArray("TText", kItemsLegend);

    for (Int_t i = 0; i<kItemsLegend; i++)
    {
        //const Int_t y = il*40;

        //
        // Construct gui elements of legend. Use new-operator with placement
        //
        //TBox  *newbox = new ((*fLegend)[il])  TBox (650, y-500, 700, y-460);
        //TText *newtxt = new ((*fLegText)[il]) TText(720, y-480, text);

        TBox  *newbox = new ((*fLegend)[i])  TBox;
        TText *newtxt = new ((*fLegText)[i]) TText;

        const Float_t lvl = 50. / kItemsLegend * i;

        newbox->SetFillColor(GetColor(lvl));

        newtxt->SetTextSize(0.025);
        newtxt->SetTextAlign(12);
    }
}

// ------------------------------------------------------------------------
//
//
MCamDisplay::~MCamDisplay()
{
    delete fPixels;
    delete fLegend;
    delete fLegText;
}

// ------------------------------------------------------------------------
//
//
void MCamDisplay::Draw(Option_t *option)
{
    //
    // if no canvas is yet existing to draw into, create a new one
    //
    if (!gPad)
        fDrawingPad = new TCanvas("CamDisplay", "Magic Camera Display", 0, 0, 750, 600);
    else
    {
        gPad->Clear();
        fDrawingPad = gPad;
    }

    //
    // FIXME: This class should be a TPad, so that it is informed if the
    // ratio between height and size changes!
    //
    const Float_t ratio = (Float_t)gPad->GetWw()/gPad->GetWh();

    //
    // The recommended size ratio is 5:4
    //
    Int_t x, y;

    if (ratio > 5./4.)
    {
        x = (UInt_t)((ratio-1)*3*fRange);
        y = fRange;
    }
    else
    {
        x = 3*fRange/2;
        y = (UInt_t)(fRange/ratio);
    }

    //
    // Setup the correct environment
    //
    gStyle->SetPalette(1, 0);

    gPad->Range(-fRange, -y, x, y);
    gPad->SetFillColor(22);

    //
    // Draw all pixels of the camera
    //  (means apend all pixelobjects to the current pad)
    //
    for (UInt_t i=0; i<fNumPixels; i++)
        (*this)[i].Draw();

    //
    // draw legend
    //
    const Float_t H = 0.9*fRange;
    const Float_t h = 2./kItemsLegend;

    const Float_t w = fRange/sqrt(fNumPixels);

    for (Int_t i=0; i<kItemsLegend; i++)
    {
        TBox *box = GetBox(i);
        box->SetX1(fRange);
        box->SetX2(fRange+w);
        box->SetY1(H*( i   *h - 1.));
        box->SetY2(H*((i+1)*h - 1.));
        box->Draw();

        TText *txt = GetText(i);
        txt->SetX(fRange+1.5*w);
        txt->SetY(H*((i+0.5)*h - 1.));
        txt->Draw();
    }
}

// ------------------------------------------------------------------------
//
//
void MCamDisplay::DrawPhotNum(const MCerPhotEvt *event)
{
    fDrawingPad->cd();

    //
    // loop over all pixels in the MCerPhotEvt and
    // determine the Pixel Id and the content
    //
    Reset();

    //
    //  if the autoscale is true, set the values for the range for
    //  each event
    //
    if (fAutoScale)
    {
        fMinPhe = event->GetNumPhotonsMin();
        fMaxPhe = event->GetNumPhotonsMax();

        if (fMaxPhe < 20.)
            fMaxPhe = 20.;

        UpdateLegend();
    }

    //
    //   update the colors in the picture
    //
    const Int_t entries = event->GetNumPixels();

    for (Int_t i=0; i<entries; i++)
    {
        const MCerPhotPix &pix = (*event)[i];

        if (!pix.IsPixelUsed())
            continue;

        SetPixColor(pix);
    }

    //
    // Update display physically
    //
    gPad->Modified();
    gPad->Update();
}

// ------------------------------------------------------------------------
//
//
void MCamDisplay::DrawPhotErr(const MCerPhotEvt *event)
{
    fDrawingPad->cd();

    //
    // reset the all pixel colors to a default value
    //
    Reset();

    //
    // loop over all pixels in the MCerPhotEvt and
    // determine the Pixel Id and the content
    //
    const Int_t entries = event->GetNumPixels();

    for (Int_t i=0 ; i<entries; i++)
    {
        const MCerPhotPix &pix = (*event)[i];

        SetPixColor(pix);
    }

    //
    // Update display physically
    //
    gPad->Modified();
    gPad->Update();
}


// ------------------------------------------------------------------------
//
// reset the all pixel colors to a default value
//
void MCamDisplay::Reset()
{
    for (UInt_t i=0; i<fNumPixels; i++)
        (*this)[i].SetFillColor(10);
} 

// ------------------------------------------------------------------------
//
//   Here we calculate the color index for the current value.
//   The color index is defined with the class TStyle and the
//   Color palette inside. We use the command gStyle->SetPalette(1,0)
//   for the display. So we have to convert the value "wert" into
//   a color index that fits the color palette.
//   The range of the color palette is defined by the values fMinPhe
//   and fMaxRange. Between this values we have 50 color index, starting
//   with 0 up to 49.
//
Int_t MCamDisplay::GetColor(Float_t val)
{
    //
    //   first treat the over- and under-flows
    //
    const Float_t maxcolidx = 49.0;

    if (val >= fMaxPhe)
        return gStyle->GetColorPalette(maxcolidx);

    if (val <= fMinPhe)
        return gStyle->GetColorPalette(0);

    //
    // calculate the color index
    //
    const Float_t ratio  = (val-fMinPhe) / (fMaxPhe-fMinPhe);
    const Int_t   colidx = (Int_t)(maxcolidx*ratio + .5);

    return gStyle->GetColorPalette(colidx);
}

// ------------------------------------------------------------------------
//
//    change the text on the legend according to the range of the
//    Display
//
void MCamDisplay::UpdateLegend()
{
    char text[10];

    for (Int_t il=0; il < kItemsLegend; il++)
    {
        const Float_t val = fMinPhe + (Float_t)il/kItemsLegend * (fMaxPhe-fMinPhe) ;

        sprintf(text, "%5.1f", val);

        TText &txt = *GetText(il);

        txt.SetText(txt.GetX(), txt.GetY(), text);
    }
}
