/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  06/2001 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MWriteAsciiFile                                                         //
//                                                                         //
// If you want to store a single container into an Ascii file you have     //
// to use this class. You must know the name of the file you wanne write   //
// (you should know it) and the name of the container you want to write.   //
// This can be the name of the class or a given name, which identifies     //
// the container in a parameter container list (MParList).                 //
// The container is writte to the ascii file if its ReadyToSave flag is     //
// set (MParContainer)                                                     //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MWriteAsciiFile.h"

#include <fstream.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

ClassImp(MWriteAsciiFile);

// --------------------------------------------------------------------------
//
// Specify the name of the ascii output file 'filename' and the name
// of the container you want to write. (Normally this is the name
// of the class (eg. MHillas) but it can also be a different name which
// identifies the container in the parameter list.
//
MWriteAsciiFile::MWriteAsciiFile(const char *filename, const char *contname,
                                 const char *name, const char *title)
    : fOut(NULL), fContainer(NULL)
{
    *fName  = name  ? name  : "MWriteAsciiFile";
    *fTitle = title ? title : "Task to write one container to an ascii file";

    fNameFile      = filename;
    fNameContainer = contname;

    fOut = new ofstream(fNameFile);
}

// --------------------------------------------------------------------------
//
// Specify a the name of the ascii output file 'filename' and a pointer to
// the container you want to write. 
//
MWriteAsciiFile::MWriteAsciiFile(const char *filename, MParContainer *cont,
                                 const char *name, const char *title)
    : fOut(NULL), fContainer(cont)
{
    *fName  = name  ? name  : "MWriteAsciiFile";
    *fTitle = title ? title : "Task to write one container to an ascii file";

    fNameFile      = filename;
    fNameContainer = cont->GetName();

    fOut = new ofstream(fNameFile);
}

// --------------------------------------------------------------------------
//
// Destructor. Delete the output file if necessary (it is closed
// automatically by its destructor.
//
MWriteAsciiFile::~MWriteAsciiFile()
{
    delete fOut;
}

// --------------------------------------------------------------------------
//
// Check if our container is ready for writing. If so write it.
//
void MWriteAsciiFile::CheckAndWrite() const
{
    if (fContainer->IsReadyToSave())
        fContainer->AsciiWrite(*fOut);
}

// --------------------------------------------------------------------------
//
// Return open state of the file
//
Bool_t MWriteAsciiFile::IsFileOpen() const
{
    return (bool)(*fOut);
}

// --------------------------------------------------------------------------
//
// If the container is yet unknown and the name of it is known only, try
// to get the container from the parameter list.
//
Bool_t MWriteAsciiFile::GetContainer(MParList *pList)
{
    //
    // Try to find the container which should be stored.
    //
    if (fContainer)
        return kTRUE;

    fContainer = (MParContainer*)pList->FindObject(fNameContainer);
    if (fContainer)
        return kTRUE;

    *fLog << dbginf << "Cannot find parameter container '" << fContainer << "'." << endl;
    return kFALSE;
}
